/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Environment/ExtensionKit.cs#12 $
 * $DateTime: 2008/02/22 14:42:12 $
 * gLbgNX
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Travis.ORT;
using Poderosa;
using Poderosa.Preferences;
using Poderosa.Commands;

using IParameterProvider=Bellagio.Values.IParameterProvider;
using FunctionLibrary=Bellagio.Values.FunctionLibrary;

using Bellagio.Values;
using Bellagio.Chart;
using Bellagio.FigureBoard;
using Bellagio.Alert;
using Bellagio.Script;
using Bellagio.Screening;


#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Environment {

    public abstract class ExtensionKitItem : OriginReportableObject {
        public ORText id;
        public ORText description;

        protected ExtensionKit _owner;
        private string _fullID;

        public ExtensionKit Owner {
            get {
                return _owner;
            }
        }


        public string FullID {
            get {
                if(_fullID==null) {
                    _fullID = new StringBuilder().Append(_owner.id.ParseMandatoryString()).Append('.').Append(id.ParseMandatoryString()).ToString();
                }
                return _fullID;
            }
        }

        internal virtual void Load(ExtensionKit owner) {
            _owner = owner;
        }
        internal virtual void Unload() {
        }
    }

    public class ExtensionKitItemWithParameter : ExtensionKitItem, IDynamicPreferenceNode, IParameterProvider, StringWithParameterParser.ISite {
        public ORCollectionMember<ExtensionParameter> parameter;
        public ExtensionKitItemWithParameter() {
            parameter = new ORCollectionMember<ExtensionParameter>();
        }

        internal override void Load(ExtensionKit owner) {
            base.Load(owner);
            foreach(ExtensionParameter p in parameter) {
                p.Load(this);
            }
        }
        internal override void Unload() {
            base.Unload();
            foreach(ExtensionParameter p in parameter) {
                p.Unload();
            }
        }

        INamedValue IParameterProvider.FindParameter(string name) {
            return FindExtensionParameter(name);
        }

        public ExtensionParameter FindExtensionParameter(string name) {
            foreach(ExtensionParameter p in parameter) {
                if(p.id.Value==name) return p;
            }
            return null;
        }
        #region IExtensiblePreferenceNode
        string IDynamicPreferenceNode.PreferenceID {
            get {
                return id.Value;
            }
        }

        IDynamicPreferenceNode IDynamicPreferenceNode.ParentNode {
            get {
                return _owner;
            }
        }
        #endregion

        string StringWithParameterParser.ISite.FindParameter(string name) {
            ExtensionParameter ep = FindExtensionParameter(name);
            return ep==null? null : BellagioRoot.ExtensionKitPreference.InternalGetValue(ep);
        }
    }


    public class ExtensionParameter : OriginReportableObject, IDynamicPreferenceItem, INamedValue {
        public ORText id;
        public ORText initial;
        private ExtensionKitItemWithParameter _schema;

        public void Load(ExtensionKitItemWithParameter schema) {
            _schema = schema;
            BellagioRoot.ExtensionKitPreference.RegisterItem(this);
        }
        public void Unload() {
        }

        public string DefaultValue {
            get {
                return initial.Value;
            }
        }
        public string Value {
            get {
                return BellagioRoot.ExtensionKitPreference.InternalGetValue(this);
            }
            set {
                _bvalue = null;
                BellagioRoot.ExtensionKitPreference.InternalSetValue(this, value);
            }
        }

        public string PreferenceID {
            get {
                return id.Value;
            }
        }

        public IDynamicPreferenceNode ParentNode {
            get {
                return _schema;
            }
        }

        #region INamedValue
        private BV _bvalue;
        public BT BT {
            get {
                //TODO lZbgĂ^͕ςȂ悤ɂȂ
                if(_bvalue==null) GetValue(null);
                return _bvalue.BT;
            }
        }
        public BV GetValue(EvalContext ctx) {
            if(_bvalue==null) {
                string a = this.Value;
                if(a.IndexOf('.')!=-1)
                    _bvalue = new BDouble(Double.Parse(a));
                else {
                    _bvalue = new BInt(Int32.Parse(a));

                }

            }
            return _bvalue;
        }
        public string Name {
            get {
                return id.Value;
            }
        }
        #endregion
    }

    public abstract class ViewSchemaBase : ExtensionKitItemWithParameter {
        internal IGeneralCommand _command;

        public ViewSchemaBase() {
        }

        internal IGeneralCommand SessionOpenCommand {
            get {
                return _command;
            }
        }
    }

    public abstract class ViewSchemaCollection<T> : IORCollectionMemberT<T> {
        //protected ORTypeInfo _itemSchema;

        public abstract void Clear();
        public abstract int Count { get; }
        public abstract void Add(T item);
        public abstract T this[int index] { get; }

        public void Add(object item) {
            Add((T)item);
        }
        public void AddAll(IORCollectionMemberT<T> coll) {
            foreach(T e in coll) Add(e);
        }
        public abstract IEnumerator GetEnumerator();
    }

    public class ExtensionKit : OriginReportableObject, IDynamicPreferenceNode {
        public ORText id;
        public ORText version;
        //public RankingSchemaCollection ranking;
        //public ScatterSchemaCollection scatter;
        public ChartSchemaCollection realtimeChart;
        public ChartSchemaCollection staticChart;
        public FigureBoardSchemaCollection figureBoard;
        public AlertSchemaCollection alert;
        public ScreeningSchemaCollection screening;
        public AutoTradingSchemaCollection autotrading;
        public ORCollectionMember<IndicatorDefinition> indicatorDef;
        public ORCollectionMember<PilotAutoTrading> pilotAutotrading;
        public ORText functionDefinitions; //TODO ̕Gg悤ɂ

        private string _filename;
        private List<FunctionLibrary.UserDefinedFunction> _userDefinedFunctions;

        public ExtensionKit(string filename) {
            _filename = filename;

            id = version = new ORText();
            //ranking = new RankingSchemaCollection();
            //scatter = new ScatterSchemaCollection();
            realtimeChart = new ChartSchemaCollection();
            staticChart = new ChartSchemaCollection();
            figureBoard = new FigureBoardSchemaCollection();
            alert = new AlertSchemaCollection();
            screening = new ScreeningSchemaCollection();
            autotrading = new AutoTradingSchemaCollection();
            indicatorDef = new ORCollectionMember<IndicatorDefinition>();
            pilotAutotrading = new ORCollectionMember<PilotAutoTrading>();
        }

        public string FileName {
            get {
                return _filename;
            }
        }

        public void Load() {
            bool succeeded = false;
            try {
                id.ParseMandatoryString(); //ȗĂȂƂmF
                //ŜFix
                //foreach(ExtensionKitItem x in ranking) x.Load(this);
                //foreach(ExtensionKitItem x in scatter) x.Load(this);
                foreach(ExtensionKitItem x in realtimeChart) x.Load(this);
                foreach(ExtensionKitItem x in staticChart) x.Load(this);
                foreach(ExtensionKitItem x in figureBoard) x.Load(this);
                foreach(ExtensionKitItem x in alert) x.Load(this);
                foreach(ExtensionKitItem x in indicatorDef) x.Load(this);
                foreach(ExtensionKitItem x in screening) x.Load(this);
                foreach(ExtensionKitItem x in autotrading) x.Load(this);
                foreach(PilotAutoTrading x in pilotAutotrading) x.Load(this);

                RegisterUserFunctions();
                succeeded = true;
            }
            finally {
                if(!succeeded) Unload();
            }
        }
        //AA[h͂ɈˑZbV݂ȂƂɂłȂB
        //[hɎŝŃA[hAƂgȂOKAĂŒ̂̂[h邱Ƃ͂łȂB
        //ɂƂĂxZbVjA߂č\zAƂ邵Ȃ낤BꉞAeZbV͋NXL[}IuWFNgł͂ȂOŎQƂł悤ɂȂĂ̂łgB
        public void Unload() {
            //foreach(ExtensionKitItem x in ranking) x.Unload();
            //foreach(ExtensionKitItem x in scatter) x.Unload();
            foreach(ExtensionKitItem x in realtimeChart) x.Unload();
            foreach(ExtensionKitItem x in staticChart) x.Unload();
            foreach(ExtensionKitItem x in figureBoard) x.Unload();
            foreach(ExtensionKitItem x in alert) x.Unload();
            foreach(ExtensionKitItem x in indicatorDef) x.Unload();
            foreach(ExtensionKitItem x in screening) x.Unload();
            foreach(ExtensionKitItem x in autotrading) x.Unload();
            foreach(PilotAutoTrading x in pilotAutotrading) x.Unload();

            UnregisterUserFunctions();

            //ő\ExtensionPreff^b`
            BellagioRoot.ExtensionKitPreference.UnregisterNode(this);
        }


        //֐̓o^@ꉞp[Xx邱Ƃ\͉\...
        private void RegisterUserFunctions() {
            if(functionDefinitions.IsOmitted) return;
            _userDefinedFunctions = new List<FunctionLibrary.UserDefinedFunction>();

            string name_space = id.ParseMandatoryString();
            FunctionDefinitionList fl = Expression.ParseFunctionDefinitionList(functionDefinitions);
            foreach(FunctionDefinition fd in fl) {
                FunctionLibrary.UserDefinedFunction def = BellagioRoot.Functions.User.DefineUserFunction(name_space, fd.Name, fd.Parameters, fd.ReturnType, fd.BodyExpression, new ObjectOrigin(_filename, functionDefinitions.Line + fd.LineNumber - 1, 0));
                _userDefinedFunctions.Add(def);
            }

            //֐̎QƏ̂ŁASDefineĂ珇RpCKv
            foreach(FunctionLibrary.UserDefinedFunction def in _userDefinedFunctions)
                def.Compile();
        }
        private void UnregisterUserFunctions() {
            if(_userDefinedFunctions==null) return;

            foreach(FunctionLibrary.UserDefinedFunction def in _userDefinedFunctions)
                BellagioRoot.Functions.User.RemoveFunction(def);
        }


        public ChartSchema FindRealTimeChart(string name) {
            return (ChartSchema)FindItem(name, realtimeChart);
        }
        public ChartSchema FindStaticChart(string name) {
            return (ChartSchema)FindItem(name, staticChart);
        }
        /*
        public ScatterSchema FindScatter(string name) {
            return (ScatterSchema)FindItem(name, scatter);
        }
        public RankingSchema FindRanking(string name) {
            return (RankingSchema)FindItem(name, ranking);
        }
        */
        internal FigureBoardSchema FindFigureBoard(string name) {
            return (FigureBoardSchema)FindItem(name, figureBoard);
        }
        internal IndicatorDefinition FindIndicatorDef(string name) {
            return (IndicatorDefinition)FindItem(name, indicatorDef);
        }
        internal ScreeningSchema FindScreening(string name) {
            return (ScreeningSchema)FindItem(name, screening);
        }
        public AutoTradingSchema FindAutoTrading(string name) {
            return (AutoTradingSchema)FindItem(name, autotrading);
        }
        private ExtensionKitItem FindItem(string name, IORCollectionMember coll) {
            foreach(ExtensionKitItem x in coll)
                if(x.id.Value==name) return x;
            return null;
        }

        public IList CollectStaticChart(IList c) {
            return CollectItem(c, staticChart);
        }
        public IList CollectRealTimeChart(IList c) {
            return CollectItem(c, realtimeChart);
        }
        /*
        public IList CollectScatter(IList c) {
            return CollectItem(c, scatter);
        }
        public IList CollectRanking(IList c) {
            return CollectItem(c, ranking);
        }
        */
        public IList CollectFigureBoard(IList c) {
            return CollectItem(c, figureBoard);
        }
        public IList CollectAlert(IList c) {
            return CollectItem(c, alert);
        }
        public IList CollectScreening(IList c) {
            return CollectItem(c, screening);
        }
        public IList CollectAutoTrading(IList c) {
            return CollectItem(c, autotrading);
        }
        private IList CollectItem(IList c, IORCollectionMember m) {
            foreach(ExtensionKitItem x in m) c.Add(x);
            return c;
        }

        //IBellagioPreferenceNode
        public string PreferenceID {
            get { return id.Value; }
        }

        public IDynamicPreferenceNode ParentNode {
            get { return null; }
        }
    }

    //eLXgɃp[^𖄂ߍދ@\
    // p[^́A(1) $name  (2) ${name} ̂ꂩ̌`B{p[^Ɏĝ(2)̂
    public class EmbeddedParameterParser {
        public interface ISite {
            void Parameter(string name);
            void Literal(char ch);
        }
        private ISite _handler;
        private IORText _locInfo;
        private int _parameterNameStartPos;
        private char[] _src;
        private State _state;

        private enum State {
            Default,
            Dollar,
            NormalParam,
            BracketParam
        }

        public EmbeddedParameterParser(ISite pp) {
            _handler = pp;
        }

        public string TextLocationString {
            get {
                return _locInfo.LocationString;
            }
        }

        public void Parse(ORText text) {
            _locInfo = text;
            ParseInternal(text.ParseMandatoryString().ToCharArray());
        }
        public void Parse(string text) {
            _locInfo = null; //not available
            ParseInternal(text.ToCharArray());
        }
        private void ParseInternal(char[] src) {
            _state = State.Default;
            _src = src;

            for(int i=0; i<_src.Length; i++)
                ProcessChar(i, _src[i]);

            if(_state==State.NormalParam) {
                AppendParameter(_src.Length);
            }
        }

        private void ProcessChar(int pos, char ch) {
            if(_state==State.Default) {
                if(ch=='$')
                    _state = State.Dollar;
                else
                    _handler.Literal(ch);
            }
            else if(_state==State.Dollar) {
                if(ch=='{') {
                    _state = State.BracketParam;
                    _parameterNameStartPos = pos+1;
                }
                else {
                    _parameterNameStartPos = pos;
                    _state = State.NormalParam;
                }
            }
            else if(_state==State.NormalParam) {
                if(!IsParameterLetter(ch) || (int)ch>=256) { //Char.IsLetterł͓{trueɂȂĂ܂
                    AppendParameter(pos);
                    ProcessChar(pos, ch);
                }
            }
            else if(_state==State.BracketParam) {
                if(ch=='}')
                    AppendParameter(pos);
            }
        }

        private void AppendParameter(int lastPos) {
            string name = new string(_src, _parameterNameStartPos, lastPos-_parameterNameStartPos);
            _handler.Parameter(name);
            _state = State.Default;
        }

        private static bool IsParameterLetter(char ch) {
            if('a'<=ch && ch<='z')
                return true;
            else if('A'<=ch && ch<='Z')
                return true;
            else if('0'<=ch && ch<='9')
                return true;
            else if(ch=='_')
                return true;
            else
                return false;
        }
    }
    //Ŝ𕶎񉻂^CvB悭ĝł
    public class StringWithParameterParser : EmbeddedParameterParser.ISite {
        private Dictionary<string, string> _additionalStringMap; //ParameterProviderƂ͕ʂɕϊe[uقƂp
        private StringBuilder _result;
        private ISite _handler;
        private EmbeddedParameterParser _parser;

        public StringWithParameterParser(ISite handler) {
            _result = new StringBuilder();
            _parser = new EmbeddedParameterParser(this);
            _handler = handler;
        }
        public void AddToStringMap(string key, string value) {
            if(_additionalStringMap==null) _additionalStringMap = new Dictionary<string, string>();
            _additionalStringMap.Add(key, value);
        }
        public string Parse(ORText text) {
            if(text.IsOmitted) return "";
            _result.Remove(0, _result.Length);
            _parser.Parse(text);
            return _result.ToString();
        }
        public string Parse(string text) {
            _result.Remove(0, _result.Length);
            _parser.Parse(text);
            return _result.ToString();
        }


        public interface ISite {
            string FindParameter(string name);
        }

        public void Parameter(string name) {
            string value;
            if(_additionalStringMap!=null) {
                if(_additionalStringMap.TryGetValue(name, out value)) {
                    _result.Append(value);
                    return;
                }
            }

            value = _handler.FindParameter(name);
            if(value==null)
                throw new BellagioException(String.Format("{0} {1}`Ă܂", _parser.TextLocationString, name));

            _result.Append(value);
        }

        public void Literal(char ch) {
            _result.Append(ch);
        }
    }

    public class ArgumentValueFormatter : StringWithParameterParser.ISite {
        private BV[] _values;
        private EvaluatorBuildContextWithArgs _ctx;

        public ArgumentValueFormatter(EvaluatorBuildContextWithArgs ctx) {
            _ctx = ctx;
            _values = ctx.EvalArguments();
        }


        public string FindParameter(string name) {
            int index = _ctx.IndexFromName(name);
            if(index==-1)
                return "";
            else {
                BVFormatter formatter = new DefaultBVFormatter();
                _values[index].Format(formatter);
                return formatter.Result;
            }
        }

    }

}
