/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/ChartDocumentBase.cs#15 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * `[gƔ̃ZbVAstatic`[g̃ZbV̋Lp[g
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

using Bellagio.Evaluators;
using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Drawing;
using Bellagio.Forms;

using Travis.ORT;
using Travis.Collections;

namespace Bellagio.Chart {
    public interface ICandleExtraModifier {
        bool IsHighlight(Candle candle);
    }

    public abstract class ChartDocumentBase : BellagioDocumentBase, IChartDataSource {
        private ChartSchema _schema;
        protected RuntimeIndicatorSet _indicators;
        protected RuntimeOscillatorGroupSet _oscillatorGroupSet;
        protected EvalContext _evalContext;
        protected Stock _stock;

        protected SubseqQuote _contextQuote; //indicator̕]ɗpƂ̃ReLXg
        protected IndicatorValueCache _indicatorValueCache;
        protected ICandleExtraModifier _candleExtraModifier;
        
        //RXgN^Ăяo`CompleteCreation܂
        private bool _constructing;

        public ChartDocumentBase() {
            _constructing = true;
            _indicators = new RuntimeIndicatorSet();
            _evalContext = new EvalContext();
            _oscillatorGroupSet = new RuntimeOscillatorGroupSet();
            _contextQuote = new SubseqQuote(null, 0, 0); //܂J
        }

        public Stock CurrentStock {
            get {
                return _stock;
            }
        }
        public IndicatorValueCache ValueCache {
            get {
                return _indicatorValueCache;
            }
        }
        public ChartSchema SourceSchema {
            get {
                return _schema;
            }
        }
        public ICandleExtraModifier CandleExtraModifier {
            get {
                return _candleExtraModifier;
            }
            set {
                _candleExtraModifier = value;
            }
        }

        private StringWithParameterParser CreateStringWithParameterParser(EvaluatorBuildContextWithArgs cc, string unit_name) {
            StringWithParameterParser spp = new StringWithParameterParser(new ArgumentValueFormatter(cc));
            spp.AddToStringMap("unit", unit_name); //""""Ȃǂ̉p
            return spp;
        }

        //̎
        public void ConstructWithSchema(ChartSchema schema) {
            Debug.Assert(schema!=null);
            _schema = schema;
            Dictionary<OscillatorGroupSchema, RuntimeOscillatorGroup> oscillator_group_map = new Dictionary<OscillatorGroupSchema, RuntimeOscillatorGroup>();

            for(int i=0; i<schema.oscillatorGroup.Count; i++) {
                OscillatorGroupSchema sgr = schema.oscillatorGroup[i];
                RuntimeOscillatorGroup rgr = new RuntimeOscillatorGroup(sgr, sgr.id.Value, i);
                AddOscillatorGroup(rgr);
                oscillator_group_map.Add(sgr, rgr);
            }
            for(int i = 0; i<schema.indicator.Count; i++) {
                IndicatorSchema ind = schema.indicator[i];
                RuntimeIndicator ri = CreateRuntimeIndicator(ind, schema, _indicators.ElementCount);

                //IV[^O[vRuntimeIndicatorǉ
                if(ri.Location.IsOscillator) {
                    RuntimeOscillatorGroup og;
                    if(ind.OscillatorGroup==null) { //XL[}xŎw肳ĂȂΐpRuntimeOscillatorGroup
                        og = new RuntimeOscillatorGroup(null, "auto", _oscillatorGroupSet.Count);
                        AddOscillatorGroup(og);
                    }
                    else
                        og = oscillator_group_map[ind.OscillatorGroup];
                    Debug.Assert(og!=null);
                    og.AddOscillator(ri);
                }

                _indicators.AddIndicator(ri);
                ri.CompleteCreation();
            }

            CompleteCreation(); //ō쐬I
        }
        private RuntimeIndicator CreateRuntimeIndicator(IndicatorSchema schema, ChartSchema chart, int index_base) {
            IndicatorDefinition ind_def = schema.IndicatorDefinition;
            ISpecialIndicatorPaint special_paint = null;
            if(!ind_def.specialPaint.IsOmitted)
                special_paint = SpecialIndicatorPaintLib.GetFromName(ind_def.specialPaint.ParseMandatoryString());

            EvaluatorBuildContextWithArgs arg_context = EvaluatorBuilderFromText.CreateContextWithArgs(chart, ind_def.ParsedParameterDefinition, schema.ParsedArguments, schema.args.LocationString);
            arg_context.ResultContext.ContextType = QuoteType.instance;

            StringWithParameterParser spp = CreateStringWithParameterParser(arg_context, chart.ChartUnitName);

            IndicatorLocation location = IndicatorLocation.Parse(ind_def.location);
            RuntimeIndicator ri = new RuntimeIndicator(schema, schema.label.IsOmitted? "(O`)" : spp.Parse(schema.label), location, arg_context, special_paint);
            ri.SharesAppearance = ind_def.sharesAppearance.ParseOptionalBool(false);
            LocalVariable[] arg_vars = arg_context.Arguments();
            BV[] arg_values = arg_context.EvalArguments();


            try {
                //GgƂRuntimeIndicatorElement
                for(int j=0; j<ind_def.element.Count; j++) {
                    IndicatorDefinition.Element elem_def = ind_def.element[j];
                    ORText raw_label = elem_def.label.IsOmitted? ind_def.label : elem_def.label;
                    RuntimeIndicatorElement rie = new RuntimeIndicatorElement(
                        j,
                        index_base + j,
                        ri,
                        raw_label,
                        AssureDoubleType(EvaluatorBuilderFromText.BuildStandAlone(elem_def.expression, null, QuoteType.instance, arg_vars, arg_values)),
                        elem_def.forward.IsOmitted? null : EvaluatorBuilderFromText.BuildStandAlone(elem_def.forward, null, null, arg_vars, arg_values));
                    rie.Build(spp);
                    //TODO Evaluatoř^lłȂ΃G[

                    IDynamicPreferenceItem apparence_item = schema.AppearancePreferences[ri.SharesAppearance? 0 : j];
                    rie.BPenStyle = BPenStyle.Parse(apparence_item.Value);
                    rie.Visible = rie.BPenStyle.visible;
                    rie.FigureAdjustment = location==IndicatorLocation.Chart? FigureAdjustment.Price : RuntimeIndicatorElement.ParseFigureAdjustment(elem_def.adjustFigures.ParseOptionalString(null));
                    ri.AddElement(rie, elem_def);
                }

                return ri;
            }
            catch(Exception ex) {
                BUtil.ShowWarningMessageBox(ex.Message);
                return null;
            }
        }

        private void AddOscillatorGroup(RuntimeOscillatorGroup gr) {
            Debug.Assert(_constructing);
            _oscillatorGroupSet.Add(gr);
        }


        protected virtual void CompleteCreation() {
            _constructing = false;
            _indicatorValueCache = new IndicatorValueCache(_indicators.ElementCount);
        }


        //̓IύX
        public virtual void SetCurrentStock(Stock stock) {
            _stock = stock;
            _candleExtraModifier = null;
            ClearIndicatorValueCache();
        }
        public virtual void SuspendStock() {
            ClearIndicatorValueCache();
            ResetSectionInfo();
            _documentStatus = BellagioDocumentStatus.Loading;
        }
        //͕ςf[^蒼
        public virtual void ReloadQuote() {
            ClearIndicatorValueCache();
        }
        private void ClearIndicatorValueCache() {
            _indicatorValueCache.Clear();
        }

        protected abstract Quote SourceQuote { get; }
        protected abstract bool AdjustEvalContext(EvalContext context, int data_index);

        #region IChartDataSource
        public RuntimeIndicatorSet Indicators {
            get {
                return _indicators;
            }
        }
        public RuntimeOscillatorGroupSet OscillatorGroupSet {
            get {
                return _oscillatorGroupSet;
            }
        }
        public Stock Stock {
            get {
                return _stock;
            }
        }
        public abstract int LastClosePrice { get; }
        public abstract ChartSectionInfo SectionInfo { get; }
        public abstract void ResetSectionInfo();
        public abstract Candle GetCandleAt(CHARTPOSITION position);
        public abstract int GetDateOrTimeAt(CHARTPOSITION position);
        public abstract bool GetChartPositionAt(int raw_time, out CHARTPOSITION pos);
        public abstract int GetNextDateOrTimeAt(CHARTPOSITION position);
        public virtual bool IsTimeLabelRequired(int raw_time, out string str) {
            str = null;
            return false;
        }
        public virtual SplitInfo FindSplitInfo(int raw_time) {
            return null;
        }
        public virtual string GetTimeLabel(int raw_time) {
            return "";
        }

        public double GetIndicatorValue(RuntimeIndicatorElement ind, CHARTPOSITION position) {
            Debug.Assert(!_constructing);
            ChartDrawingStats.IncrValueRequestCount();

            int data_index = position.ToDataIndexSoft();
            if(data_index==-1) {
                //̈ɂf[^CfbNX̒l ̈ɂĂ͂ꂽ`Ŋi[
                if(!position.IsEmpty && position.SectionType==ChartSectionType.Forward && ind.FutureLength > position.PositionInSection)
                    data_index = this.SectionInfo.GetDataLength() + position.PositionInSection - ind.FutureLength;
                else
                    return Double.NaN;
            }
            else
                data_index -= ind.FutureLength;

            if(data_index < 0) return Double.NaN; //CfbNXČʃAEgȂ\Ȃ

            Debug.Assert(data_index < this.SectionInfo.GetDataLength());

            BV value = _indicatorValueCache.TryGetValue(data_index, ind.Index);
            if(value==null) { //qbg~XȂ̂ŌvZ
                ChartDrawingStats.IncrValueCacheMissCount();
                bool r = AdjustEvalContext(_evalContext, data_index);
                if(!r) return Double.NaN; //̃P[Xł̓LbVւ̔f͂Ȃ

                BV.Let(ref value, ind.Evaluator.Eval(_evalContext));
                _indicatorValueCache.SetValue(data_index, ind.Index, value);
            }

            if(value.IsNil)
                return Double.NaN;
            else {
                double double_result = ((BDouble)value).Value;
                if(ind.FigureAdjustment==FigureAdjustment.Price)
                    return _stock.Profile.AdjustPriceD(double_result);
                else
                    return double_result;
            }
        }
        public CandleExtraChart GetCandleExtraChart(CHARTPOSITION position) {
            Quote pr = this.SourceQuote;
            int current_index = position.ToDataIndex();

            CandleExtraChart extra = _indicatorValueCache.TryGetExtra(current_index);
            if(extra!=null) return extra; //LbVɂAOK!

            //ȉLbV~X
            extra = new CandleExtraChart();
            int start = current_index - 4;
            int end   = current_index + 4; //͈͍̔͂Œ

            //߂̔
            if(start<0 || end>=pr.Count) {
                extra.fushiHigh = false;
                extra.fushiLow = false;
            }
            else {
                Candle current = pr.CandleAt(current_index);
                int high = current.High;
                extra.fushiHigh = true;
                for(int index=start; index<=end; index++) {
                    if(index!=current_index && pr.CandleAt(index).High >= high) {
                        extra.fushiHigh = false;
                        break; //lXVꂽfalse
                    }
                }

                int low = current.Low;
                extra.fushiLow = true;
                for(int index=start; index<=end; index++) {
                    if(index!=current_index && pr.CandleAt(index).Low <= low) {
                        extra.fushiLow = false;
                        break;
                    }
                }

            }

            //
            Candle c = pr.CandleAtOrNull(current_index);
            if(c!=null)
                extra.splitInfo = this.FindSplitInfo(c.Time);

            //nCCg
            if(_candleExtraModifier!=null && c!=null)
                extra.highlight = _candleExtraModifier.IsHighlight(c);

            _indicatorValueCache.SetExtra(current_index, extra);

            return extra;
        }

        #endregion


        public void SetOscillatorVisible(bool value) {
            foreach(RuntimeIndicator ri in _indicators.Indicators)
                if(ri.Location.IsOscillator) ri.SetVisible(value);
        }

        public virtual void Dispose() {
            _indicators.Dispose();
        }

        private static StandAloneEvaluator AssureDoubleType(StandAloneEvaluator src) {
            if(src.BT==BT.Int)
                return new StandAloneEvaluator(EvaluatorUtil.CreateI2D(src.BuildContext, (Evaluator)src.Content), src.BuildContext, src.Args);
            else if(src.BT==BT.Double)
                return src; //̂܂
            else
                throw new BellagioException("Indicator͐lłȂ΂܂");
        }
    }

    public class IndicatorValueCache {
        private int _indicatorCount;
        private List<BV[]> _data; //ȉQListCountCapacityƂv
        private List<CandleExtraChart> _extras;

        public IndicatorValueCache(int ic) {
            _indicatorCount = ic;
            _data = new List<BV[]>(1000); //߃Lp
            _extras = new List<CandleExtraChart>(1000);
        }

        //LbVqbg΂AȂnull
        public BV TryGetValue(int data_index, int indicator_index) {
            Debug.Assert(BUtil.IsExecutingInMainThread);
            if(_data.Count <= data_index) return null;
            BV[] a = _data[data_index];
            if(a==null) return null;

            Debug.Assert(a.Length==_indicatorCount);
            return a[indicator_index];
        }
        public void SetValue(int data_index, int indicator_index, BV value) {
            Debug.Assert(BUtil.IsExecutingInMainThread);
            if(_data.Count <= data_index) ExpandSize(data_index+1);
            BV[] a = _data[data_index];
            if(a==null) {
                a = new BV[_indicatorCount];
                _data[data_index] = a;
            }

            Debug.Assert(a.Length==_indicatorCount);
            a[indicator_index] = value;
        }
        public CandleExtraChart TryGetExtra(int data_index) {
            if(_extras.Count <= data_index) return null;

            return _extras[data_index];
        }
        public void SetExtra(int data_index, CandleExtraChart value) {
            if(_extras.Count <= data_index) ExpandSize(data_index+1);
            _extras[data_index] = value;
        }

        private void ExpandSize(int count) {
            while(_data.Capacity < count) {
                _data.Capacity = _data.Capacity*2;
                _extras.Capacity = _data.Capacity;
            }

            while(_data.Count < count) {
                _data.Add(null); //܂zƂ܂ł͂Ȃ
                _extras.Add(null);
            }
        }

        //ĝ݃NA
        public void Clear() {
            Debug.Assert(BUtil.IsExecutingInMainThread);
            foreach(BV[] a in _data) {
                if(a!=null) {
                    for(int i=0; i<a.Length; i++) a[i] = null;
                }
            }
            for(int i=0; i<_extras.Count; i++) {
                if(_extras[i]!=null) _extras[i] = null;
            }


        }
        //indicator̂݃NA
        public void ClearByIndicatorIndex(int index) {
            Debug.Assert(BUtil.IsExecutingInMainThread);
            foreach(BV[] a in _data) {
                if(a!=null) {
                    a[index] = null;
                }
            }
        }
        //̑ɑΉ̂݃NA
        public void ClearByDataIndex(int index) {
            Debug.Assert(BUtil.IsExecutingInMainThread);
            if(_data.Count <= index) return;
            _data[index] = null;
            _extras[index] = null;
        }
    }

}
