/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: Plugin.cs,v 1.6 2006/08/19 10:43:43 dan-iwasaki Exp $
 */
using System;
using System.Text;
using System.Reflection;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;

using Poderosa.Util.Collections;
using Poderosa.Boot;

namespace Poderosa.Plugins
{
    /// <summary>
    /// <ja>
    /// vOC̃Xe[^X܂B
    /// </ja>
    /// <en>
    /// Return the status of the plug-in.
    /// </en>
    /// </summary>
	public enum PluginStatus {
        /// <summary>
        /// <ja>OŎQƂꂽ̏</ja>
        /// <en>State only declared.</en>
        /// </summary>
		Declared,  //OŎQƂꂽ̏
        /// <summary>
        /// <ja>NXƂă[hł</ja>
        /// <en>Loaded as class.</en>
        /// </summary>
		Loaded,    //NXƂă[hł
        /// <summary>
        /// <ja>[h͂łˑsȂǂŖꂽ</ja>
        /// <en>It was nullified by the uncertainty etc. dependence ahead though it was possible to load.</en>
        /// </summary>
        Disabled,  //[h͂łˑsȂǂŖꂽ
        /// <summary>
        /// <ja>ɓǂݍ܂Aғς݂̏</ja>
        /// <en>Loaded successfully and activated.</en>
        /// </summary>
		Activated  //ғς݂̏
	}

    public static class PoderosaPluginEnv {
#if DEBUG
        public static bool MULTIPLE_INSTANCE_ALLOWED = true;
#else
        public static bool MULTIPLE_INSTANCE_ALLOWED = false;
#endif
        public static string WINDOW_CAPTION_KEY = "͂イNTX"; //TODO AvP[V{̂ݒ肷dg݂͂Ƒς
        public static string MODULE_FILENAME_KEY = "HatchukunTX";
    }

    internal class Plugin : IPluginInfo {
		private PluginManager _manager;
		private Assembly _assembly;
		private PluginStatus _status;

		private PluginInfoAttribute _attribute;
		private Plugin[] _dependencies;
		private Type _pluginType;
		private IPlugin _instance;

		public Plugin(PluginManager manager, Assembly assembly, Type type) {
			_manager = manager;
			_assembly = assembly;
            _pluginType = type;
			_status = PluginStatus.Declared;
		}

		public PluginInfoAttribute PluginInfo {
			get {
				return _attribute;
			}
		}
		public IPlugin Instance {
			get {
				return _instance;
			}
		}
		public string TypeName {
			get {
				return _pluginType.FullName;
			}
		}

		public Plugin[] Dependencies {
			get {
				return _dependencies;
			}
			set {
				_dependencies = value;
			}
		}


		public GenericResult TryToLoad() {
            string typename = _pluginType.FullName;
			if(!typeof(IPlugin).IsAssignableFrom(_pluginType)) {
				_manager.Tracer.Trace("PluginManager.Messages.IPluginIsNotImplemented", typename);
				return GenericResult.Failed;
			}

			object[] attrs = _pluginType.GetCustomAttributes(typeof(PluginInfoAttribute), false);
			if(attrs.Length!=1) {
				_manager.Tracer.Trace("PluginManager.Messages.PluginInfoAttributeNotFound", typename);
				return GenericResult.Failed;
			}

			_attribute = (PluginInfoAttribute)attrs[0];
			string id = _attribute.ID;
			if(id==null || id.Length==0) {
				_manager.Tracer.Trace("PluginManager.Messages.IDNotFound", typename);
				return GenericResult.Failed;
			}
			_status = PluginStatus.Loaded;

			return GenericResult.Succeeded;
		}

        public void Disable() {
            _status = PluginStatus.Disabled;
        }

		public GenericResult Instantiate() {
			_instance = (IPlugin)_assembly.CreateInstance(_pluginType.FullName);
			_status = PluginStatus.Activated;
			return GenericResult.Succeeded;
        }

        #region IPluginInfo
        public IPlugin Body {
            get {
                return _instance;
            }
        }

        public PluginInfoAttribute PluginInfoAttribute {
            get {
                return _attribute;
            }
        }
        public PluginStatus Status {
            get {
                return _status;
            }
        }

        public IAdaptable GetAdapter(Type adapter) {
            return _manager.PoderosaWorld.AdapterManager.GetAdapter(this, adapter);
        }
        #endregion
    }

	internal class PluginManager : IPluginManager, IPluginInspector {
		private InternalPoderosaWorld _world;
        private List<string> _disabledPluginIDs;
		private List<Plugin> _allPlugins;
		private TypedHashtable<string, Plugin> _idToPlugin;
        private TypedHashtable<string, ExtensionPoint> _idToExtensionPoint;
		private List<Plugin> _orderedPlugins;
        private Plugin _currentInitializingPlugin;
		private ITracer _tracer;

		public PluginManager(InternalPoderosaWorld pw) {
			_world = pw;
            _disabledPluginIDs = new List<string>();
            _idToExtensionPoint = new TypedHashtable<string, ExtensionPoint>();
        }

        public ITracer Tracer {
			get {
				return _tracer;
			}
		}

		public bool HasError {
			get {
				return !_tracer.Document.IsEmpty;
			}
		}

		public IPlugin[] GetOrderedPlugins() {
			IPlugin[] r = new IPlugin[_orderedPlugins.Count];
			for(int i=0; i<r.Length; i++)
				r[i] = ((Plugin)_orderedPlugins[i]).Instance;
			return r;
        }

        //Nɂ
      	public void InitializePlugins(PoderosaStartupContext sc) {
            try {
                _tracer = sc.Tracer;
                Debug.Assert(_tracer != null);

                List(sc.PluginManifest);
                Load();
                Order();
                Instantiate();
            } catch (Exception ex) {
                _tracer.Trace(ex);
            }
		}

        //I
        public void Shutdown() {
            //tŏI
            for(int i=_orderedPlugins.Count-1; i>=0; i--) {
                try {
                    Plugin p = _orderedPlugins[i];
                    if(p.Instance!=null)
                        p.Instance.TerminatePlugin();
                } catch (Exception ex) {
                    _tracer.Trace(ex);
                }
            }
        }

        #region IPluginManager
        public object FindPlugin(string id, Type adaptertype) {
            Plugin p = _idToPlugin[id];
            if(p == null)
                return null;
            else {
                if(p.Status != PluginStatus.Activated) throw new InvalidOperationException("Plugin is not activated");

                return p.Instance.GetAdapter(adaptertype);
            }
        }

        public IExtensionPoint CreateExtensionPoint(string id, Type requiredInterface, IPlugin owner) {
            if(_currentInitializingPlugin==null && id!=ExtensionPoint.ROOT) //[g
                throw new InvalidOperationException(InternalPoderosaWorld.Strings.GetString("PluginManager.Messages.NewExtensionPointOutsideInit"));
            if(_idToExtensionPoint.Contains(id))
                throw new ArgumentException(InternalPoderosaWorld.Strings.GetString("PluginManager.Messages.DuplicatedExtensionPointID"));
            ExtensionPoint e = new ExtensionPoint(id, requiredInterface, owner);
            _idToExtensionPoint[id] = e;
            return e;
        }
        public IExtensionPoint FindExtensionPoint(string id) {
            return _idToExtensionPoint[id];
        }
        //[hÕvOCIDw肵ă[hȂݒ
        public void DisablePlugin(string[] ids) {
            foreach(string t in ids)
                _disabledPluginIDs.Add(t);
        }
        #endregion

		private void List(PluginManifest manifest) {
			_allPlugins = new List<Plugin>();
			foreach(object asm_def_ in manifest.Children) {
				StructuredText asm_def = asm_def_ as StructuredText;
				if(asm_def==null) continue; //G[nhO

				string asm_name = asm_def.Name;
				PluginManifest.AssemblyNode asmnode = null;
				try {
					asmnode = manifest.LoadAssemblyNode(asm_def);
				}
				catch(Exception) {
					_tracer.Trace("PluginManager.Messages.AssemblyLoadError", asm_name);
					continue;
				}

                asmnode.TryToBind(_tracer);
                
                foreach(Type t in asmnode.PluginTypes) {
                    _allPlugins.Add(new Plugin(this, asmnode.Assembly, t));
                }
			}
		}

		private void Load() {
			_idToPlugin = new TypedHashtable<string, Plugin>();
			foreach(Plugin p in _allPlugins) {

				if(p.TryToLoad()==GenericResult.Failed) {
					_tracer.Trace("PluginManager.Messages.BootWithoutThisPlugin", p.TypeName);
					continue;
				}

                string id = p.PluginInfo.ID;
                if(_idToPlugin.Contains(id)) {
					_tracer.Trace("PluginManager.Messages.IDDuplication", p.TypeName, p.PluginInfo.ID);
					continue;
				}
				

                if(_disabledPluginIDs.Contains(id)) {
                    p.Disable();
                    continue;
                }

                _idToPlugin.Add(id, p);
            }
		}

		private void Order() {
			List<Plugin> unordered = new List<Plugin>();
			_orderedPlugins = new List<Plugin>();
			
			//TODO ȒP̂߂ɃRȀԂ̓YĐݒ肵@Ƃ΁Aidorg.poderosa.coreŎn܂̂͗D悷Ȃ
			foreach(Plugin p in _allPlugins) {
				if(p.Status==PluginStatus.Loaded) {
					string d = p.PluginInfo.Dependencies;
					if(d==null || d.Length==0) {
						_orderedPlugins.Add(p); //ɂˑĂȂz͐ɓĂ
						continue;
					}

					string[] t = d.Split(';');
					Plugin[] dependencies = new Plugin[t.Length];
					bool failed = false;
					for(int i=0; i<t.Length; i++) {
						//TODO o[WwT|[gH
                        Plugin r = _idToPlugin[t[i]];
						if(r==null || r.Status==PluginStatus.Disabled) {
							_tracer.Trace("PluginManager.Messages.DependencyNotFound", p.TypeName, t[i]);
							failed = true;
							break;
						}
						dependencies[i] = r;
					}

                    if(failed) {
                        p.Disable();
                    }
                    else { //success
                        p.Dependencies = dependencies;
                        unordered.Add(p);
                    }
				}
			}

			//Ԃ̍쐬
			while(unordered.Count>0) {
				bool found = false;
				for(int i=0; i<unordered.Count; i++) {
					Plugin p = unordered[i];
                    Plugin dep = FindDisabledPlugin(p.Dependencies);
                    if(dep!=null) {
                        _tracer.Trace("PluginManager.Messages.DependencyNotFound", p.TypeName, dep.PluginInfo.ID);
                        unordered.RemoveAt(i); //߂Ȃ̂Ă邱ƂB̂Ƃorderedɂ͂ꂸɔ
                        found = true;
                        break; //for𔲂
                    }

					if(AllContainedInOrderedPlugins(p.Dependencies)) {
						unordered.RemoveAt(i);
						_orderedPlugins.Add(p);
						found = true;
						break; //for𔲂
					}
				}

				if(!found) { //ꏄďłȂzˑ
					_tracer.Trace("PluginManager.Messages.DependencyLoopError", FormatIDs(unordered));
					break; //while𔲂
				}
			}
		}

		private void Instantiate() {
			foreach(Plugin p in _orderedPlugins) {
				try {
                    if(p.Status==PluginStatus.Disabled) continue;

					if(p.Instantiate()==GenericResult.Failed) {
						_tracer.Trace("PluginManager.Messages.PluginInitializeFailed", p.PluginInfo.ID);
						continue;
					}
                    _currentInitializingPlugin = p;
					p.Instance.InitializePlugin(_world);
                    _currentInitializingPlugin = null;
				}
				catch(Exception ex) {
					_tracer.Trace("PluginManager.Messages.PluginInitializeFailed", p.PluginInfo.ID);
					_tracer.Trace(ex);
				}
			}
		}

		private bool AllContainedInOrderedPlugins(Plugin[] ps) {
			foreach(Plugin p in ps)
				if(!_orderedPlugins.Contains(p)) return false;
			return true;
		}
        private Plugin FindDisabledPlugin(Plugin[] ps) {
            foreach(Plugin p in ps)
                if(p.Status==PluginStatus.Disabled) return p;
            return null;
        }

		private static string FormatIDs(ICollection<Plugin> ps) {
			StringBuilder bld = new StringBuilder();
			foreach(Plugin p in ps) {
				if(bld.Length > 0) bld.Append(';');
				bld.Append(p.PluginInfo.ID);
			}
			return bld.ToString();
		}

        #region IPlugin
        public void InitializePlugin(IPoderosaWorld poderosa) {
        }

        public void TerminatePlugin() {
        }
        #endregion

        #region IAdaptable
        public IAdaptable GetAdapter(Type adapter) {
            return _world.AdapterManager.GetAdapter(this, adapter);
        }
        #endregion

        public IPoderosaWorld PoderosaWorld {
            get {
                return _world;
            }
        }

        #region IPluginInspector
        public IEnumerable<IPluginInfo> Plugins {
            get {
                return new ConvertingEnumerable<IPluginInfo>(_orderedPlugins);
            }
        }

        public IEnumerable<IExtensionPoint> ExtensionPoints {
            get {
                return new ConvertingEnumerable<IExtensionPoint>(_idToExtensionPoint.Values);
            }
        }
        public IPluginInfo GetPluginInfo(IPlugin plugin) {
            foreach(Plugin p in _allPlugins) {
                if(p.Instance==plugin) return p;
            }
            return null;
        }
        #endregion
    }
}
