/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: ToolBarEx.cs,v 1.11 2006/08/23 06:08:53 dan-iwasaki Exp $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;

using Poderosa.Commands;

namespace Poderosa.Forms {
    /// <summary>
    /// <ja>
    /// c[o[Ɋ܂܂vfC^[tFCXłB
    /// </ja>
    /// <en>
    /// Base interface that shows element included in toolbar.
    /// </en>
    /// </summary>
    public interface IToolBarElement : IAdaptable {
        /// <summary>
        /// <ja>
        /// vf̃c[`bveLXg܂B
        /// </ja>
        /// <en>
        /// The tooltip text of the element is shown. 
        /// </en>
        /// </summary>
        string ToolTipText { get; }
    }

    /// <summary>
    /// <ja>
    /// c[o[̃xC^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows label in toolbar.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂xꍇÃC^[tFCXIuWFNgɁA<seealso cref="ToolBarLabelImpl">ToolBarLabelImpl</seealso>gƂł܂B
    /// </ja>
    /// <en>
    /// <seealso cref="ToolBarLabelImpl">ToolBarLabelImpl</seealso> can be used instead of making the object that mounts this interface when the developer creates labels. 
    /// </en>
    /// </remarks>
    public interface IToolBarLabel : IToolBarElement {
        /// <summary>
        /// <ja>x̃eLXgłB</ja>
        /// <en>Text of the label.</en>
        /// </summary>
        string Text { get; }
        /// <summary>
        /// <ja>x̕łBPʂ̓sNZłB</ja>
        /// <en>Width of the label. The unit is a pixel. </en>
        /// </summary>
        int Width { get; }
    }

    /// <summary>
    /// <ja>
    /// c[o[̃{^C^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows button in toolbar.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂{^ꍇÃC^[tFCXIuWFNgɁA<seealso cref="ToolBarCommandButtonImpl">ToolBarCommandButtonImpl</seealso>gƂł܂B
    /// </ja>
    /// <en>
    /// <seealso cref="ToolBarCommandButtonImpl">ToolBarCommandButtonImpl</seealso> can be used instead of making the object that implements this interface when the developer makes the button. 
    /// </en>
    /// </remarks>
    public interface IToolBarCommandButton : IToolBarElement {
        /// <summary>
        /// <ja>
        /// {^NbNꂽƂɎsR}hłB
        /// </ja>
        /// <en>
        /// It is a command executed when the button is clicked. 
        /// </en>
        /// </summary>
        IPoderosaCommand Command { get; }
        /// <summary>
        /// <ja>
        /// {^\ʂɕ\ACRłB
        /// </ja>
        /// <en>
        /// Icon displayed on the surface of the button. 
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ACȒ傫16~16sNZłȂ΂Ȃ܂B
        /// </ja>
        /// <en>
        /// The size of the icon should be 16~16 pixels. 
        /// </en>
        /// </remarks>
        Image Icon { get; }

        //added for bellagio
        string Text { get; }
        ContentAlignment ImageAlign { get; }
    }

    /// <summary>
    /// <ja>
    /// c[o[̃R{{bNXC^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows combobox in toolbar.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂R{{bNXꍇÃC^[tFCXIuWFNgɁA<seealso cref="ToolBarComboBoxImpl">ToolBarComboBoxImpl</seealso>gƂł܂B
    /// </ja>
    /// <en>
    /// <seealso cref="ToolBarComboBoxImpl">ToolBarComboBoxImpl</seealso> can be used instead of making the object that implements this interface when the developer makes the combo box. 
    /// </en>
    /// </remarks>
    public interface IToolBarComboBox : IToolBarElement {
        /// <summary>
        /// <ja>
        /// R{{bNXɕ\IƂȂACełB
        /// </ja>
        /// <en>
        /// Item that becomes choices displayed in combo box.
        /// </en>
        /// </summary>
        string[] Items { get; }
        /// <summary>
        /// <ja>
        /// R{{bNX̕łBPʂ̓sNZłB
        /// </ja>
        /// <en>Width of the combo box. The unit is a pixel. </en>
        /// </summary>
        int Width { get; }
        /// <summary>
        /// <ja>̃R{{bNXIł邩ǂ܂B</ja>
        /// <en>It is shown whether this combo box can be selected. </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>target for execution. </en></param>
        /// <returns><ja>IłƂɂtrueAłȂƂɂfalseԂ܂B</ja><en>False is returned when it is not true so when it is possible to select it. </en></returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is a toolbar that belongs to <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>.
        /// </en>
        /// </remarks>
        bool IsEnabled(ICommandTarget target);
        /// <summary>
        /// <ja>
        /// ݑIĂACẽCfbNXԍԂ܂B
        /// </ja>
        /// <en>
        /// Return the index of the item that has been selected now.
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja><paramref name="target">target</paramref>̃CfbNXʒuԂ܂B</ja>
        /// <en>Return index position of the <paramref name="target">target</paramref></en>
        /// </returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is a toolbar that belongs to <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>.
        /// </en>
        /// </remarks>
        int GetSelectedIndex(ICommandTarget target);
        /// <summary>
        /// <ja>
        /// R{{bNXőIĂIωƂɌĂяo郁\bhłB
        /// </ja>
        /// <en>
        /// Method of call when choices that have been selected in combobox change
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution.</en></param>
        /// <param name="selectedIndex"><ja>[U[IACẽCfbNXԍłB</ja><en>Index of item that user selected.</en></param>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        void OnChange(ICommandTarget target, int selectedIndex);
    }

    /// <summary>
    /// <ja>
    /// c[o[̃gO{^C^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows Toggle button in toolbar.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂gO{^ꍇÃC^[tFCXIuWFNgɁA<seealso cref="ToolBarToggleButtonImpl">ToolBarToggleButtonImpl</seealso>gƂł܂B
    /// </ja>
    /// <en>
    /// <seealso cref="ToolBarToggleButtonImpl">ToolBarToggleButtonImpl</seealso> can be used instead of making the object that implements this interface when the developer makes the toggle button. 
    /// </en>
    /// </remarks>
    public interface IToolBarToggleButton : IToolBarElement {
        /// <summary>
        /// <ja>
        /// {^\ʂɕ\ACRłB
        /// </ja>
        /// <en>
        /// Icon displayed on surface of button
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ACȒ傫16~16sNZłȂ΂Ȃ܂B
        /// </ja>
        /// <en>
        /// The size of the icon should be 16~16 pixels. 
        /// </en>
        /// </remarks>
        Image Icon { get; }
        /// <summary>
        /// <ja>̃gO{^Ił邩ǂ܂B</ja>
        /// <en>It is shown whether this toggle button can be selected. </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja>IłƂɂtrueAłȂƂɂfalseԂ܂B</ja><en>False is returned when it is not true so when it is possible to select it. </en></returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        bool IsEnabled(ICommandTarget target);
        /// <summary>
        /// <ja>̃gO{^̃I^It̏Ԃ܂B</ja>
        /// <en>The state of on/off of this toggle button is shown. </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja>
        /// <en>Target for execution</en>
        /// </param>
        /// <returns><ja>ÎƂiłԁĵƂɂtrueAItłƂiłȂԁĵƂɂfalseԂ܂B</ja>
        /// <en>False is returned when it is true off (state that doesn't dent) when turning it on (state that has dented). </en>
        /// </returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        bool IsChecked(ICommandTarget target);
        /// <summary>
        /// <ja>
        /// gO{^̃I^It̏ԂςƂɌĂяo郁\bhłB
        /// </ja>
        /// <en>
        /// Method of the call when the state of on/off of the toggle button changes. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <param name="is_checked"><ja>IɂꂽƂɂtrueAItɂꂽƂɂfalsełB</ja>
        /// <en>When turned off true, it is false when turned on. </en>
        /// </param>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is a toolbar that belongs to <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>.
        /// </en>
        /// </remarks>
        void OnChange(ICommandTarget target, bool is_checked);
    }

    //New for Bellagio
    public interface IToolBarDropDownButton : IToolBarElement {
        IPoderosaMenuItem[] Items { get; } //bIMenuItemzBj[`łȂƂl邪...
        string Text { get; }
        Image Icon { get; } //{^̃ACR
        bool IsEnabled(ICommandTarget target);
    }

    /// <summary>
    /// <ja>
    /// c[o[C^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows toolbar.
    /// </en>
    /// </summary>
    public interface IToolBar : IAdaptable {
        /// <summary>
        /// <ja>c[o[EBhEłB</ja>
        /// <en>Window to which toolbar belongs</en>
        /// </summary>
        IPoderosaMainWindow ParentWindow { get; }
        /// <summary>
        /// <ja>ׂĂ̗vfĕ`悵܂B</ja>
        /// <en>It draws in all elements again. </en>
        /// </summary>
        void RefreshAll();
        /// <summary>
        /// <ja>w肵R|[lgĕ`悵܂B</ja>
        /// <en>It draws in the specified component again. </en>
        /// </summary>
        /// <param name="component"><ja>ĕ`悵R|[lg</ja><en>Component where it wants to draw again</en></param>
        void RefreshComponent(IToolBarComponent component);
        /// <summary>
        /// <ja>
        /// c[o[̈ʒu𕶎Ƃč\̂Ԃ܂B
        /// </ja>
        /// <en>
        /// Return the one that the position of the toolbar was composed as a character string.
        /// </en>
        /// </summary>
        /// <returns><ja>c[o[̈ʒu</ja><en>Character string that makes position of toolbar format</en></returns>
        /// <remarks>
        /// <ja>
        /// ̃\bh߂ꂽ́APreferenceɃc[o[ʒuۑƂɎg܂B
        /// </ja>
        /// <en>
        /// When the toolbar position is preserved in Preference, the character string returned from this method is used. 
        /// </en>
        /// </remarks>
        string FormatLocations();
    }

    /// <summary>
    /// <ja>
    /// c[o[R|[lgC^[tFCXłB
    /// </ja>
    /// <en>
    /// Interface that shows toolbar component.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// c[o[̃xA{^AgO{^AR{{bNX́A<see cref="ToolBarElements">ToolBarElemnentsvpeB</see>ɔzƂĊi[܂B
    /// </ja>
    /// <en>
    /// The label, the button, the toggle button, and the combobox in the toolbar are stored in the <see cref="ToolBarElements">ToolBarElemnents property</see> as an array. 
    /// </en>
    /// </remarks>
    /// <example>
    /// <ja>
    /// c[o[R|[lg܂B
    /// <code>
    /// [assembly: PluginDeclaration(typeof(MyPlugin.HelloWorldPlugin))]
    /// namespace MyPlugin
    /// {
    ///    [PluginInfo(ID="jp.example.helloworld", Version="1.0",
    ///        Dependencies="org.poderosa.core.window")]
    ///
    ///    // ł̓vOCgIToolBarComponent
    ///    internal class HelloWorldPlugin : PluginBase, IToolBarComponent
    ///    {
    ///        private IToolBarElement[] _elements;
    ///
    ///        public override void InitializePlugin(IPoderosaWorld poderosa)
    ///        {
    ///            base.InitializePlugin(poderosa);
    ///            
    ///            // i1jR}hIuWFNgpӂ
    ///            PoderosaCommandImpl btncommand = new PoderosaCommandImpl(
    ///              delegate(ICommandTarget target)
    ///              {
    ///                  // sꂽƂ̃R}h
    ///                  MessageBox.Show("{^NbN܂");
    ///                  return CommandResult.Succeeded;
    ///              },
    ///              delegate(ICommandTarget target)
    ///              {
    ///                  // R}hsł邩ǂfQ[g
    ///                  return true;
    ///              }
    ///            );
    ///
    ///            // i2jvfƂă{^imyImage16~16̃rbg}bvj
    ///            System.Drawing.Image myImage = 
    ///              new System.Drawing.Bitmap("摜t@C");
    ///            ToolBarCommandButtonImpl btn = ]
    ///              new ToolBarCommandButtonImpl(btncommand, myImage);
    ///
    ///            // vfƂĐݒ
    ///            _elements = new IToolBarElement[]{ btn };
    ///
    ///            // i3jg|Cgēo^
    ///            IExtensionPoint toolbarExt = 
    ///              poderosa.PluginManager.FindExtensionPoint("org.poderosa.core.window.toolbar");
    ///            // o^
    ///            toolbarExt.RegisterExtension(this);
    ///        }
    ///
    ///        public IToolBarElement[] ToolBarElements
    ///        {
    ///            // vfԂ
    ///            get { return _elements; }
    ///        }
    ///    }
    ///}
    /// </code>
    /// </ja>
    /// <en>
    /// The example of making the toolbar component is shown. 
    /// <code>
    /// [assembly: PluginDeclaration(typeof(MyPlugin.HelloWorldPlugin))]
    /// namespace MyPlugin
    /// {
    ///    [PluginInfo(ID="jp.example.helloworld", Version="1.0",
    ///        Dependencies="org.poderosa.core.window")]
    ///
    ///    // Here, implmenent IToolBarComponent to this plug-in.
    ///    internal class HelloWorldPlugin : PluginBase, IToolBarComponent
    ///    {
    ///        private IToolBarElement[] _elements;
    ///
    ///        public override void InitializePlugin(IPoderosaWorld poderosa)
    ///        {
    ///            base.InitializePlugin(poderosa);
    ///            
    ///            // (1) Prepare the command object.
    ///            PoderosaCommandImpl btncommand = new PoderosaCommandImpl(
    ///              delegate(ICommandTarget target)
    ///              {
    ///                  // Command when executed.
    ///                  MessageBox.Show("Button is clicked.");
    ///                  return CommandResult.Succeeded;
    ///              },
    ///              delegate(ICommandTarget target)
    ///              {
    ///                  // Delegate that shows whether command can be executed
    ///                  return true;
    ///              }
    ///            );
    ///
    ///            // (2)Create the button as element (myImage is a bitmap that size is 16x16)
    ///            System.Drawing.Image myImage = 
    ///              new System.Drawing.Bitmap("Graphics file name.");
    ///            ToolBarCommandButtonImpl btn = ]
    ///              new ToolBarCommandButtonImpl(btncommand, myImage);
    ///
    ///            // Set as element.
    ///            _elements = new IToolBarElement[]{ btn };
    ///
    ///            // (3)Retrieve the extension point and regist.
    ///            IExtensionPoint toolbarExt = 
    ///              poderosa.PluginManager.FindExtensionPoint("org.poderosa.core.window.toolbar");
    ///            // Regist
    ///            toolbarExt.RegisterExtension(this);
    ///        }
    ///
    ///        public IToolBarElement[] ToolBarElements
    ///        {
    ///            // Return the element.
    ///            get { return _elements; }
    ///        }
    ///    }
    ///}
    /// </code>
    /// </en>
    /// </example>
    public interface IToolBarComponent : IAdaptable {
        /// <summary>
        /// <ja>
        /// c[o[R|[lgɊ܂܂AxA{^AgO{^AR{{bNX̔złB
        /// </ja>
        /// <en>
        /// Inclusion in toolbar component, and array of label, button, toggle button, and combobox
        /// </en>
        /// </summary>
        IToolBarElement[] ToolBarElements { get; }

        Size BarImageSize { get; } //New for Bellagio
    }

    //eToolBarElement̕W
    /// <summary>
    /// <ja>
    /// c[o[̗vf̊ƂȂNXłB
    /// </ja>
    /// <en>
    /// Class that becomes base of element of toolbar.
    /// </en>
    /// </summary>
    public abstract class ToolBarElementImpl : IToolBarElement {
        public virtual IAdaptable GetAdapter(Type adapter) {
            return WindowManagerPlugin.Instance.PoderosaWorld.AdapterManager.GetAdapter(this, adapter);
        }
        /// <summary>
        /// <ja>
        /// c[`bveLXgԂ܂B
        /// </ja>
        /// <en>
        /// Return the tooltip text.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ftHgł́A當i""jԂ悤ɎĂ܂BKvɉăI[o[ChĂB
        /// </ja>
        /// <en>
        /// In default, to return the null character (""), it is implemente. Please do override if necessary. 
        /// </en>
        /// </remarks>
        public virtual string ToolTipText {
            get {
                return "";
            }
        }
    }
    /// <summary>
    /// <ja>
    /// c[o[vf̃x\@\񋟂܂B
    /// </ja>
    /// <en>
    /// The function to compose the label of the toolbar element is offered. 
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂͂̃NXp邱ƂŁA<seealso cref="IToolBarLabel">IToolBarLabel</seealso>
    /// IuWFNg\ł܂B
    /// </ja>
    /// <en>
    /// The developer can compose the object that has <seealso cref="IToolBarLabel">IToolBarLabel</seealso> by using this class. 
    /// </en>
    /// </remarks>
    public class ToolBarLabelImpl : ToolBarElementImpl, IToolBarLabel {
        /// <summary>
        /// <ja>
        /// J`ϐłB
        /// </ja>
        /// <en>
        /// Internal variable that shows culture information
        /// </en>
        /// </summary>
        protected StringResource _res;
        /// <summary>
        /// <ja>
        /// J`gǂϐłB
        /// </ja>
        /// <en>
        /// It is an internal variable that shows whether to use culture information. 
        /// </en>
        /// </summary>
        protected bool _usingStringResource;
        /// <summary>
        /// <ja>
        /// xϐłBPʂ̓sNZłB
        /// </ja>
        /// <en>
        /// It is an internal variable that shows the width of the label. The unit is a pixel. 
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ̃vpeB̒l<seealso cref="Width">WidthvpeB</seealso>Ԃ܂B
        /// </ja>
        /// <en>
        /// The value of this property is returned by the Width property. 
        /// </en>
        /// </remarks>
        protected int _width;
        /// <summary>
        /// <ja>
        /// xƂĕ\eLXgێϐłB
        /// </ja>
        /// <en>
        /// It is an internal variable that holds the text to show as the label.
        /// </en>
        /// </summary>
        protected string _text;

        /// <summary>
        /// <ja>
        /// ̃x쐬܂B
        /// </ja>
        /// <en>
        /// Create a null label.
        /// </en>
        /// </summary>
        /// <overloads>
        /// <summary>
        /// <ja>
        /// c[o[̃x쐬܂B
        /// </ja>
        /// <en>
        /// Create a label of toolbar.
        /// </en>
        /// </summary>
        /// </overloads>
        public ToolBarLabelImpl() {
        }

        /// <summary>
        /// <ja>J`w肵ăx쐬܂B</ja>
        /// <en>Create label specified with culture information</en>
        /// </summary>
        /// <param name="res"><ja>J`ł</ja><en>Culture information.</en></param>
        /// <param name="text"><ja>xɕ\eLXgIDłB</ja><en>The text ID to show on the label.</en></param>
        /// <param name="width"><ja>x̕łBPʂ̓sNZłB</ja><en>WIdth of the label. The unit is a pixel.</en></param>

        public ToolBarLabelImpl(StringResource res, string text, int width) {
            _res = res;
            _usingStringResource = true;
            _text = text;
            _width = width;
        }

        /// <summary>
        /// <ja>
        /// eLXgƕw肵ăx쐬܂B
        /// </ja>
        /// <en>
        /// Create the label specifying the text and width. 
        /// </en>
        /// </summary>
        /// <param name="text"><ja>xɕ\eLXgłB</ja><en>The text to show on the label.</en></param>
        /// <param name="width"><ja>x̕łBPʂ̓sNZłB</ja><en>WIdth of the label. The unit is a pixel.</en></param>
        public ToolBarLabelImpl(string text, int width) {
            _usingStringResource = false;
            _text = text;
            _width = width;
        }
        
        /// <summary>
        /// <ja>
        /// xɕ\eLXgԂ܂B
        /// </ja>
        /// <en>
        /// Return the text to show on the label.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// J`t̃RXgN^ō쐬ꂽꍇɂ́A<seealso cref="StringResource">StringResource</seealso>
        /// <see cref="StringResource.GetString">GetString\bh</see>Ăяoꂽʂ߂܂B
        /// </ja>
        /// <en>
        /// The result that the <see cref="StringResource.GetString">GetString method</see> of <seealso cref="StringResource">StringResource</seealso> is called returns when made by the constructor with culture information. 
        /// </en>
        /// </remarks>
        public virtual string Text {
            get { return _usingStringResource? _res.GetString(_text) : _text; }
        }

        /// <summary>
        /// <ja>
        /// xԂ܂BPʂ̓sNZłB
        /// </ja>
        /// <en>
        /// Return the width of the label. The unit is a pixel. 
        /// </en>
        /// </summary>
        public virtual int Width {
            get { return _width; }
        }
    }

    /// <summary>
    /// <ja>
    /// c[o[vf̃{^\@\񋟂܂B
    /// </ja>
    /// <en>
    /// Offer the function to compose the button of the toolbar element.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂͂̃NXp邱ƂŁA<seealso cref="IToolBarCommandButton">IToolBarCommandButton</seealso>
    /// IuWFNg\ł܂B
    /// </ja>
    /// <en>
    /// The developer can compose the object that has <seealso cref="IToolBarCommandButton">IToolBarCommandButton</seealso> by using this class. 
    /// </en>
    /// </remarks>
    public class ToolBarCommandButtonImpl : ToolBarElementImpl, IToolBarCommandButton {
        /// <summary>
        /// <ja>
        /// c[o[NbNꂽƂɎsR}hłB
        /// </ja>
        /// <en>
        /// Command executed when toolbar is clicked.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// R}h̓RXgN^ɂĐݒ肳A<seealso cref="Command">CommandvpeB</seealso>ŕԂ܂B
        /// </ja>
        /// <en>
        /// The command is set by the constractor, and return by <seealso cref="Command">Command property</seealso>.
        /// </en>
        /// </remarks>
        protected IPoderosaCommand _command;
        /// <summary>
        /// <ja>
        /// ACRێϐłBACȒ傫16~16sNZłȂ΂Ȃ܂B
        /// </ja>
        /// <en>
        /// Inside variable that holds the icon. The size of the icon should be 16~16 pixels. 
        /// </en>
        /// </summary>
        /// <ja>
        /// ACR̓RXgN^ɂĐݒ肳A<seealso cref="Icon">IconvpeB</seealso>ŕԂ܂B
        /// </ja>
        /// <en>
        /// The icon is set by the constractor, and return by <seealso cref="Command">Command property</seealso>.
        /// </en>
        protected Image _icon;

        protected string _text;
        protected ContentAlignment _imageAlign;

        /// <summary>
        /// <ja>
        /// c[o[̃{^쐬܂B
        /// </ja>
        /// <en>
        /// Create the button of the toolbar.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// Ȃ̃RXgN^ł́AR}hsꂸAACRݒ肳܂B
        /// </ja>
        /// <en>
        /// For the constructor who doesn't have the argument, as for anything, the command is not executed, and the icon is not set. 
        /// </en>
        /// </remarks>
        public ToolBarCommandButtonImpl() {
        }
        /// <summary>
        /// <ja>
        /// c[o[̃{^쐬܂B
        /// </ja>
        /// <en>
        /// Create the button on the toolbar.
        /// </en>
        /// </summary>
        /// <param name="command"><ja>{^NbNꂽƂɎsR}hłB</ja><en>Command that execuses when the button is clicked</en></param>
        /// <param name="icon"><ja>{^ɕ\ACRłBACȒ傫16~16hbgłȂ΂Ȃ܂</ja>
        /// <en>
        /// Icon that show on the button. The size of the icon should be 16~16 pixels. 
        /// </en></param>
        public ToolBarCommandButtonImpl(IPoderosaCommand command, Image icon) {
            _command = command;
            _icon = icon;
        }

        /// <summary>
        /// <ja>
        /// {^NbNꂽƂɎsR}h܂B
        /// </ja>
        /// <en>
        /// The command executed when the button is clicked is shown. 
        /// </en>
        /// </summary>
        public virtual IPoderosaCommand Command {
            get {
                return _command;
            }
        }

        /// <summary>
        /// <ja>
        /// {^ɕ\ACRłB
        /// </ja>
        /// <en>
        /// Icon displayed in button
        /// </en>
        /// </summary>
        public virtual Image Icon {
            get {
                return _icon;
            }
        }

        public virtual string Text {
            get {
                return _text;
            }
        }
        public ContentAlignment ImageAlign {
            get {
                return _imageAlign;
            }
        }
    }

    /// <summary>
    /// <ja>
    /// c[o[vf̃R{{bNX\@\񋟂܂B
    /// </ja>
    /// <en>
    /// Offers the function to compose the combobox of the toolbar element. 
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// <para>
    /// J҂͂̃NXp邱ƂŁA<seealso cref="IToolBarComboBox">IToolBarComboBox</seealso>
    /// IuWFNg\ł܂B
    /// </para>
    /// <para>
    /// ̃NX͒ۃNXłAЂȌ`ɂ܂BKvɉăI[o[ChKvłB
    /// </para>
    /// </ja>
    /// <en>
    /// <para>
    /// The developer can compose the object that has <seealso cref="IToolBarComboBox">IToolBarComboBox</seealso> by using this class. 
    /// </para>
    /// <para>
    /// This class is an abstraction class. Therefore, this is a model. Override is necessary if necessary. 
    /// </para>
    /// </en>
    /// </remarks>
    public abstract class ToolBarComboBoxImpl : ToolBarElementImpl, IToolBarComboBox {
        /// <summary>
        /// <ja>
        /// R{{bNX̑IiACejϐłB
        /// </ja>
        /// <en>
        /// Variable that shows item of combobox.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ̒ĺA<seealso cref="Items">ItemsvpeB</seealso>Ԃ܂B
        /// </ja>
        /// <en>
        /// This variable is returned from <seealso cref="Items">Items property</seealso>.
        /// </en>
        /// </remarks>
        protected string[] _items;

        /// <summary>
        /// <ja>
        /// R{{bNX̕łBPʂ̓sNZłB
        /// </ja>
        /// <en>
        /// Width of the combobox. This unit is pixel.
        /// </en>
        /// </summary>
        /// <remarks>
        /// <ja>
        /// ̒ĺA<seealso cref="Width">WidthvpeB</seealso>Ԃ܂B
        /// </ja>
        /// <en>
        /// This variable is returned from <seealso cref="Items">Width property</seealso>.
        /// </en>
        /// </remarks>
        protected int _width;

        /// <summary>
        /// <ja>
        /// R{{bNX̑IiACej܂B
        /// </ja>
        /// <en>
        /// The item of the combobox is shown. 
        /// </en>
        /// </summary>
        public virtual string[] Items {
            get {
                return _items;
            }
        }

        /// <summary>
        /// <ja>
        /// R{{bNX̕łBPʂ̓sNZłB
        /// </ja>
        /// <en>
        /// Return the width of the combobox. The unit is a pixel. 
        /// </en>
        /// </summary>
        public virtual int Width {
            get {
                return _width;
            }
        }

        /// <summary>
        /// <ja>
        /// R{{bNXIł邩ǂ܂B
        /// </ja>
        /// <en>
        /// It is shown whether the combobox can be selected. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja>IłƂɂtrueAłȂƂɂfalseԂ܂B</ja><en>False is returned when it is not true so when it is possible to select it. </en></returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public virtual bool IsEnabled(ICommandTarget target) {
            return true;
        }

        /// <summary>
        /// <ja>
        /// ݑIĂACẽCfbNXԍԂ܂B
        /// </ja>
        /// <en>
        /// The index number of the item that has been selected now is returned. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja><paramref name="target">target</paramref>̃CfbNXʒuԂ܂B</ja>
        /// <en>Return index position of the <paramref name="target">target</paramref></en>
        /// </returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public abstract int GetSelectedIndex(ICommandTarget target);
        /// <summary>
        /// <ja>
        /// R{{bNXőIĂIωƂɌĂяo郁\bhłB
        /// </ja>
        /// <en>
        /// It is a method of the call when choices that have been selected in the combobox change. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <param name="selectedIndex"><ja>[U[IACẽCfbNXԍłB</ja><en>It is an index number of the item that the user selected. </en></param>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public virtual void OnChange(ICommandTarget target, int selectedIndex) {
        }
    }

    /// <summary>
    /// <ja>
    /// c[o[vf̃gO{^\@\񋟂܂B
    /// </ja>
    /// <en>
    /// Offers the function to compose the toggle button of the toolbar element.
    /// </en>
    /// </summary>
    /// <remarks>
    /// <ja>
    /// J҂͂̃NXp邱ƂŁA<seealso cref="IToolBarToggleButton">IToolBarToggleButton</seealso>
    /// IuWFNg\ł܂B
    /// </ja>
    /// <en>
    /// The developer can compose the object that has <seealso cref="IToolBarToggleButton">IToolBarToggleButton</seealso> by using this class. 
    /// </en>
    /// </remarks>
    public abstract class ToolBarToggleButtonImpl : ToolBarElementImpl, IToolBarToggleButton {
        /// <summary>
        /// <ja>
        /// ACRϐłBACȒ傫16~16sNZłȂ΂Ȃ܂B
        /// </ja>
        /// <en>
        /// Inside variable that holds the icon. The size of the icon should be 16~16 pixels. 
        /// </en>
        /// </summary>
        protected Image _icon;

        /// <summary>
        /// <ja>
        /// ACRԂ܂B
        /// </ja>
        /// <en>
        /// Return the icon.
        /// </en>
        /// </summary>
        public virtual Image Icon {
            get {
                return _icon;
            }
        }

        /// <summary>
        /// <ja>
        /// gO{^Ił邩ǂ܂B
        /// </ja>
        /// <en>
        /// It is shown whether the toggle button can be selected. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja>IłƂɂtrueAłȂƂɂfalseԂ܂B</ja><en>False is returned when it is not true so when it is possible to select it. </en></returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public virtual bool IsEnabled(ICommandTarget target) {
            return true;
        }

        /// <summary>
        /// <ja>
        /// gO{^̃I^It̏ԂԂ܂B
        /// </ja>
        /// <en>
        /// Return the state of on/off of the toggle button.
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <returns><ja>ÎƂiłƂjɂtrueAIt̂ƂiłȂƂjɂfalseԂ܂B</ja><en>False is returned when true off (When not denting) when turning it on (When denting). </en></returns>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public virtual bool IsChecked(ICommandTarget target) {
            return false;
        }

        /// <summary>
        /// <ja>
        /// I^It̏ԂωƂɌĂяo郁\bhłB
        /// </ja>
        /// <en>
        /// It is a method of the call when the state of on/off changes. 
        /// </en>
        /// </summary>
        /// <param name="target"><ja>s̑ΏۂƂȂ^[QbgłB</ja><en>Target for execution</en></param>
        /// <param name="is_checked"><ja>I^It̏ԂłBtruêƂɂ̓IiłԁjAfalsêƂɂ̓ItiłȂԁjłB</ja><en>It is a state of on/off. At on (state that has dented) and false at true, it is off (state that doesn't dent). </en></param>
        /// <remarks>
        /// <ja>
        /// <paramref name="target">target</paramref>́Ãc[o[<see cref="IPoderosaMainWindow">IPoderosaMainWindow</see>łB
        /// </ja>
        /// <en>
        /// <paramref name="target">target</paramref> is <see cref="IPoderosaMainWindow">IPoderosaMainWindow</see> that this toolbar belongs. 
        /// </en>
        /// </remarks>
        public abstract void OnChange(ICommandTarget target, bool is_checked);
    }
}
