/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: ToolBar.cs,v 1.7 2006/11/03 09:17:55 okajima Exp $
 */
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;

using Poderosa.UI;
using Poderosa.Commands;
using Poderosa.Util;

namespace Poderosa.Forms {
    internal class PoderosaToolStripContainer : ToolStripContainer, IToolBar {
        private IPoderosaMainWindow _parent;
        private string _initialLocationInfo;

        private List<ToolStrip> _toolStrips;
        private ToolStrip _currentToolStrip;

        public PoderosaToolStripContainer(IPoderosaMainWindow parent, string location_info) {
            _parent = parent;
            _initialLocationInfo = location_info;

            //this.Height = 25; //K؂ȃTCY߂@킩Ȃ
            this.Dock = DockStyle.Fill;
            //Eɓ̂͌ꂵB߂B
            this.LeftToolStripPanelVisible = false;
            this.RightToolStripPanelVisible  = false;
            this.BottomToolStripPanelVisible = false;
            this.TopToolStripPanelVisible = WindowManagerPlugin.Instance.WindowPreference.OriginalPreference.ShowsToolBar;
            this.AllowDrop = true;
            CreateInternal();
        }

        private void CreateInternal() {
            IToolBarComponent[] components = (IToolBarComponent[])WindowManagerPlugin.Instance.PoderosaWorld.PluginManager.FindExtensionPoint(WindowManagerConstants.TOOLBARCOMPONENT_ID).GetExtensions();

            _toolStrips = new List<ToolStrip>();
            string[] locations = _initialLocationInfo.Split(',');
            this.TopToolStripPanel.Size = new Size(TryParseInt(locations, 0), TryParseInt(locations, 1));
            int index = 1;
            foreach(IToolBarComponent comp in PositionDesignationSorter.SortItems(components)) {
                //Location̔
                Point pt = new Point(TryParseInt(locations, index*2), TryParseInt(locations, index*2+1));
                CreateToolBarComponent(comp, pt);
                index++;
            }
        }

        //ʒuw̓tH[OnLoadɂȂႢ̂B߂ǂ
        public void RestoreLayout() {
            ICommandTarget target = (ICommandTarget)_parent.GetAdapter(typeof(ICommandTarget));

            ToolStripPanel panel = this.TopToolStripPanel;
            panel.BeginInit();
            panel.SuspendLayout();
            bool location_available = _initialLocationInfo.Length>0;

            //ToolStripPanelւ̒ǉ͂ȂsvcB
            //BeginInitSuspendLayoutĂԂǂłlqႤB
            //Ȃ̂ł₢ANȂǈʒu񂪂ȂƂControls.AddRangeňꊇo^.NETɔCAȍ~͈ʒuwƂjł

            if(!location_available)
                panel.Controls.AddRange(_toolStrips.ToArray());
            
            foreach(ToolStrip t in _toolStrips) {
                if(location_available) panel.Join(t, t.Location);
                foreach(ToolStripItem c in t.Items) {
                    ControlTagBase tag = c.Tag as ControlTagBase;
                    if(tag!=null) {
                        RefreshElement(c, tag, target);
                    }
                }
            }
            
            panel.ResumeLayout();
            panel.EndInit();
        }
        public void ReloadPreference(ICoreServicePreference pref) {
            this.TopToolStripPanelVisible = pref.ShowsToolBar;
        }
        public void Reload() {
            //{͑S\zׂ
            RefreshAll();
        }

        //݂Top
        private ControlCollection GetContents() {
            return this.TopToolStripPanel.Controls;
        }

        private void CreateToolBarComponent(IToolBarComponent comp, Point pt) {
            //̒IToolBar̊e\bhĂ΂A̓o^s
            _currentToolStrip = new ToolStrip();
            _currentToolStrip.ImageScalingSize = comp.BarImageSize;

            IToolBarElement[] elements = comp.ToolBarElements;
            foreach(IToolBarElement e in elements) {
                if(e is IToolBarCommandButton)
                    DefineCommandButton(comp, (IToolBarCommandButton)e);
                else if(e is IToolBarLabel)
                    DefineLabel(comp, (IToolBarLabel)e);
                else if(e is IToolBarComboBox)
                    DefineComboBox(comp, (IToolBarComboBox)e);
                else if(e is IToolBarToggleButton)
                    DefineToggleButton(comp, (IToolBarToggleButton)e);
                else if(e is IToolBarDropDownButton)
                    DefineDropDownButton(comp, (IToolBarDropDownButton)e);
                else
                    throw new ArgumentException("Unexpected IToolBarElement type");
            }
            _currentToolStrip.Location = pt;
            _toolStrips.Add(_currentToolStrip);
            Debug.WriteLineIf(DebugOpt.BuildToolBar, "toolbar " + comp.GetType().Name + " location="+_currentToolStrip.Location.ToString());
        }


        private void DefineCommandButton(IToolBarComponent comp, IToolBarCommandButton element) {
            ToolStripButton button = new ToolStripButton();
            button.Image = element.Icon;
            string text = element.Text;
            if(text!=null && text.Length > 0) {
                button.ImageAlign = element.ImageAlign;
                button.Text = text;
            }
            button.Tag = new ButtonTag(GetCommandTarget(), comp, element);
            button.Size = button.Image.Size;
            button.Click += delegate(object sender, EventArgs args) { DoCommand(element.Command); };
            string tooltip = element.ToolTipText;
            if(tooltip!=null)
                button.ToolTipText = tooltip;

            _currentToolStrip.Items.Add(button);
        }
        private void DefineLabel(IToolBarComponent comp, IToolBarLabel element) {
            ToolStripLabel label = new ToolStripLabel();
            label.Width = element.Width;
            label.Text = element.Text;
            label.Tag = new LabelTag(comp, element);
            label.TextAlign = ContentAlignment.MiddleRight;
            _currentToolStrip.Items.Add(label);
        }
        private void DefineComboBox(IToolBarComponent comp, IToolBarComboBox element) {
            ToolStripComboBox cb = new ToolStripComboBox();
            cb.Items.AddRange(element.Items);
            cb.Size = new Size(element.Width, cb.Height); //Width𒼐ڐݒ肵Ă炵BȂ񂶂Ⴂ
            cb.DropDownStyle = ComboBoxStyle.DropDownList;
            ComboBoxTag tag = new ComboBoxTag(GetCommandTarget(), comp, element);
            cb.Tag = tag;
            cb.SelectedIndexChanged += new EventHandler(tag.OnSelectedIndexChanged);
            if(!RuntimeUtil.IsZeroLength(element.ToolTipText)) cb.ToolTipText = element.ToolTipText;

            _currentToolStrip.Items.Add(cb);
        }
        private void DefineToggleButton(IToolBarComponent comp, IToolBarToggleButton element) {
            ToolStripButton tb = new ToolStripButton();
            tb.Image = element.Icon;
            ToggleButtonTag tag = new ToggleButtonTag(GetCommandTarget(), comp, element);
            tb.Tag = tag;
            tb.Click += new EventHandler(tag.OnClick);
            if(!RuntimeUtil.IsZeroLength(element.ToolTipText)) tb.ToolTipText = element.ToolTipText;

            _currentToolStrip.Items.Add(tb);
        }
        private void DefineDropDownButton(IToolBarComponent comp, IToolBarDropDownButton element) {
            ToolStripDropDownButton btn = new ToolStripDropDownButton();
            btn.Image = element.Icon;
            btn.Text = element.Text;
            DropDownButtonTag tag = new DropDownButtonTag(GetCommandTarget(), comp, element);
            btn.Tag = tag;
            btn.DropDownOpening += new EventHandler(tag.OnDropDown);
            if(!RuntimeUtil.IsZeroLength(element.ToolTipText)) btn.ToolTipText = element.ToolTipText;

            _currentToolStrip.Items.Add(btn);
        }

        #region IToolBar
        //UIAdjustment
        public void RefreshComponent(IToolBarComponent component) {
            ICommandTarget target = (ICommandTarget)_parent.GetAdapter(typeof(ICommandTarget));
            foreach(ToolStrip st in GetContents()) {
                //TODO ^O̊֘AtHvł
                foreach(ToolStripItem c in st.Items) {
                    ControlTagBase tag = c.Tag as ControlTagBase;
                    if(tag!=null && tag.OwnerComponent==component) {
                        RefreshElement(c, tag, target);
                    }
                }
            }
        }
        public void RefreshAll() {
            ICommandTarget target = (ICommandTarget)_parent.GetAdapter(typeof(ICommandTarget));
            foreach(ToolStrip st in GetContents()) {
                foreach(ToolStripItem c in st.Items) {
                    ControlTagBase tag = c.Tag as ControlTagBase;
                    if(tag!=null) {
                        RefreshElement(c, tag, target);
                    }
                }
            }
        }
        public IPoderosaMainWindow ParentWindow {
            get {
                return _parent;
            }
        }
        public string FormatLocations() {
            StringBuilder bld = new StringBuilder();
            //ŏ̓TCY
            bld.Append(this.TopToolStripPanel.Width.ToString());
            bld.Append(',');
            bld.Append(this.TopToolStripPanel.Height.ToString());
            foreach(ToolStrip st in _toolStrips) {
                bld.Append(',');
                bld.Append(st.Location.X.ToString());
                bld.Append(',');
                bld.Append(st.Location.Y.ToString());
            }
            return bld.ToString();
        }
        #endregion

        private void RefreshElement(ToolStripItem c, ControlTagBase tag, ICommandTarget target) {
            tag.Refresh(c);
        }

        #region IAdaptable
        public IAdaptable GetAdapter(Type adapter) {
            return WindowManagerPlugin.Instance.PoderosaWorld.AdapterManager.GetAdapter(this, adapter);
        }
        #endregion

        private void DoCommand(IPoderosaCommand command) {
            try {
                ICommandTarget target = GetCommandTarget();
                Debug.Assert(target!=null);
                if(command.CanExecute(target))
                    CommandManagerPlugin.Instance.Execute(command, target);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        private ICommandTarget GetCommandTarget() {
            return (ICommandTarget)_parent.GetAdapter(typeof(ICommandTarget));
        }

        private static int TryParseInt(string[] values, int index) {
            if(values.Length <= index)
                return 0;
            else
                return ParseUtil.ParseInt(values[index], 0);
        }

        //UI
        protected override void OnCreateControl() {
            base.OnCreateControl();
            this.RefreshAll();
        }
        protected override void OnDragEnter(DragEventArgs drgevent) {
            base.OnDragEnter(drgevent);
            try {
                WindowManagerPlugin.Instance.BypassDragEnter(this.ParentForm, drgevent);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }
        protected override void OnDragDrop(DragEventArgs drgevent) {
            base.OnDragDrop(drgevent);
            try {
                WindowManagerPlugin.Instance.BypassDragDrop(this.ParentForm, drgevent);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        private abstract class ControlTagBase {
            private IToolBarComponent _ownerComponent;

            public ControlTagBase(IToolBarComponent component) {
                _ownerComponent = component;
            }

            public IToolBarComponent OwnerComponent {
                get {
                    return _ownerComponent;
                }
            }

            public abstract void Refresh(ToolStripItem c);
        }

        private class ButtonTag : ControlTagBase {
            private IToolBarCommandButton _commandButton;
            private IGeneralCommand _generalCommand;
            private ICommandTarget _target;

            public ButtonTag(ICommandTarget target, IToolBarComponent owner, IToolBarCommandButton cmdButton) : base(owner) {
                _target = target;
                _commandButton = cmdButton;
                _generalCommand = (IGeneralCommand)_commandButton.Command.GetAdapter(typeof(IGeneralCommand)); //擾łȂnull
            }

            public IPoderosaCommand AssociatedCommand {
                get {
                    return _commandButton.Command;
                }
            }

            public override void Refresh(ToolStripItem c) {
                c.Image = _commandButton.Icon;
                c.Enabled = _commandButton.Command.CanExecute(_target);
                /*
                if(_generalCommand!=null && _generalCommand.Description.Length>0)
                    c.ToolTipText = _generalCommand.Description;
                */
            }
        }

        private class LabelTag : ControlTagBase {
            private IToolBarLabel _label;
            public LabelTag(IToolBarComponent owner, IToolBarLabel label)
                : base(owner) {
                _label = label;
            }
            public override void Refresh(ToolStripItem c) {
                c.Text = _label.Text;
            }
        }

        private class ComboBoxTag : ControlTagBase {
            private IToolBarComboBox _handler;
            private ICommandTarget _target;
            private bool _eventHandlerGuard;

            public ComboBoxTag(ICommandTarget target, IToolBarComponent component, IToolBarComboBox handler) : base(component) {
                _target = target;
                _handler = handler;
            }

            public override void Refresh(ToolStripItem combobox_) {
                Debug.Assert(combobox_.Tag==this);
                _eventHandlerGuard = true;
                try {
                    ToolStripComboBox combobox = (ToolStripComboBox)combobox_;
                    combobox.Enabled = _handler.IsEnabled(_target);
                    combobox.Items.Clear();
                    combobox.Items.AddRange(_handler.Items); //TODO ڐςǂhandlerɐq˂悤ɂł
                    if(combobox.Enabled)
                        combobox.SelectedIndex = _handler.GetSelectedIndex(_target);
                    else
                        combobox.SelectedIndex = -1;
                }
                finally {
                    _eventHandlerGuard = false;
                }
            }

            //NOTE OnChangeRefreshnhĂẪR[hʂ̂ōċAh
            public void OnSelectedIndexChanged(object sender, EventArgs args) {
                if(_eventHandlerGuard) return;
                ToolStripComboBox cb = sender as ToolStripComboBox;
                Debug.Assert(cb!=null && cb.Tag==this);

                try {
                    _eventHandlerGuard = true;
                    if(cb.Enabled) //ꉞK[h
                        _handler.OnChange(_target, cb.SelectedIndex);
                }
                catch(Exception ex) {
                    RuntimeUtil.ReportException(ex);
                }
                finally {
                    _eventHandlerGuard = false;
                }
            }
        }

        private class ToggleButtonTag : ControlTagBase {
            private IToolBarToggleButton _handler;
            private ICommandTarget _target;
            private bool _eventHandlerGuard;

            public ToggleButtonTag(ICommandTarget target, IToolBarComponent component, IToolBarToggleButton handler)
                : base(component) {
                _target = target;
                _handler = handler;
            }

            public override void Refresh(ToolStripItem button_) {
                Debug.Assert(button_.Tag==this);
                ToolStripButton button = (ToolStripButton)button_;
                button.Enabled = _handler.IsEnabled(_target);
                button.Checked = button.Enabled && _handler.IsChecked(_target); //enabledchecked̂߂̕Kv
                button.ToolTipText = _handler.ToolTipText;
            }

            public void OnClick(object sender, EventArgs args) {
                if(_eventHandlerGuard) return;
                ToolStripButton tb = sender as ToolStripButton;
                Debug.Assert(tb!=null && tb.Tag==this);

                try {
                    bool value = tb.Checked;
                    _eventHandlerGuard = true;
                    tb.Checked = !value;
                    _handler.OnChange(_target, !value); //]lZbg
                }
                catch(Exception ex) {
                    RuntimeUtil.ReportException(ex);
                }
                finally {
                    _eventHandlerGuard = false;
                }
            }
        }

        private class DropDownButtonTag : ControlTagBase {
            private IToolBarDropDownButton _handler;
            private ICommandTarget _target;
            public DropDownButtonTag(ICommandTarget target, IToolBarComponent component, IToolBarDropDownButton handler)
                : base(component) {
                _target = target;
                _handler = handler;
            }
            public override void Refresh(ToolStripItem button_) {
                Debug.Assert(button_.Tag==this);
                ToolStripDropDownButton button = (ToolStripDropDownButton)button_;
                button.Enabled = _handler.IsEnabled(_target);
            }
            public void OnDropDown(object sender, EventArgs args) {
                try {
                    ToolStripDropDownButton button = (ToolStripDropDownButton)sender;
                    button.DropDown.Items.Clear();
                    foreach(IPoderosaMenuItem item in _handler.Items) {
                        ToolStripMenuItem mi = new ToolStripMenuItem();
                        mi.Text = item.Text;
                        mi.Checked = item.IsChecked(_target);
                        mi.Enabled = item.IsEnabled(_target);
                        IGeneralCommand cmd = item.AssociatedCommand as IGeneralCommand;
                        if(cmd!=null)
                            mi.ShortcutKeyDisplayString = WinFormsUtil.FormatShortcut(CommandManagerPlugin.Instance.CurrentKeyBinds.GetKey(cmd));
                        DropDownButtonItemTag tag = new DropDownButtonItemTag(_target, item);
                        mi.Click += new EventHandler(tag.OnItemClicked);
                        mi.Tag = tag;
                        button.DropDown.Items.Add(mi);
                    }
                }
                catch(Exception ex) {
                    RuntimeUtil.ReportException(ex);
                }

            }
        }
        private class DropDownButtonItemTag {
            private ICommandTarget _target;
            private IPoderosaMenuItem _item;
            public DropDownButtonItemTag(ICommandTarget target, IPoderosaMenuItem item) {
                _target = target;
                _item = item;
            }
            public void OnItemClicked(object sender, EventArgs args) {
                try {
                    IPoderosaCommand command = _item.AssociatedCommand;
                    if(command.CanExecute(_target))
                        CommandManagerPlugin.Instance.Execute(command, _target);
                }
                catch(Exception ex) {
                    RuntimeUtil.ReportException(ex);
                }
            
            }
        }
    }
}
