/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: Session.cs,v 1.8 2006/08/14 12:41:17 okajima Exp $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

using Poderosa.Util.Collections;
using Poderosa.Plugins;
using Poderosa.Forms;
//using Poderosa.Document;
using Poderosa.Util;

[assembly: PluginDeclaration(typeof(Poderosa.Sessions.SessionManagerPlugin))]

namespace Poderosa.Sessions
{
    //PoderosaMainWindowĂԂ߂̃C^tF[X
    internal interface ISessionManagerForPoderosaWindow {
        PrepareCloseResult CloseMultipleDocuments(ClosingContext context, IPoderosaDocument[] documents);
    }

    internal interface ISessionManagerForViewSplitter {
        void ChangeLastAttachedViewForAllDocuments(IPoderosaView closing_view, IPoderosaView alternative);
        void ChangeLastAttachedViewForWindow(IPoderosaMainWindow window, IPoderosaView alternative);
        void SetLastAttachedViewForAllDocuments(IPoderosaView view);
    }

    [PluginInfo(ID=SessionManagerPlugin.PLUGIN_ID, Version=VersionInfo.PODEROSA_VERSION, Author=VersionInfo.PROJECT_NAME, Dependencies="org.poderosa.core.window")]
	internal class SessionManagerPlugin :
        PluginBase,
        ISessionManager,
        ISessionManagerForViewSplitter
    {
        public const string PLUGIN_ID = "org.poderosa.core.sessions";
        private static SessionManagerPlugin _instance;
        private TypedHashtable<ISession, SessionHost> _sessionMap;
        private TypedHashtable<IPoderosaDocument, DocumentHost> _documentMap;
        private ActivateContext _activateContext;
        private IExtensionPoint _docViewRelationHandler;
        private ListenerList<IActiveDocumentChangeListener> _activeDocumentChangeListeners;
        private ListenerList<ISessionListener> _sessionListeners;

        public override void InitializePlugin(IPoderosaWorld poderosa) {
            _instance = this;
            base.InitializePlugin(poderosa);
            _sessionMap = new TypedHashtable<ISession, SessionHost>();
            _documentMap = new TypedHashtable<IPoderosaDocument, DocumentHost>();
            _docViewRelationHandler = poderosa.PluginManager.CreateExtensionPoint("org.poderosa.core.sessions.docViewRelationHandler", typeof(IDocViewRelationEventHandler), this);
            _activeDocumentChangeListeners = new ListenerList<IActiveDocumentChangeListener>();
            _activeDocumentChangeListeners.Add(new WindowCaptionManager());

            _sessionListeners = new ListenerList<ISessionListener>();
        }
        public override void TerminatePlugin() {
            base.TerminatePlugin();
            //͖ĂȂƁI
            Debug.Assert(_sessionMap.Count==0);
            Debug.Assert(_documentMap.Count==0);
        }

        public static SessionManagerPlugin Instance {
            get {
                return _instance;
            }
        }

        public IEnumerable<ISession> AllSessions {
            get {
                List<ISession> r = new List<ISession>();
                foreach(ISession session in _sessionMap.Keys) r.Add(session);
                return r;
            }
        }

        public IPoderosaDocument[] GetDocuments(IPoderosaMainWindow window) {
            List<IPoderosaDocument> r = new List<IPoderosaDocument>();
            foreach(DocumentHost dh in _documentMap.Values) {
                if(dh.LastAttachedView!=null && dh.LastAttachedView.ParentForm==window) r.Add(dh.Document);
            }
            return r.ToArray();
        }

        public void StartNewSession(ISession session, IPoderosaView firstView) {
            firstView = AdjustToOuterView(firstView);
            SessionHost host = new SessionHost(this, session);
            _sessionMap.Add(session, host);
            session.InternalStart(host);
            foreach(ISessionListener listener in _sessionListeners) listener.OnSessionStart(session);

            //̎_ŁAȂƂhLgȂƂȂBQȏ͕sAłȂ
            if(host.DocumentCount==0) throw new InvalidOperationException("session must register at least one document in InternalStart()");
            AttachDocumentAndView(host.DocumentAt(0), firstView);
        }

        public void AttachDocumentAndView(IPoderosaDocument document, IPoderosaView view) {
            view = AdjustToOuterView(view);
            DocumentHost dh = FindDocumentHost(document);
            Debug.Assert(dh!=null, "the document must be registered by calling ISessionHost#RegisterDocument");

            if(view.Document==document) {
                Debug.Assert(dh.CurrentView==view);
                return; //Ȃ
            }

            IPoderosaView previous_view = dh.CurrentView; //֘AÂw肷hLgƂƌĂr[
            IPoderosaForm last_window = ViewToForm(dh.LastAttachedView); //ƂƂ̏LEBhEB߂ĂAttachłnullł邱Ƃɂイ

            //݂̊֘AU؂
            if(previous_view!=null) {
                Debug.WriteLineIf(DebugOpt.DumpDocumentRelation, "Detach Prev View " + ViewName(previous_view));
                dh.DetachView();
            }
            Debug.Assert(dh.CurrentView==null);

            //ڑɃhLg݂Ă΂؂藣
            IPoderosaDocument existing_doc = view.Document;
            if(existing_doc!=null) { //Ώۂ̃r[ɌÂ̂ЂĂO
                DocumentHost eh = FindDocumentHost(existing_doc);
                Debug.Assert(eh.CurrentView==view);
                Debug.WriteLineIf(DebugOpt.DumpDocumentRelation, String.Format("Detach Destination View doc={0} view={1}", existing_doc.GetType().Name, ViewName(view)));
                eh.DetachView();
            }

            //VK̐ڑ
            Debug.Assert(view.Document==null && dh.CurrentView==null); //AttachłĂ邱ƊmF
            dh.AttachView(view);

            //ړ邱ƂŐVKɌ悤ɂȂhLgT
            if(previous_view!=null && previous_view!=view) {
                DocumentHost new_visible_doc = ShowBackgroundDocument(previous_view);
                Debug.Assert(new_visible_doc!=dh);
            }

            //hLgۗLEBhEωʒmBAttachłlast_mainwindow==nullł邱Ƃɒ
            if(last_window!=view.ParentForm) {
                if(last_window!=null) NotifyRemove(last_window, document);
                NotifyAdd(ViewToForm(view), document);
            }

            FireDocViewRelationChange();
        }

        //ISessionManageȑIn@ׂ̂̓hLg
        public PrepareCloseResult CloseDocument(IPoderosaDocument document) {
            DocumentHost dh = FindDocumentHost(document);
            Debug.Assert(dh!=null);
            SessionHost sh = dh.SessionHost;
            PrepareCloseResult r = sh.Session.PrepareCloseDocument(document);
            if(r==PrepareCloseResult.Cancel) return r;

            CleanupDocument(dh);
            if(r==PrepareCloseResult.TerminateSession || sh.DocumentCount==0) CleanupSession(sh);
            return r;
        }

        public PrepareCloseResult TerminateSession(ISession session) {
            SessionHost sh = FindSessionHost(session);
            PrepareCloseResult r = sh.Session.PrepareCloseSession();
            Debug.Assert(r!=PrepareCloseResult.ContinueSession, "sanity");
            if(r==PrepareCloseResult.Cancel) return r;

            CleanupSession(sh);
            return r;
        }

        public PrepareCloseResult CloseMultipleDocuments(ClosingContext context, IPoderosaDocument[] documents) {
            List<SessionHost> sessions = CreateSessionHostCollection();
            foreach(SessionHost sh in sessions) sh.CMP_ClosingDocumentCount = 0; //JEgZbg

            foreach(IPoderosaDocument doc in documents) {
                DocumentHost dh = FindDocumentHost(doc);
                dh.SessionHost.CMP_ClosingDocumentCount++;
            }
            //܂łŁAeSessionHostƂɉ̃hLg悤ƂĂ邩JEgꂽB
            //ɂꂼɂď͂߂
            PrepareCloseResult result = PrepareCloseResult.TerminateSession;
            foreach(SessionHost sh in sessions) {
                if(sh.CMP_ClosingDocumentCount==0) continue; //eȂ

                if(sh.CMP_ClosingDocumentCount==sh.DocumentCount) { //ZbV̑ShLgꍇ
                    PrepareCloseResult r = TerminateSession(sh.Session);
                    sh.CMP_PrepareCloseResult = r;
                    if(r==PrepareCloseResult.TerminateSession)
                        context.AddClosingSession(sh);
                    else if(r==PrepareCloseResult.Cancel)
                        result = PrepareCloseResult.Cancel; //łLZΑŜLZƂ
                }
                else { //ꕔ̃hLgBꂪʓ|
                    //TODO unsupported
                    Debug.Assert(false, "unsupported");
                }
            }

            //ɂăZbV
            foreach(SessionHost sh in context.ClosingSessions) {
                CleanupSession(sh);
            }

            return result;
        }

        public void RefreshDocumentStatus(IPoderosaDocument document) {
            DocumentHost dh = FindDocumentHost(document);
            Debug.Assert(dh!=null);
            IPoderosaForm f = ViewToForm(dh.LastAttachedView);
            
            //Ɖ
            IPoderosaMainWindow mw = (IPoderosaMainWindow)f.GetAdapter(typeof(IPoderosaMainWindow));
            if(mw!=null)
                mw.DocumentTabFeature.Update(document);
            else
                ((IPoderosaPopupWindow)f.GetAdapter(typeof(IPoderosaPopupWindow))).UpdateStatus();
        }

        private void CleanupDocument(DocumentHost dh) {
            IPoderosaForm owner_window = ViewToForm(dh.LastAttachedView);
            IPoderosaView visible_view = dh.CurrentView;
            bool was_active = false;
            if(visible_view!=null) {
                was_active = visible_view.AsControl().Focused;
                dh.DetachView();
                FireDocViewRelationChange();
            }

            if(owner_window!=null) {
                NotifyRemove(owner_window, dh.Document);
            }

            dh.SessionHost.CloseDocument(dh.Document);
            _documentMap.Remove(dh.Document);

            //hLg̃r[Ăꍇ́Äʒu̕ʂ̃hLg
            //TODO EBhEƂ͕͂̏sv
            if(visible_view!=null && visible_view.ParentForm.GetAdapter(typeof(IPoderosaMainWindow))!=null) {
                ShowBackgroundDocument(visible_view);
                if(was_active && visible_view.Document!=null)
                    ActivateDocument(visible_view.Document, ActivateReason.InternalAction);
            }
        }
        internal void CleanupSession(SessionHost sh) { //SessionHostĂ΂̂internal
            foreach(ISessionListener listener in _sessionListeners) listener.OnSessionEnd(sh.Session);
            foreach(IPoderosaDocument doc in sh.ClonedDocuments) {
                CleanupDocument(FindDocumentHost(doc));
            }
            sh.Session.InternalTerminate();
            _sessionMap.Remove(sh.Session);
        }

        /**
         * Activatẽ[g
         * NOTE dR[͂ŃubN悤ɂB
         * ANeBuȃhLgω̂́A
         *   - ViewNbNătH[JXςƂ
         *   - ^uNbNƂ
         *   - L[{[hV[gJbgAPoderosãR[hƂ
         * ̂RB
         * ̂̂ǂł邩w肵ĂĂԁBႦ΁AFocusړ̂Ƃ͉߂Focus()Ă΂ȂȂǓŏꍇȂ
         */  
        public void ActivateDocument(IPoderosaDocument document, ActivateReason reason) {
            Debug.Assert(document!=null);

            //lXg̖h~ FocusnCxgnhƂǂĂĂ΂Ă܂̂
            if(_activateContext!=null) return;

            try {
                _activateContext = new ActivateContext(document, reason);

                DocumentHost dh = FindDocumentHost(document);
                Debug.Assert(dh!=null);

                if(dh.CurrentView!=null) { //ɌĂꍇ
                    if(reason!=ActivateReason.ViewGotFocus) SetFocusToView(dh.CurrentView); //[ŨtH[JXw肾ꍇ͂ɔC
                }
                else { //Ă͂Ȃꍇ
                    IPoderosaView view = dh.LastAttachedView;
                    Debug.Assert(view!=null); //dg݂ǂɂقBׂ͂ĂDocumentŏAttachDocumentAndView邱Ƃz肵Ă
                    AttachDocumentAndView(document, view);
                    Debug.Assert(dh.CurrentView==view);
                    if(!view.AsControl().Focused) view.AsControl().Focus();
                }

                Debug.Assert(dh.CurrentView.Document==document);


                //ʒm
                NotifyActivation(ViewToForm(dh.CurrentView), document, reason);
            }
            finally {
                _activateContext = null;
                if(DebugOpt.DumpDocumentRelation) DumpDocumentRelation();
            }
        }

        //SessionHostĂ΂n
        public void RegisterDocument(IPoderosaDocument document, SessionHost sessionHost) {
            _documentMap.Add(document, new DocumentHost(this, sessionHost, document));
        }

        public DocumentHost FindDocumentHost(IPoderosaDocument document) {
            return _documentMap[document];
        }
        public SessionHost FindSessionHost(ISession session) {
            return _sessionMap[session];
        }

        internal IEnumerable<DocumentHost> GetAllDocumentHosts() {
            return new ConvertingEnumerable<DocumentHost>(_documentMap.Values);
        }

        //View̃}[WłActivate
        public void ChangeLastAttachedViewForAllDocuments(IPoderosaView closing_view, IPoderosaView alternative) {
            closing_view = AdjustToOuterView(closing_view);
            alternative = AdjustToOuterView(alternative);
            foreach(DocumentHost dh in _documentMap.Values) {
                if(dh.LastAttachedView==closing_view) {
                    dh.AlternateView(alternative);
                    FireDocViewRelationChange();
                }
            }
        }
        public void ChangeLastAttachedViewForWindow(IPoderosaMainWindow window, IPoderosaView alternative) {
            alternative = AdjustToOuterView(alternative);
            foreach(DocumentHost dh in _documentMap.Values) {
                if(dh.LastAttachedView.ParentForm==window) {
                    dh.AlternateView(alternative);
                    FireDocViewRelationChange();
                }
            }
        }

        public void SetLastAttachedViewForAllDocuments(IPoderosaView view) {
            view = AdjustToOuterView(view);
            foreach(DocumentHost dh in _documentMap.Values) {
                dh.AlternateView(view);
            }
            FireDocViewRelationChange();
        }

        //vieẅʒuɂVK̃hLg悤ɂBhLg̕\ʒuςƂAƂɎs
        private DocumentHost ShowBackgroundDocument(IPoderosaView view) {
            DocumentHost new_visible_doc = FindNewVisibleDoc(view);
            if(new_visible_doc!=null) {
                new_visible_doc.AttachView(view);
            }
            return new_visible_doc;
        }

        private List<SessionHost> CreateSessionHostCollection() {
            List<SessionHost> r = new List<SessionHost>();
            foreach(object t in _sessionMap.Values)
                r.Add((SessionHost)t);
            return r;
        }

        private DocumentHost FindNewVisibleDoc(IPoderosaView view) {
            view = AdjustToOuterView(view);
            //TODO ͂B̃[A^uł̏ԁAANeBuɂȂԂLAȂǕ̎il邵AvOCŊgׂƂ
            foreach(DocumentHost dh in _documentMap.Values) {
                if(dh.LastAttachedView==view) return dh;
            }
            return null;
        }

        private IPoderosaForm ViewToForm(IPoderosaView view) {
            if(view==null)
                return null;
            else {
                return (IPoderosaForm)view.ParentForm.GetAdapter(typeof(IPoderosaForm));
            }
        }

        private void FireDocViewRelationChange() {
            foreach(IDocViewRelationEventHandler eh in _docViewRelationHandler.GetExtensions())
                eh.OnDocViewRelationChange();
        }

        //Listener
        public void AddActiveDocumentChangeListener(IActiveDocumentChangeListener listener) {
            _activeDocumentChangeListeners.Add(listener);
        }
        public void RemoveActiveDocumentChangeListener(IActiveDocumentChangeListener listener) {
            _activeDocumentChangeListeners.Remove(listener);
        }
        public void AddSessionListener(ISessionListener listener) {
            _sessionListeners.Add(listener);
        }
        public void RemoveSessionListener(ISessionListener listener) {
            _sessionListeners.Remove(listener);
        }

        private void NotifyActivation(IPoderosaForm form, IPoderosaDocument document, ActivateReason reason) {
            Debug.Assert(document!=null);
            IPoderosaMainWindow window = (IPoderosaMainWindow)form.GetAdapter(typeof(IPoderosaMainWindow));

            if(window!=null) {
                //Tabւ̒ʒmBTabClick̂ƂTabOŏĂ̂OK
                if(reason!=ActivateReason.TabClick)
                    window.DocumentTabFeature.Activate(document);
                //listenerւ̒ʒm
                foreach(IActiveDocumentChangeListener listener in _activeDocumentChangeListeners)
                    listener.OnDocumentActivated(window, document);
            }
        }

        private void NotifyAdd(IPoderosaForm form, IPoderosaDocument document) {
            IPoderosaMainWindow window = (IPoderosaMainWindow)form.GetAdapter(typeof(IPoderosaMainWindow));
            if(window!=null)
                window.DocumentTabFeature.Add(document);
        }

        private void NotifyRemove(IPoderosaForm form, IPoderosaDocument document) {
            IPoderosaMainWindow window = (IPoderosaMainWindow)form.GetAdapter(typeof(IPoderosaMainWindow));
            if(window!=null) {
                IPoderosaDocument former = window.DocumentTabFeature.ActiveDocument;
                window.DocumentTabFeature.Remove(document);
                //TODO ANeBuȂ̂Lꏊς邱ƂŃ^uւ̒ʒmɂ鐧񂩂
                if(former==document) {
                    foreach(IActiveDocumentChangeListener listener in _activeDocumentChangeListeners)
                        listener.OnDocumentDeactivated(window);
                }
            }
        }

        private IPoderosaDocument ViewToActiveDocument(IPoderosaView view) {
            IPoderosaForm form = view.ParentForm;
            IPoderosaMainWindow window = (IPoderosaMainWindow)form.GetAdapter(typeof(IPoderosaMainWindow));
            if(window!=null)
                return window.DocumentTabFeature.ActiveDocument;
            else
                return view.Document;
        }

        //r[ɃtH[JXZbgԂɂB|bvAbvEBhȄꍇA܂EBhE[hĂȂP[X̂łɒӁI
        private void SetFocusToView(IPoderosaView view) {
            IPoderosaForm form = view.ParentForm;
            IPoderosaPopupWindow popup = (IPoderosaPopupWindow)form.GetAdapter(typeof(IPoderosaPopupWindow));
            if(popup!=null) {
                if(!popup.AsForm().Visible) {
                    popup.UpdateStatus();
                    popup.AsForm().Show();
                }
            }

            if(!view.AsControl().Focused)
                view.AsControl().Focus(); //ɃEBhE͌Ă
        }

        private static IPoderosaView AdjustToOuterView(IPoderosaView view) {
            //ContentReplaceableSite΂̐egp
            IContentReplaceableViewSite s = (IContentReplaceableViewSite)view.GetAdapter(typeof(IContentReplaceableViewSite));
            if(s!=null)
                return s.CurrentContentReplaceableView;
            else
                return view;
        }

        private void DumpDocumentRelation() {
            Debug.WriteLine("[DocRelation]");
            foreach(DocumentHost dh in _documentMap.Values) {
                Debug.WriteLine(String.Format("  doc {0}, current={1}, last={2}", dh.Document.GetType().Name, ViewName(dh.CurrentView), ViewName(dh.LastAttachedView)));
            }
        }
        private static string ViewName(IPoderosaView view) {
            if(view==null)
                return "null";
            else {
                IContentReplaceableView rv = (IContentReplaceableView)view.GetAdapter(typeof(IContentReplaceableView));
                if(rv!=null)
                    return rv.GetCurrentContent().GetType().Name;
                else
                    return view.GetType().Name;
            }
        }
    }

    internal class SessionHost : ISessionHost {
        private SessionManagerPlugin _parent;
        private ISession _session;
        private List<IPoderosaDocument> _documents;

        //ȉ̃oSessionManager#CloseMultipleDocumentsɂ̂ݎgpB
        //XbhZ[tł͂ȂȂ邪ɖ͂Ȃ낤
        private int _closingDocumentCount;
        private PrepareCloseResult _prepareCloseResult;

        public SessionHost(SessionManagerPlugin parent, ISession session) {
            _parent = parent;
            _session = session;
            _documents = new List<IPoderosaDocument>();
        }

        public ISession Session {
            get {
                return _session;
            }
        }
        public int DocumentCount {
            get {
                return _documents.Count;
            }
        }
        public IEnumerable<IPoderosaDocument> Documents {
            get {
                return _documents;
            }
        }
        public IPoderosaDocument DocumentAt(int index) {
            return _documents[index];
        }
        public IEnumerable<IPoderosaDocument> ClonedDocuments {
            get {
                return new List<IPoderosaDocument>(_documents);
            }
        }

        #region ISessionHost
        public void RegisterDocument(IPoderosaDocument document) {
            _parent.RegisterDocument(document, this);
            _documents.Add(document);
        }
        //zXgĂZbVIɏIꍇ
        public void TerminateSession() {
            _parent.CleanupSession(this);
        }
        public IPoderosaForm GetParentFormFor(IPoderosaDocument document) {
            DocumentHost dh = _parent.FindDocumentHost(document);
            Debug.Assert(dh!=null, "document must be alive");
            IPoderosaView view = dh.LastAttachedView;
            if(view!=null)
                return view.ParentForm; //ꂪ݂ȂOK
            else
                return WindowManagerPlugin.Instance.ActiveWindow; //ƔC̎
        }
        #endregion

        public void CloseDocument(IPoderosaDocument document) {
            Debug.Assert(_documents.Contains(document));
            _session.InternalCloseDocument(document);
            _documents.Remove(document);
        }

        //ȉCloseMultipleDocumentŎgp
        public int CMP_ClosingDocumentCount {
            get {
                return _closingDocumentCount;
            }
            set {
                _closingDocumentCount = value;
            }
        }
        public PrepareCloseResult CMP_PrepareCloseResult {
            get {
                return _prepareCloseResult;
            }
            set {
                _prepareCloseResult = value;
            }
        }
    }

    internal class DocumentHost {
        private SessionManagerPlugin _manager;
        private SessionHost _sessionHost;
        private IPoderosaDocument _document;
        private IPoderosaView _currentView;
        private IPoderosaView _lastAttachedView;

        public DocumentHost(SessionManagerPlugin manager, SessionHost sessionHost, IPoderosaDocument document) {
            _manager = manager;
            _sessionHost = sessionHost;
            _document = document;
        }

        public IPoderosaView LastAttachedView {
            get {
                return _lastAttachedView;
            }
        }

        public IPoderosaView CurrentView {
            get {
                return _currentView;
            }
        }
        public SessionHost SessionHost {
            get {
                return _sessionHost;
            }
        }
        public IPoderosaDocument Document {
            get {
                return _document;
            }
        }

        //r[Ƃ̊֘AtύX
        public void AttachView(IPoderosaView view) {
            _lastAttachedView = view;
            _currentView = view;

            Type viewclass = _document.OwnerSession.GetCorrespondingViewType(_document);
            IContentReplaceableView rv = (IContentReplaceableView)view.GetAdapter(typeof(IContentReplaceableView));
            IPoderosaView internalview = rv==null? view : rv.AssureViewClass(viewclass); //ContentReplaceableVieŵƂ͒ggp
            Debug.Assert(viewclass==internalview.GetType());
            _sessionHost.Session.InternalAttachView(_document, internalview);
        }
        public void DetachView() {
            Debug.Assert(_currentView!=null);
            IContentReplaceableView rv = (IContentReplaceableView)_currentView.GetAdapter(typeof(IContentReplaceableView));
            IPoderosaView internalview = rv==null? _currentView : rv.GetCurrentContent(); //ContentReplaceableVieŵƂ͒ggp
            _sessionHost.Session.InternalDetachView(_document, internalview);

            if(rv!=null && rv.AsControl().Visible)
                rv.AssureEmptyViewClass();

            _currentView = null;
        }

        //ViewȂǂőւ̃r[ɒu
        public void AlternateView(IPoderosaView view) {
            if(_currentView!=null) DetachView();
            _lastAttachedView = view;
        }



    }

    internal class ClosingContext {
        private enum CloseType {
            OneDocument,
            OneSession,
            OneWindow,
            AllWindows
        }

        private CloseType _type;
        private IPoderosaMainWindow _window; //_type==OneWindoŵƂ̂݃ZbgÂƂnull
        private List<SessionHost> _closingSessions;

        public ClosingContext(IPoderosaMainWindow window) {
            _type = CloseType.OneWindow;
            _window = window;
            _closingSessions = new List<SessionHost>();
        }

        public void AddClosingSession(SessionHost sh) {
            Debug.Assert(_type==CloseType.OneWindow);
            _closingSessions.Add(sh);
        }
        public IEnumerable<SessionHost> ClosingSessions {
            get {
                return _closingSessions;
            }
        }
    }

    internal class ActivateContext {
        private IPoderosaDocument _document;
        private ActivateReason _reason;

        public ActivateContext(IPoderosaDocument document, ActivateReason reason) {
            _document = document;
            _reason = reason;
        }
    }
}
