/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Script/ExpressionFormatter.cs#6 $
 * $DateTime: 2007/10/25 12:56:12 $
 * Expression̕
 */
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Text;
#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Script {

    //TESTINFO ExpressionParserTests.cs
    public class ExpressionFormatter : ExpressionVisitor<object> {
    	
	    private StringBuilder _buf;
	    private bool _excessiveBracket;
	    private bool _spaceAtBinaryOp;

        public ExpressionFormatter() {
        }
        public bool ExcessiveBracket {
            get {
                return _excessiveBracket;
            }
            set {
                _excessiveBracket= value;
            }
        }
        public bool SpaceAtBinaryOp {
            get {
                return _spaceAtBinaryOp;
            }
            set {
                _spaceAtBinaryOp = value;
            }
        }
    	
    	
	    //C
	    public String Format(Expression expr) {
		    _buf = new StringBuilder();
		    expr.Apply(this);
		    string result = _buf.ToString();
            _buf = null;
            return result;
	    }

	    public override object BinaryOp(BinaryOpExpression expr) {
		    if(_excessiveBracket) _buf.Append('(');
		    expr.Left.Apply(this);
		    if(_spaceAtBinaryOp) _buf.Append(' ');
		    _buf.Append(expr.OperatorAsString);
		    if(_spaceAtBinaryOp) _buf.Append(' ');
		    expr.Right.Apply(this);
		    if(_excessiveBracket) _buf.Append(')');
		    return null;
	    }
        public override object Condition(ConditionalExpression expr) {
		    if(_excessiveBracket) _buf.Append('(');
		    expr.Condition.Apply(this);
		    _buf.Append("? ");
		    expr.TrueCase.Apply(this);
		    _buf.Append(" : ");
		    expr.FalseCase.Apply(this);
		    if(_excessiveBracket) _buf.Append(')');
		    return null;
	    }

        public override object UnaryOp(UnaryOpExpression expr) {
		    if(_excessiveBracket) _buf.Append('(');
		    _buf.Append(expr.OperatorAsString);
		    expr.Content.Apply(this);
		    if(_excessiveBracket) _buf.Append(')');
		    return null;
	    }

        public override object LiteralNumber(LiteralNumberExpression expr) {
    	    _buf.Append(expr.Value.ToString());
    	    return null;
        }

        public override object LiteralBoolean(LiteralBooleanExpression expr) {
    	    _buf.Append(expr.Value? "true" : "false");
    	    return null;
        }
        public override object LiteralTime(LiteralTimeExpression expr) {
            _buf.Append("@t");
            _buf.Append(expr.Value);
            return null;
        }
        public override object LiteralDate(LiteralDateExpression expr) {
            _buf.Append("@d");
            _buf.Append(expr.Value);
            return null;
        }
        public override object LiteralString(LiteralStringExpression expr) {
            _buf.Append('"');
            _buf.Append(expr.Value);
            _buf.Append('"');
            return null;
        }
        public override object Symbol(SymbolExpression expr) {
            _buf.Append(expr.Symbol);
            return null;
        }
        public override object Parameter(ParameterExpression expr) {
            _buf.Append('$');
            _buf.Append(expr.Symbol);
            return null;
        }

        public override object Array(ArrayExpression expr) {
            if(_excessiveBracket) _buf.Append('(');
            expr.Target.Apply(this);
            _buf.Append('[');
            expr.Index.Apply(this);
            _buf.Append(']');
            if(_excessiveBracket) _buf.Append(')');
            return null;
        }
        public override object Invoke(InvokeExpression expr) {
            if(_excessiveBracket) _buf.Append('(');
            expr.Target.Apply(this);
            _buf.Append('(');
            bool first = true;
            foreach(Expression arg in expr) {
                if(!first) _buf.Append(", ");
                arg.Apply(this);
                first = false;
            }
            _buf.Append(')');
            if(_excessiveBracket) _buf.Append(')');
            return null;
        }
        public override object Reference(ReferenceExpression expr) {
            if(_excessiveBracket) _buf.Append('(');
            expr.Left.Apply(this);
            _buf.Append('.');
            expr.Right.Apply(this);
            if(_excessiveBracket) _buf.Append(')');
            return null;
        }
        public override object Lambda(LambdaExpression expr) {
            if(_excessiveBracket) _buf.Append('(');
            _buf.Append("lambda ((");
            ArgList(expr.Args);
            _buf.Append(") ");
            expr.Body.Apply(this);
            _buf.Append(")");
            if(_excessiveBracket) _buf.Append(')');
            return null;
        }
        public override object Let(LetExpression expr) {
            if(_excessiveBracket) _buf.Append('(');
            _buf.Append("let ((");
            for(int i=0; i<expr.LocalVarCount; i++) {
                if(i>0) _buf.Append(", ");
                _buf.Append(expr.LocalNameAt(i));
                _buf.Append("=");
                expr.LocalExprAt(i).Apply(this);
            }
            _buf.Append(") ");
            expr.Body.Apply(this);
            _buf.Append(")");
            if(_excessiveBracket) _buf.Append(')');
            return null;
            
        }
        private void ArgList(ParameterListDefinition al) {
            for(int i=0; i<al.Count; i++) {
                if(i>0) _buf.Append(", ");
                _buf.Append(al.TypeDescAt(i).Format());
                _buf.Append(" ");
                _buf.Append(al.NameAt(i));
            }
        }
    }

    /*
      iPhoneŎg߂ExpressionFormatter
    @iPhoneł̓[UExpressionƂ͂ȂAANTLRV@ɑΉȂObjective-C͎gȂA
    @p[XN̕SɂȂ肩˂ȂAƂƂAŏg[NNSString*[]œnA|[hL@ɔzu邱Ƃœ͑ΉB
    @ꂾƐlԂExpression̂͂ǂ̂ŁAɃtH[}b^ĕϊ@\B

    @\vfό̂̂͂gAƂ
      #inv ( <expr> <expr> )
      #lambda ( <type> <name> | <expr>)
      #let ( <name> <expr> | <expr>)
    @̂悤ɂ
    @
    @́A#Ŏn܂g[N͓\̂́A$Ŏn܂̂̓p[^A''ň͂܂ꂽ͕̂Apł͂܂̂symbolALiteralNumberAƂ悤ɗeՂɋʂł
    @P -  #-@Ƃ
    */

    public class OCExpressionFormatter : ExpressionVisitor<object> {

        private List<string> _tokens;

        public OCExpressionFormatter() {
            _tokens = new List<string>();
        }

        //C
        public List<string> Format(Expression expr) {
            _tokens.Clear();
            expr.Apply(this);
            return _tokens;
        }

        public override object BinaryOp(BinaryOpExpression expr) {
            _tokens.Add(expr.OperatorAsString); //ƂExpressionɂƓBUnaryOpł͈Ⴄ̂Œ
            expr.Left.Apply(this);
            expr.Right.Apply(this);
            return null;
        }
        public override object Condition(ConditionalExpression expr) {
            _tokens.Add("#cond");
            expr.Condition.Apply(this);
            expr.TrueCase.Apply(this);
            expr.FalseCase.Apply(this);
            return null;
        }

        public override object UnaryOp(UnaryOpExpression expr) {
            if(expr.Operator==UnaryOperator.Inverse)
                _tokens.Add("#-");
            else
                _tokens.Add("!");
            expr.Content.Apply(this);
            return null;
        }

        public override object LiteralNumber(LiteralNumberExpression expr) {
            _tokens.Add(expr.Value.ToString());
            return null;
        }

        public override object LiteralBoolean(LiteralBooleanExpression expr) {
            _tokens.Add(expr.Value? "true" : "false");
            return null;
        }
        public override object LiteralTime(LiteralTimeExpression expr) {
            _tokens.Add("@t" + expr.Value);
            return null;
        }
        public override object LiteralDate(LiteralDateExpression expr) {
            _tokens.Add("@d" + expr.Value);
            return null;
        }
        public override object LiteralString(LiteralStringExpression expr) {
            StringBuilder bld = new StringBuilder();
            bld.Append('\'').Append(expr.Value).Append('\'');
            _tokens.Add(bld.ToString());
            return null;
        }
        public override object Symbol(SymbolExpression expr) {
            _tokens.Add(expr.Symbol);
            return null;
        }
        public override object Parameter(ParameterExpression expr) {
            _tokens.Add("$" + expr.Symbol);
            return null;
        }

        public override object Array(ArrayExpression expr) {
            _tokens.Add("#arr");
            expr.Target.Apply(this);
            expr.Index.Apply(this);
            return null;
        }
        public override object Invoke(InvokeExpression expr) {
            _tokens.Add("#inv(");
            expr.Target.Apply(this);
            foreach(Expression arg in expr) {
                arg.Apply(this);
            }
            _tokens.Add(")");
            return null;
        }
        public override object Reference(ReferenceExpression expr) {
            _tokens.Add("#ref");
            expr.Left.Apply(this);
            expr.Right.Apply(this);
            return null;
        }
        public override object Lambda(LambdaExpression expr) {
            _tokens.Add("#lambda(");
            ArgList(expr.Args);
            _tokens.Add("|");
            expr.Body.Apply(this);
            _tokens.Add(")");
            return null;
        }
        public override object Let(LetExpression expr) {
            _tokens.Add("#let(");
            for(int i=0; i<expr.LocalVarCount; i++) {
                _tokens.Add(expr.LocalNameAt(i));
                expr.LocalExprAt(i).Apply(this);
            }
            _tokens.Add("|");
            expr.Body.Apply(this);
            _tokens.Add(")");
            return null;

        }
        private void ArgList(ParameterListDefinition al) {
            for(int i=0; i<al.Count; i++) {
                _tokens.Add(al.TypeDescAt(i).Format());
                _tokens.Add(al.NameAt(i));
            }
        }

        //ʏexpressionăfobOR\[ɓ\t
        public static void DumpExpression(string expression_text) {
            Expression expr = Expression.ParseExpression(expression_text);
            OCExpressionFormatter fmt = new OCExpressionFormatter();
            List<string> result = fmt.Format(expr);

            //dump
            Debug.WriteLine(String.Format("//OCFormat: {0}", expression_text));
            int line_len = 0;
            for(int i=0; i<result.Count; i++) {
                string t = String.Format("{0}@\"{1}\"", line_len==0? "" : ", ", result[i]);
                Debug.Write(t);
                line_len += t.Length;
                if(line_len > 60 || i==result.Count-1) {
                    Debug.WriteLine("");
                    line_len = 0;
                }
            }
        }
    }

#if UNITTEST
    //s͂Ƃ肠UnitTest

    [TestFixture]
    public class OCFormatterTests {
        [Test]
        public void Basic() {
            OCExpressionFormatter.DumpExpression("let( (b = ma(len)) b - 2 * bollinger_unit(this, b, len) )");
        }
    }
#endif
}
