/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/PseudoStock.cs#16 $
 * $DateTime: 2008/04/17 18:20:55 $
 * 
 * [Jf[^̎擾Aɂf[^Ȃǂ̋[IȖ̋@\BeXgfɊ
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.IO;

using Poderosa;
using BTime=Bellagio.Values.BTime;
using TimeAndSales=Bellagio.Values.TimeAndSales;
using TickData=Bellagio.Values.TickData;

using Bellagio.Ordering;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {

    //ʂēNXłQ̃C^tF[XĂ
    public class DemonstrationDataSource : IItaDataSource, ITickDataSource, ILightIntraDayTradeSource, IOrderDataSource {
        private ITickDataHost _tickHost;
        private IItaDataHost _itaHost;
        private IOrderDataHost _orderHost;
        private ILightIntraDayTradeHost _lightDTHost;
        private List<Stock> _tickOpened;
        private List<Stock> _itaOpened;

        public DemonstrationDataSource() {
            _tickOpened = new List<Stock>();
            _itaOpened = new List<Stock>();
        }

        AsyncOpResult ITickDataSource.OpenTickData(Stock stock, ref string message) {
            Debug.Assert(_tickHost!=null);
            Debug.Assert(stock!=null);
            _tickOpened.Add(stock);
            RandomStock rs = stock.Profile as RandomStock;
            if(rs!=null) 
                _tickHost.OnInitialPriceInfo(stock, 0, 20071001, rs.PrevClose, 0, 0,0,  0, 0, 0, 0);
            
            return AsyncOpResult.Succeeded;
        }

        void ITickDataSource.CloseTickData(Stock stock) {
            _tickOpened.Remove(stock);
        }

        AsyncOpResult IItaDataSource.OpenItaData(Stock stock, ref string message) {
            Debug.Assert(_itaHost!=null);
            _itaOpened.Add(stock);
            RandomStock rs = stock.Profile as RandomStock;
            if(rs!=null)
                _itaHost.SetItaAll(stock, rs.Feeder.Ask, rs.Feeder.Bid);
            return AsyncOpResult.Succeeded;
        }

        void IItaDataSource.CloseItaData(Stock stock) {
            _itaOpened.Remove(stock);
        }

        void IPrimaryStreamDataBase.Prepare(IPrimaryStreamDataHostBase site) {
            if(_tickHost==null) _tickHost = site as ITickDataHost;
            if(_itaHost==null) _itaHost = site as IItaDataHost;
            if(_orderHost==null) _orderHost = site as IOrderDataHost;
            if(_lightDTHost==null) _lightDTHost = site as ILightIntraDayTradeHost;
        }

        void IPrimaryStreamDataBase.Terminate() {
        }


        IAdaptable IAdaptable.GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        //f[^XVʒm@fȂ̂Ń^C}[쓮ɂȂ
        public void UpdateTick(Stock stock, TickData tick) {
            if(_tickHost!=null && _tickOpened.Contains(stock))
                _tickHost.OnTickData(stock, tick.Time, tick.Price, tick.Volume, 0, 0); //VWAPڂ
        }
        public void UpdateIta(Stock stock, Ita ask, Ita bid) {
            if(_itaHost!=null && _itaOpened.Contains(stock))
                _itaHost.SetItaAll(stock, ask, bid);
        }

        private static DemonstrationDataSource _instance;
        public static DemonstrationDataSource Instance {
            get {
                if(_instance==null) _instance = new DemonstrationDataSource();
                return _instance;
            }
        }


        AsyncOpResult ILightIntraDayTradeSource.OpenLightIntraDayTrade(Stock stock, ref string message) {
            PseudoIndex pi = (PseudoIndex)stock.Profile;
            _lightDTHost.SetInitialData(stock, pi.PrevClose);
            return AsyncOpResult.Succeeded;
        }

        void ILightIntraDayTradeSource.CloseLightIntraDayTrade(Stock stock) {
        }



        AsyncOpResult IOrderDataSource.GetStockPositionList(ref string message) {
            _orderHost.SetStockPositionList(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, CreateDemonstrationTradePositionList(), null);
            return AsyncOpResult.Succeeded;
        }

        AsyncOpResult IOrderDataSource.GetStockOrderList(ref string message) {
            _orderHost.SetStockOrderList(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, CreateDemonstrationTradeOrderList(), null);
            return AsyncOpResult.Succeeded;
        }
        AsyncOpResult IOrderDataSource.GetDerivativePositionList(ref string message) {
            _orderHost.SetDerivativePositionList(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, CreateDemonstrationDerivativePositionList(), null);
            return AsyncOpResult.Succeeded;
        }

        AsyncOpResult IOrderDataSource.GetDerivativeOrderList(ref string message) {
            _orderHost.SetDerivativeOrderList(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, CreateDemonstrationDerivativeOrderList(), null);
            return AsyncOpResult.Succeeded;
        }

        AsyncOpResult IOrderDataSource.GetOrderDetailInfo(string orderID, OrderStatusCodeCategory category, out TradeOrderItem result) {
            result = null;
            return AsyncOpResult.Failed;
        }

        private TradePositionList CreateDemonstrationTradePositionList() {
            TradePositionList c = new TradePositionList();

            TradePositionItem item = new TradePositionItem(GetStock(8002));
            item.DealType = OrderDealType.Genbutsu;
            item.LongShort = LongShort.Long;
            item.CurrentPrice100 = 97000;
            item.AddPositionElement(CreateTradePositionElement(20070820, 950, 10000, 0, 0, 0));
            c.AddItem(item);

            item = new TradePositionItem(GetStock(9984));
            item.DealType = OrderDealType.CreditSeido;
            item.LongShort = LongShort.Short;
            item.CurrentPrice100 = 220000;
            item.AddPositionElement(CreateTradePositionElement(20070820, 2220, 4000, 0, 1000, 0));
            c.AddItem(item);

            item = new TradePositionItem(GetStock(8035));
            item.DealType = OrderDealType.CreditSeido;
            item.LongShort = LongShort.Long;
            item.CurrentPrice100 = 820000;
            item.AddPositionElement(CreateTradePositionElement(20070820, 8300, 1000, 0, 1234, 567));
            c.AddItem(item);

            item = new TradePositionItem(GetStock(5401));
            item.DealType = OrderDealType.CreditIppan;
            item.LongShort = LongShort.Long;
            item.CurrentPrice100 = 83000;
            item.AddPositionElement(CreateTradePositionElement(20070820, 850, 10000, 10000, 1234, 567));
            c.AddItem(item);

            return c;
        }
        private TradePositionList CreateDemonstrationDerivativePositionList() {
            TradePositionList c = new TradePositionList();

            StockProfileCollection gs = BellagioRoot.GlobalStockCollection;
            TradePositionItem item = new TradePositionItem(gs.FindExact("NF").Primary);
            item.DealType = OrderDealType.Derivatives;
            item.LongShort= LongShort.Long;
            item.CurrentPrice100 = 1700000;
            item.AddPositionElement(CreateTradePositionElement(20070820, 16900, 1, 0, 0, 0));
            c.AddItem(item);

            return c;
        }
        private TradePositionItem.Element CreateTradePositionElement(int date, int price, int volume, int ordering, int keihi, int risoku) {
            TradePositionItem.Element e = new TradePositionItem.Element();
            e.DateAsInt = date;
            e.Price100 = price;
            e.Volume = volume;
            e.OrderingVolume = ordering;
            e.Keihi = keihi;
            e.Risoku = risoku;
            return e;
        }


        private TradeOrderList CreateDemonstrationTradeOrderList() {
            TradeOrderList c = new TradeOrderList();
            TradeOrderItem item = new TradeOrderItem(GetStock(5405));
            item.Condition = OrderCondition.Sashine;
            item.DealType = OrderDealType.Genbutsu;
            item.Sashine = 603;
            item.Volume = 25000;
            item.OrderStatus = OrderStatus.Preparing;
            item.AddTradeItem(603, 25000, 9*3600+480);
            c.AddItem(item);

            item = new TradeOrderItem(GetStock(8802));
            item.DealType = OrderDealType.CreditSeido;
            item.NewOrClose = OrderNewClose.Close;
            item.Condition = OrderCondition.Nariyuki;
            item.OrderStatus = OrderStatus.Complete;
            item.Volume = 2000;
            item.AddTradeItem(3100, 2000, 9*3600);
            c.AddItem(item);

            item = new TradeOrderItem(GetStock(6758));
            item.DealType = OrderDealType.CreditSeido;
            item.LongShort = LongShort.Short;
            item.NewOrClose = OrderNewClose.New;
            item.Condition = OrderCondition.Hikenari;
            item.OrderStatus = OrderStatus.Cancelled;
            item.Volume = 2000;
            c.AddItem(item);

            item = new TradeOrderItem(GetStock(8411));
            item.DealType = OrderDealType.Genbutsu;
            item.NewOrClose = OrderNewClose.New;
            item.Condition = OrderCondition.Sashine;
            item.Sashine = 730000;
            item.OrderStatus = OrderStatus.Changed;
            item.Volume = 20;
            item.AddTradeItem(729000, 15, 9*3600+120);
            c.AddItem(item);

            item = new TradeOrderItem(GetStock(3727));
            item.DealType = OrderDealType.Genbutsu;
            item.NewOrClose = OrderNewClose.Close;
            item.Condition = OrderCondition.Sashine;
            item.OrderStatus = OrderStatus.Expired;
            item.Sashine = 140000;
            item.Volume = 10;
            item.PTSOrder = PTSOrder.CarryOverIfExpire;
            c.AddItem(item);

            item = new TradeOrderItem(GetStock(7974));
            item.DealType = OrderDealType.Genbutsu;
            item.NewOrClose = OrderNewClose.Close;
            item.Condition = OrderCondition.Sashine;
            item.OrderStatus = OrderStatus.Preparing;
            item.Sashine = 15200;
            item.Volume = 300;
            item.PTSOrder = PTSOrder.CarriedOverByExpire;
            c.AddItem(item);

            return c;
        }
        private TradeOrderList CreateDemonstrationDerivativeOrderList() {
            TradeOrderList c = new TradeOrderList();
            StockProfileCollection gs = BellagioRoot.GlobalStockCollection;
            TradeOrderItem item = new TradeOrderItem(gs.FindExact("NF").Primary);
            item.Condition = OrderCondition.Sashine;
            item.DealType = OrderDealType.Derivatives;
            item.LongShort = LongShort.Long;
            item.Sashine = 16900;
            item.Volume = 3;
            item.OrderStatus = OrderStatus.Ordered;
            item.TransactionStatus = OrderTransactionStatus.Waiting;
            c.AddItem(item);

            return c;
        }

        private IOrderItemFormatter _orderItemFormatter;
        public IOrderItemFormatter OrderItemFormatter {
            get {
                if(_orderItemFormatter==null) _orderItemFormatter = new DefaultOrderItemFormatter();
                return _orderItemFormatter;
            }
        }

        public void SendNewOrCloseOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage) {
            result = null;
            successfulMessage = null;
        }
        public void SendSpecifiedElementCloseOrder(TradeOrderItem item, PositionAndVolumeList list, string pincode, out TradeOrderItem result, out string successfulMessage) {
            result = null;
            successfulMessage = null;
        }
        public void SendChangeOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage) {
            result = null;
            successfulMessage = null;
        }
        public void SendCancelOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage) {
            result = null;
            successfulMessage = null;
        }
        public AsyncOpResult GetStockPowerInfo(ref string message) {
            _orderHost.SetStockPowerInfo(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, new StockBuyingPowerInfo(), null);
            return AsyncOpResult.Succeeded;
        }
        public AsyncOpResult GetDerivativePowerInfo(ref string message) {
            _orderHost.SetDerivativePowerInfo(DataResponseCode.SyncResponse|DataResponseCode.Succeeded, new DerivativeBuyingPowerInfo(), null);
            return AsyncOpResult.Succeeded;
        }
        public AsyncOpResult GetStockSnapshot(AbstractStockProfile prof, ref string message) {
            return AsyncOpResult.Failed;
        }
        public AsyncOpResult GetDerivativeSnapshot(DerivativeStockProfile prof, ref string message) {
            return AsyncOpResult.Failed;
        }


        private Stock GetStock(int code) {
            return BellagioRoot.GlobalStockCollection.FindExact(code.ToString()).Primary;
        }
    }

    public abstract class PseudoStock : AbstractStockProfile {
        protected int _prevClose;
        protected TimeManager.Tick _callback;

        public PseudoStock(string name, string code, StockExchange m, int prevClose, StockProfileFlags f)
            : base(name, code, f, 1000) {
            CreatePrimary( m, StockExchangeSubType.Ichibu, StockFlags.None);
            _prevClose = prevClose;
        }
        public override string VolumeUnitString {
            get { return ""; }
        }
        public override int ImplicitPriceFactor {
            get { return 1; }
        }
        public override string CodeForDataSource() {
            return _code;
        }

        public int PrevClose {
            get {
                return _prevClose;
            }
        }
        public abstract void Run();

        public void Stop() {
            if(_callback==null) return;
            BellagioRoot.TimeManager.RemoveTimeManagerListener(_callback);
            _callback = null;
        }

        protected abstract void OnTick(BTime time);
    }

    //œ삷鉼zIȖ쐬BɃfp
    public class RandomStock : PseudoStock {
        private RandomItaFeeder _feeder;
        private int _initialPrice;
        private int _priceStep;

        public RandomStock(string name, string code, StockExchange m, int price, int prevClose, int priceStep)
            : base(name, code, m, prevClose, StockProfileFlags.None) {
            _initialPrice = price;
            _priceStep = priceStep;

            _feeder = new RandomItaFeeder(5);
            _feeder.BidPower = 70;
            _feeder.PriceStep = _priceStep;
            _feeder.VolumeFactor = 100;
            _feeder.VolumeUnit = this.Primary.VolumeUnit;
            _feeder.InitIta(_initialPrice);
        }
        public int InitialPrice {
            get {
                return _initialPrice;
            }
        }
        public RandomItaFeeder Feeder {
            get {
                return _feeder;
            }
        }
        public override void Run() {
            if(_callback!=null) return;
            _callback = new TimeManager.Tick(OnTick);
            BellagioRoot.TimeManager.AddTimeManagerListener(_callback);
            string message = null;
            IntraDayTrade dt = null;
            RealtimeIta ita = null;
            BellagioRoot.IntraDayTradeProvider.Open(_primary, ref dt, ref message);
            BellagioRoot.ItaProvider.Open(_primary, ref ita, ref message); //ĂȂƁAUpdatełȂ
        }

        protected override void OnTick(BTime time) {
            Update(time);
        }


        private void Update(BTime time) {
            DemonstrationDataSource ds = DemonstrationDataSource.Instance;
            if(MarketUtil.IsMarketOpenTime(StockExchange.T, time)) {
                RealtimeIta ita = BellagioRoot.ItaProvider.Lookup(_primary);
                if(ita!=null) {
                    TickData td = _feeder.Update(ita.AskWork, ita.BidWork, time.AsInt(), 1, 3);
                    if(td!=null) ds.UpdateTick(_primary, td);
                    ds.UpdateIta(_primary, _feeder.Ask, _feeder.Bid);
                }
            }
        }
    }

    //fp̎w
    public class PseudoIndex : PseudoStock {
        private static Random _random = new Random();
        private int _value;

        public PseudoIndex(string name, string code, int prevClose) : base(name, code, StockExchange.B, prevClose, StockProfileFlags.F2) {
            _value = prevClose;
        }
        public override void Run() {
            if(_callback!=null) return;
            _callback = new TimeManager.Tick(OnTick);
            BellagioRoot.TimeManager.AddTimeManagerListener(_callback);
            string message = null;
            LightIntraDayTrade dt = null;
            BellagioRoot.LightIntraDayTradeProvider.Open(_primary, ref dt, ref message);
        }

        protected override void OnTick(BTime time) {
            _value += _random.Next(1000) - 500;
            BellagioRoot.LightIntraDayTradeProvider.Lookup(_primary).CurrentPriceR.SetValue(_value);
        }

    }

    //ʂɗ^Time&SalesĐ^Cv
    public class FixedDataStock : PseudoStock {
        private TimeAndSales _localData;
        private int _lastIndex; //ꂩo^ׂTickDatãCfNX

        public FixedDataStock(string name, string code, StockExchange m, int prevClose, TimeAndSales ts) : base(name,code, m,prevClose, StockProfileFlags.None) {
            _localData = ts;
            _lastIndex = 0;
        }
        public void Reset(TimeAndSales ts) {
            _localData = ts;
            _lastIndex = 0;
        }

        internal TimeAndSales TimeAndSales {
            get {
                return _localData;
            }
        }

        public override void Run() {
            if(_callback!=null) return;
            _callback = new TimeManager.Tick(OnTick);
            BellagioRoot.TimeManager.AddTimeManagerListener(_callback);
            string message = null;
            IntraDayTrade dt = null;
            BellagioRoot.IntraDayTradeProvider.Open(_primary, ref dt, ref message);
        }
        protected override void OnTick(BTime time) {
            IntraDayTrade q = BellagioRoot.IntraDayTradeProvider.Lookup(_primary);
            while(_localData.Count>_lastIndex && _localData.TickAt(_lastIndex).Time <= time.AsInt()) {
                TickData td = _localData.TickAt(_lastIndex++);
                q.UpdateBySingleTick(td);
            }
            //q.UpdateByTime(time);

            if(_localData.Count==_lastIndex) Stop(); //I
        }
    }

    public class DemonstrationStocks {
        private static StockProfileCollection _allDemos;

        //_쐬Af[^XVJn
        public static void CreateAndStartRandomFeed() {
            _allDemos = new StockProfileCollection();
            Debug.Assert(BellagioRoot.EnvironmentParam.RUN_LOCAL_TIMER);

           // Add("Uo", 901, 17000, 17000);
            Add("", 6502, 1050, 1060);
            Add("ۍg", 8002, 1010, 1000);
            Add("Z", 5405, 720, 725);
            Add("\tgoN", 9984, 2700, 2700);
            Add("VS", 5401, 865, 860);
            Add("z_", 7267, 4090, 4130);
            Add("쑺HD", 8604, 2435, 2380);
            Add("݂HD", 8411,  752000, 754000);
            Add("JR{", 9020,  974000, 970000);
            Add("OHn", 8802, 3700, 3700);
            Add("g^", 7203, 7350, 7300);
            Add("", 6501, 926, 920);
            Add("O䕨Y", 8031, 2130, 2130);
            Add("\j[", 6758, 6540, 6550);
            Add("AhoeXg", 6857, 5630, 5660);
            Add("AvbNX", 3727, 180000, 180000);

            AddIndex("o", "NK225", 1650000);
            AddIndex("TOPIX", "TOPIX", 165000);
        }
        private static void Add(string name, int code, int price, int prevclose) {
            RandomStock s = new RandomStock(name, code.ToString(), StockExchange.T, price, prevclose , MarketUtil.YobineIncrement(prevclose)-prevclose);
            AbstractStockProfile orig = BellagioRoot.GlobalStockCollection.FindExact(s.Code);
            if(orig!=null)
                BellagioRoot.GlobalStockCollection.Remove(orig); //Ώ 
            BellagioRoot.GlobalStockCollection.Add(s);
            s.Run();

            _allDemos.Add(s);
        }
        private static void AddIndex(string name, string code, int prevclose) {
            PseudoIndex pi = new PseudoIndex(name, code, prevclose);
            BellagioRoot.GlobalStockCollection.Add(pi);
            pi.Run();
        }


        public static StockProfileCollection AllDemos {
            get {
                return _allDemos;
            }
        }

        
    }

}
