/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/PrimaryData.cs#12 $
 * $DateTime: 2008/03/13 13:20:43 $
 * ꎟf[^B擾̂Hĉʂ֓BAƂƂɂȂ
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Poderosa;
using Poderosa.Util.Collections;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {

    public interface IPrimaryStreamDataBase : IAdaptable {
        //Ŝ̏EI
        void Prepare(IPrimaryStreamDataHostBase site);
        void Terminate();
    }
    public interface IPrimaryStreamDataHostBase : IAdaptable {
    }

    //f[^NGXg̉R[h@IǂAsłSBɃf[^\[XŌoωʒmA
    [Flags]
    public enum DataResponseCode {
        Succeeded = 1,
        Failed = 2,
        ResultMask = 3,

        //R͂PZbgĂ邱
        SyncResponse = 4,
        AsyncResponse = 8,
        DataSourceRequiresRefresh = 12,
        SyncTypeMask = 12
    }

    public enum PrimaryDataStatus {
        Initial,
        Preparing, //Ĵ҂Ă
        OK, //ғ
        Error //G[
    }


    //StockPrimaryDataIuWFNg(Subscribern)Aǂ̃f[^ω̂m邽߂ɔr邽߂̃L[
    //StockPrimaryDatȁԂɓĂ܂Bωm肽Ƃ邽߁B
    public class DataChangeTag {
        private PrimaryDataStatus _status;
        private int _updateCount; //XVꂽ Ԃł̂ݗL
        private string _errorMessage; //ԂG[̂Ƃ̂ݗL

        public DataChangeTag Copy() {
            DataChangeTag t = new DataChangeTag();
            Let(t);
            return t;
        }

        public PrimaryDataStatus Status {
            get {
                return _status;
            }
        }
        public string ErrorMessage {
            get {
                return _errorMessage;
            }
        }
        public bool IsOK {
            get {
                return _status==PrimaryDataStatus.OK;
            }
        }
        public bool IsError {
            get {
                return _status==PrimaryDataStatus.Error;
            }
        }
        public bool IsInitialOrPreparing {
            get {
                return _status==PrimaryDataStatus.Initial || _status==PrimaryDataStatus.Preparing;
            }
        }

        public void Increment() {
            Debug.Assert(_status==PrimaryDataStatus.OK);
            _updateCount++;
        }

        public void Let(DataChangeTag t) {
            _status = t._status;
            _updateCount = t._updateCount;
            _errorMessage = t._errorMessage;
        }

        public void SetStatus(PrimaryDataStatus st) {
            Debug.Assert(_status!=st); //Ԃւ̕ω͋Ȃ̂Ƃ
            Debug.Assert(st!=PrimaryDataStatus.Error);
            Debug.Assert(st!=PrimaryDataStatus.Initial);
            _status = st;
            _updateCount = 0;
        }
        public void SetErrorStatus(string message) {
            Debug.Assert(_status!=PrimaryDataStatus.Error);
            _status = PrimaryDataStatus.Error;
            _updateCount = 0;
            _errorMessage = message;
        }

        public bool Equals(DataChangeTag s) {
            return s._status==_status && s._updateCount==_updateCount;
        }
    }

    //A^Cf[^MčXVIuWFNg̊
    //Stockɑ΂ĂPCX^X쐬B
    //DaytimeTrade(Tickƕ)AAAINDEX(LightDaytimeTrade)̂R킪
    public abstract class StockPrimaryData : IAdaptable {
        private DataChangeTag _changeTag;
        protected Stock _stock;

        public StockPrimaryData(Stock stock) {
            _stock = stock;
            _changeTag = new DataChangeTag();
        }

        public Stock Stock {
            get {
                return _stock;
            }
        }

        public DataChangeTag DataChangeTag {
            get {
                return _changeTag;
            }
        }

        public PrimaryDataStatus DataStatus {
            get {
                return _changeTag.Status;
            }
        }

        //ԂŃf[^XVNƂ̎s
        public void FireDataUpdateEvent() {
            bool cancel = FireEventToSubscriberManager();
            if(!cancel) _changeTag.Increment();
        }

        //DataSubscriberManager֒ʒm ̐Es̔܂
        public abstract bool FireEventToSubscriberManager();
        public abstract void FireErrorEventToSubscriberManager(string msg);

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

    //A^Cf[^\[XŌĂԁAf[^C^tF[X
    //͔̂h̃C^tF[Xōs
    public interface IStockPrimaryDataInitializer : IAdaptable {
        void FailLoading(string msg);
        bool IsCompleted { get; }
    }

    //PrimaryDatȁsIuWFNg
    //StockPrimaryDataƂP΂PΉADataSource炤ߒł̂ݎgp
    public abstract class StockPrimaryDataInitializer : IStockPrimaryDataInitializer {
        private StockPrimaryData _target;

        public StockPrimaryDataInitializer(StockPrimaryData target) {
            _target = target;
        }
        public Stock Stock {
            get {
                return _target.Stock;
            }
        }

        //񓯊Ƀf[^擾鑀삪Ƃʒm
        /*
        protected void CompleteAsyncLoading() {
            ValueHost e = GetEntryOrNull(stock);
            Debug.Assert(e!=null && e.status==DataHostStatus.Preparing);
            if(_notifier!=null) _notifier(stock, AsyncOpResult.Succeeded, null);
            e.status = DataHostStatus.OK;
        }
        */
            /*
        public void FailLoading(string msg) {
            _target.DataChangeTag.SetErrorStatus(msg);
            if(!BUtil.IsExecutingInMainThread)
                _target.FireEventToSubscriberManager();

            ValueHost e = GetEntryOrNull(stock);
            Debug.Assert(e!=null && e.status==DataHostStatus.Preparing);
            if(_notifier!=null) _notifier(stock, AsyncOpResult.Failed, msg);
            _data.Remove(stock);
        }
            */
        public abstract bool IsCompleted { get; }
        public abstract void CompleteInitialization();
        public virtual void FailLoading(string msg) {
            _target.DataChangeTag.SetErrorStatus(msg);
            _target.FireEventToSubscriberManager();
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

    //ȉAeDataProvideȓł̏ɕzĂ낤
#if false 

    public class RealTimeStockEntry {
        private RealTimeDataSource _source;
        private ListedStock _stock;
#if UNITTEST
        internal int _count; //ڑJEg
#else
        private int _count;
#endif
        private object _tag; //IPrimaryDataSourceΉ

        //Kvȃf[^intł̎(BTimeł͂Ȃ~bP)
        private int _price;
        private int _priceTime;
        private int _prevVolume; //PO̒lƂT&S\ɂȂ
        private int _volume;
        private int _volumeTime;
        private BTime _tradeTime;
        private int _tradeTimeTime;
        private Ita _askIta;
        private Ita _bidIta;
        private int _itaTime;

        public RealTimeStockEntry(RealTimeDataSource source, ListedStock s, int ita_size) {
            _source = source;
            _stock = s;
            _count = 1;
            _askIta = new Ita(ita_size, AskBid.Ask);
            _bidIta = new Ita(ita_size, AskBid.Bid);
        }
        public object Tag {
            get {
                return _tag;
            }
            set {
                _tag = value;
            }
        }
        public ListedStock Stock {
            get {
                return _stock;
            }
        }
        public void ConnectCountInc() {
            _count++;
        }
        public int ConnectCountDec() {
            return --_count;
        }
                
        public int Price {
            get {
                return _price;
            }
            set {
                _price = value;
                _priceTime = _source.CurrentTime;
            }
        }
        public int PriceTime {
            get {
                return _priceTime;
            }
        }
        public int Volume {
            get {
                return _volume;
            }
            set {
                _prevVolume = _volume;
                _volume = value;
                _volumeTime = _source.CurrentTime;
            }
        }
        public int VolumeTime {
            get {
                return _volumeTime;
            }
        }
        public int VolumeDiff {
            get {
                return _volume - _prevVolume;
            }
        }
        public BTime TradeTime {
            get {
                return _tradeTime;
            }
            set {
                _tradeTime = value;
                _tradeTimeTime = _source.CurrentTime;
            }
        }
        public int TradeTimeTime {
            get {
                return _tradeTimeTime;
            }
        }
        public Ita AskIta {
            get {
                return _askIta;
            }
        }
        public Ita BidIta {
            get {
                return _bidIta;
            }
        }

        //̍XV͉LgƁBłȂ_itaTimeXVȂ
        public void UpdateItaPrice(AskBid askbid, int index, int price) {
            if(askbid==AskBid.Ask)
                _askIta.SetDirect(index, price, _askIta.SashineAt(index).Volume);
            else
                _bidIta.SetDirect(index, price, _bidIta.SashineAt(index).Volume);
            _itaTime = _source.CurrentTime;
        }
        public void UpdateItaVolume(AskBid askbid, int index, int volume) {
            if(askbid==AskBid.Ask)
                _askIta.SetDirect(index, _askIta.SashineAt(index).Price, volume);
            else
                _bidIta.SetDirect(index, _bidIta.SashineAt(index).Price, volume);
            _itaTime = _source.CurrentTime;
        }
        public int ItaTime {
            get {
                return _itaTime;
            }
        }
    }

    //A^Cf[^̋
    //́AyVDDE邢̓\Pbgŕ̃f[^񓯊ɗ悤Ȃ̂z肵ĂB
    //Ƃ΁ATime&Sales𔭍sɂ́Ao̕ωgKɂȂ̂́Ali̕ωoxĂĂ\lȂ΂ȂȂB
    //̏ŝ}[VO^C}ł
    public class RealTimeDataSource : IRealTimeDataReceiver {

        private IPrimaryDataSource _data;
        private TypedHashtable<ListedStock, RealTimeStockEntry> _stockMap;
        private int _itaCapacityTime;
        private int _timeAndSalesCapacityTime;
        private int _manualTime; //mFpɊOZbg鎞
        private bool _useManualTime;

        public RealTimeDataSource(IPrimaryDataSource ds) {
            _data = ds;
            _stockMap = new TypedHashtable<ListedStock, RealTimeStockEntry>();
            _timeAndSalesCapacityTime = 100;
            _itaCapacityTime = 100;
        }
        public void SetManualTimeMode() {
            _useManualTime = true;
        }
        public void SetManualTime(int value) {
            _manualTime = value;
        }

        public int ConnectedCount {
            get {
                return _stockMap.Count;
            }
        }
        //ڑ
        public void Prepare() {
            _stockMap.Clear();
            _data.Prepare(this);
        }
        //I
        public void Dispose() {
            IDictionaryEnumerator de = _stockMap.GetEnumerator();
            while(de.MoveNext()) {
                RealTimeStockEntry e = (RealTimeStockEntry)de.Value;
                _data.Close(e.Stock, e.Tag);
            }
            _data.Dispose();
        }

        //Pʂł̐ڑ
        public void Connect(ListedStockCollection stocks) {
            foreach(ListedStock stock in stocks)
                Connect(stock);
        }
        public void Connect(ListedStock stock) {
            RealTimeStockEntry e = _stockMap[stock];
            if(e!=null)
                e.ConnectCountInc();
            else {
                e = new RealTimeStockEntry(this, stock, MarketUtil.ItaSize(stock));
                _stockMap.Add(stock, e);
                e.Tag = _data.Connect(stock);
            }
        }
        public void Close(ListedStockCollection stocks) {
            foreach(ListedStock stock in stocks)
                Close(stock);
        }
        public void Close(ListedStock stock) {
            RealTimeStockEntry e = _stockMap[stock];
            if(e!=null) {
                Debug.Assert(e.Stock==stock);
                if(e.ConnectCountDec() == 0) {
                    _data.Close(stock, e.Tag);
                    _stockMap.Remove(stock);
                }
            }
        }
        public int CurrentTime {
            get {
                if(_useManualTime)
                    return _manualTime;
                else {
                    long t = DateTime.Now.Ticks / 10000; //Ń~bP
                    return (int)t;
                }
            }
        }

        //A^Cf[^XV DDÊƂlock܂ł͕sv
        #region IRealTimeDataReceiver
        void IRealTimeDataReceiver.OnPrice(ListedStock stock, int price) {
            RealTimeStockEntry e = _stockMap[stock];
            Debug.Assert(e!=null);
            lock(this) {
                e.Price = price;
            }
        }
        void IRealTimeDataReceiver.OnVolume(ListedStock stock, int volume) {
            RealTimeStockEntry e = _stockMap[stock];
            Debug.Assert(e!=null);
            lock(this) {
                e.Volume = volume;
            }
        }
        void IRealTimeDataReceiver.OnTradeTime(ListedStock stock, BTime time) {
            RealTimeStockEntry e = _stockMap[stock];
            Debug.Assert(e!=null);
            lock(this) {
                e.TradeTime = time;
            }
        }
        void IRealTimeDataReceiver.OnItaPrice(ListedStock stock, AskBid askbid, int index, int price) {
            RealTimeStockEntry e = _stockMap[stock];
            Debug.Assert(e!=null);
            lock(this) {
                e.UpdateItaPrice(askbid, index, price);
            }
        }
        void IRealTimeDataReceiver.OnItaVolume(ListedStock stock, AskBid askbid, int index, int volume) {
            RealTimeStockEntry e = _stockMap[stock];
            Debug.Assert(e!=null);
            lock(this) {
                e.UpdateItaVolume(askbid, index, volume);
            }
        }
        #endregion
        //}[VO^C̃nh
        internal void OnMarshallingTime() {
            /*
            int current = this.CurrentTime;
            lock(this) {
                foreach(RealTimeStockEntry e in _stockMap.Values) {
                    ListedStock stock = e.Stock;
                    DaytimeTrade q = BellagioRoot.DaytimeTradeProvider.GetActive(stock);
                    if(IsTimeAndSalesGenerating(e, current)) {
                        TickData td = new TickData(e.TradeTime.AsInt(), e.Price, e.VolumeDiff, GuessTickItaRelation(e));
                        Debug.WriteLineIf(BDebugOpts.TRACE_TIME_AND_SALES_FROM_PRIMARY, String.Format("TS code={0} time={1} price={2} volume={3} dir={4}", stock.Profile.Code, BTime.DefaultFormat(new BTime(td.Time)), td.Price, td.Volume, td.TickItaRelation.ToString()));
                        q.UpdateBySingleTick(td);
                    }
                    if(IsItaGenerating(e, current)) {
                        q.RealTimeIta.AsyncUpdate(e.AskIta, e.BidIta);
                    }
                }
            }
            */
        }
        private bool IsTimeAndSalesGenerating(RealTimeStockEntry e, int current) {
            if(e.TradeTimeTime==0 || e.PriceTime==0) return false; //sďoĂ

            int t = current - e.VolumeTime;
            return t >= _timeAndSalesCapacityTime;
        }
        private bool IsItaGenerating(RealTimeStockEntry e, int current) {
            if(e.ItaTime==0) return false;

            int t = current - e.ItaTime;
            return t >= _itaCapacityTime; //Ō̔XV̎ԁAɂȂ
        }
        private TickItaRelation GuessTickItaRelation(RealTimeStockEntry e) {
            if(e.ItaTime==0) return TickItaRelation.Unknown;

            int price = e.Price;
            if(price==e.AskIta.SashineAt(0).Price)
                return TickItaRelation.Ask;
            else if(price==e.BidIta.SashineAt(0).Price)
                return TickItaRelation.Bid;
            else
                return TickItaRelation.Unknown;
        }

#if UNITTEST
        public int GetConnectionCount(ListedStock stock) {
            return _stockMap[stock]._count;
        }
        public object GetConnectionTag(ListedStock stock) {
            return _stockMap[stock].Tag;
        }
#endif

    }
#if UNITTEST
    [TestFixture]
    public class PrimaryDataTests : IPrimaryDataSource  {
        private bool _prepared;
        private bool _disposed;
        private int _connectCount;
        private int _closeCount;
        private IRealTimeDataReceiver _rec;

        void IPrimaryDataSource.Prepare(IRealTimeDataReceiver receiver) {
            _prepared = true;
            _rec = receiver;
        }

        void IPrimaryDataSource.Dispose() {
            _disposed = true;
        }

        object IPrimaryDataSource.Connect(ListedStock stock) {
            _connectCount++;
            return stock.Profile.Code;
        }

        void IPrimaryDataSource.Close(ListedStock stock, object tag) {
            _closeCount++;
            Assert.AreEqual(stock.Profile.Code, tag);
        }

        [TestFixtureSetUp]
        public void Setup() {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.RUN_PRIMARY_DATASOURCE = true;
            p.RUN_DATA_SUBSCRIBER_ENGINE = true;
            BellagioRoot.Init(p);
        }
        [TestFixtureTearDown]
        public void TearDown() {
            BellagioRoot.Terminate();
        }

        private RealTimeDataSource Init() {
            _prepared = false;
            _disposed = false;
            _connectCount = 0;
            _closeCount = 0;
            RealTimeDataSource ds = new RealTimeDataSource(this);
            ds.SetManualTimeMode();
            return ds;
        }

        [Test]
        public void ConnectClose() {
            RealTimeDataSource ds = Init();
            ds.Prepare();
            Assert.IsTrue(_prepared);

            ListedStock st = new BasicStockProfile("1", "1", StockExchange.B).Primary;
            ds.Connect(st);

            Assert.AreEqual(1, _connectCount);
            Assert.AreEqual(st.Profile.Code, ds.GetConnectionTag(st));
            Assert.AreEqual(1, ds.GetConnectionCount(st));

            //Qdڑ
            ds.Connect(st);
            Assert.AreEqual(1, _connectCount);
            Assert.AreEqual(0, _closeCount);
            Assert.AreEqual(2, ds.GetConnectionCount(st));

            //PƂ
            ds.Close(st);
            Assert.AreEqual(1, _connectCount);
            Assert.AreEqual(0, _closeCount);
            Assert.AreEqual(1, ds.GetConnectionCount(st));
            Assert.AreEqual(1, ds.ConnectedCount);

            //xBŊSɕ
            ds.Close(st);
            Assert.AreEqual(1, _closeCount);
            Assert.AreEqual(0, ds.ConnectedCount);

            ds.Dispose();
            Assert.IsTrue(_disposed);
        }

        [Test]
        public void Generation() {
            RealTimeDataSource ds = Init();
            ds.Prepare();
            ListedStock st = new BasicStockProfile("1", "1", StockExchange.B).Primary;
            DaytimeTrade q = BellagioRoot.DaytimeTradeProvider.GetActive(st);
            RealTimeIta ri = BellagioRoot.ItaProvider.GetActive(st);
            ds.Connect(st);

            ds.SetManualTime(1000);
            _rec.OnItaPrice(st, AskBid.Ask, 0, 1);
            ds.OnMarshallingTime();
            Assert.AreEqual(0, q.TimeAndSales.Count);
            Assert.AreEqual(0, ri.Ask.SashineAt(0).Price); //܂fĂȂ

            for(int i=0; i<ri.Ask.Size; i++) {
                _rec.OnItaPrice(st, AskBid.Ask, i, 1000+i);
                _rec.OnItaPrice(st, AskBid.Bid, i, 999-i);
                _rec.OnItaVolume(st, AskBid.Ask, i, 10000);
                _rec.OnItaVolume(st, AskBid.Bid, i, 10000);
            }
            ds.SetManualTime(2000);
            ds.OnMarshallingTime();
            Assert.AreEqual(0, q.TimeAndSales.Count);
            for(int i=0; i<ri.Ask.Size; i++) {
                Assert.AreEqual(1000+i, ri.Ask.SashineAt(i).Price); //͂낤
                Assert.AreEqual( 999-i, ri.Bid.SashineAt(i).Price);
                Assert.AreEqual(10000, ri.Ask.SashineAt(i).Volume);
                Assert.AreEqual(10000, ri.Bid.SashineAt(i).Volume);
            }

            _rec.OnPrice(st, 1000);
            _rec.OnVolume(st, 2000);
            _rec.OnTradeTime(st, new BTime(10,0,0));
            ds.OnMarshallingTime();
            Assert.AreEqual(0, q.TimeAndSales.Count);
            ds.SetManualTime(3000);
            //ł͂߂Ė
            ds.OnMarshallingTime();
            Assert.AreEqual(1, q.TimeAndSales.Count);
            TickData td = q.TimeAndSales.TickAt(0);
            Assert.AreEqual(1000, td.Price);
            Assert.AreEqual(36000, td.Time);
            Assert.AreEqual(2000, td.Volume);
            Assert.AreEqual(TickItaRelation.Ask, td.TickItaRelation);

            //ɂ
            _rec.OnPrice(st, 999);
            _rec.OnVolume(st, 4000);
            ds.SetManualTime(4000);
            ds.OnMarshallingTime();
            Assert.AreEqual(2, q.TimeAndSales.Count);
            td = q.TimeAndSales.TickAt(1);
            Assert.AreEqual(999, td.Price);
            Assert.AreEqual(36000, td.Time);
            Assert.AreEqual(2000, td.Volume);
            Assert.AreEqual(TickItaRelation.Bid, td.TickItaRelation);

            ds.Dispose();
        }

        [Test]
        public void TimeBound() {
            RealTimeDataSource ds = Init();
            ds.Prepare();
            ListedStock st = new BasicStockProfile("1", "1", StockExchange.B).Primary;
            DaytimeTrade q = BellagioRoot.DaytimeTradeProvider.GetActive(st);
            ds.Connect(st);

            //Ԃint̋EzĂOK?
            ds.SetManualTime(Int32.MaxValue-1000);
            _rec.OnTradeTime(st, new BTime(10, 0, 0));
            _rec.OnPrice(st, 1000);
            _rec.OnVolume(st, 1000);
            ds.SetManualTime(Int32.MaxValue);
            ds.OnMarshallingTime();
            Assert.AreEqual(1, q.TimeAndSales.Count);

            _rec.OnVolume(st, 2000);
            ds.SetManualTime(Int32.MinValue+1000);
            ds.OnMarshallingTime();
            Assert.AreEqual(2, q.TimeAndSales.Count);

        }

    }
#endif
#endif
}
