/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/DaytimeTradeProvider.cs#12 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading;

using Poderosa;
using Poderosa.Util.Collections;
using Bellagio.Values; //NSO
using Bellagio.Environment;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {

    public interface ITickDataSource : IPrimaryStreamDataBase {
        AsyncOpResult OpenTickData(Stock stock, ref string message);
        void CloseTickData(Stock stock);
    }

    public enum OpenHighLow {
        Open, High, Low
    }

    //TickDatáAf[^ڑsꂪJĂTime&Sales𖄂߁Aŏ߂DaytimeTradep\ɂȂ 
    public interface ITickDataHost : IPrimaryStreamDataHostBase {
        void OnConnectFailed(Stock stock, string message);

        //Zbg(GMO̓CfbNXÕeBbNɂȂ)
        void DynamicReset(Stock stock);

        //ڑɃZbg ȍ~TICKf[^ɏ]ėݐςĂ
        void OnInitialPriceInfo(Stock stock, int tickCount, int lastCloseDate, int lastCloseValue, int creditLong, int creditShort, int currentPrice, int currentPriceTime, int currentVolume, double currentVWAP, double currentTotalAmount);
        //PTick킩ĂƂ
        void OnTickData(Stock stock, int index, int time, int price, int volume, double vwap);

        //nlElEl̍XV
        void OnOpenHighLow(Stock stock, OpenHighLow ohl, int time, int price);
        //ŗǋCzP{̍XV
        void OnBestBidAsk(Stock stock, AskBid ask_bid, int price, int volume, SashineRemark remark);
    }

    public class IntraDayTradeProvider : StockDataProviderBaseT<IntraDayTrade, IntraDayTradeInitTick>, ITickDataHost {
        private ITickDataSource _dataSource;

        public IntraDayTradeProvider() {
        }
        protected override IntraDayTrade CreateValue(Stock stock) {
            return new IntraDayTrade(stock);
        }
        protected override IntraDayTradeInitTick CreateInitializer(IntraDayTrade value) {
            return new IntraDayTradeInitTick(value);
        }
        protected override AsyncOpResult OpenValue(IntraDayTradeInitTick initializer, ref string message) {
            EnsureDataSource();
            AsyncOpResult r = _dataSource.OpenTickData(initializer.Stock, ref message);
            if(r==AsyncOpResult.Succeeded) {
                Debug.Assert(initializer.IsCompleted);
                Debug.Assert(initializer.IntraDayTrade.DataChangeTag.IsOK);
            }
            return r;
        }
        protected override void CloseValue(Stock stock, IntraDayTrade value) {
            _dataSource.CloseTickData(stock);
            BellagioRoot.DataSubscriberManager.RemoveSubscriber(value);
        }
        public void CloseAll() {
            foreach(StockTag e in _data.Values)
                if(e.value.DataStatus==PrimaryDataStatus.OK)
                    BellagioRoot.DataSubscriberManager.RemoveSubscriber(e.value);
            _data.Clear();
        }

        public List<IntraDayTrade> GetAllActives() {
            List<IntraDayTrade> r = new List<IntraDayTrade>();
            lock(this) {
                IDictionaryEnumerator e = _data.GetEnumerator();

                while(e.MoveNext())
                    r.Add(((StockTag)e.Value).value);

                return r;
            }
        }

        /*
        void ITickDataHost.OnConnectSucceeded(Stock stock) {
            base.CompleteAsyncLoading(stock); 
            DaytimeTrade q = base.Lookup(stock);
            Debug.Assert(q.InitialTicks.IsCompleted);
            RunQuote(q);
            BellagioRoot.DataSubscriberManager.ReserveExecByStock(stock, SubscribeDataKind.Ticks);
        }
        */
        void ITickDataHost.OnConnectFailed(Stock stock, string message) {
            base.FailLoading(stock, message);
            BellagioRoot.DataSubscriberManager.ReserveErrorByStock(stock, SubscribeDataKind.Ticks, message);
        }
        void ITickDataHost.OnTickData(Stock stock, int index, int time, int price, int volume, double vwap) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);
            if(tag.value.DataStatus==PrimaryDataStatus.Preparing) {
                tag.initializer.AddTick(index, time, price, volume, vwap);
                if(tag.initializer.IsCompleted) {
                    tag.initializer.CompleteInitialization();
                    base.CompleteLoading(stock);
                }
            }
            else {
                Debug.Assert(tag.value.DataChangeTag.IsOK);
                CurrentPriceInfo cp = tag.value.CurrentPriceInfo;
                cp.CurrentPrice.SetValue(price);
                cp.CurrentPriceTime.SetValue(new BTime(time));
                cp.Volume.SetValue(cp.Volume.Value + volume);
                double amount = vwap * cp.Volume.Value * stock.Profile.ImplicitPriceFactor;
                cp.TotalAmount.SetValue(amount, MarketUtil.FormatTotalAmount(amount));
                cp.VWAP.SetValue(vwap, vwap.ToString("F3"));

                TickData td = new TickData(time, price, volume, TickItaRelation.Unknown); //TickItaRelationڂ
                Debug.WriteLineIf(BDebugOpts.TRACE_TIME_AND_SALES_FROM_PRIMARY, String.Format("TS code={0} time={1} price={2} volume={3} dir={4}", stock.Profile.Code, BTime.DefaultFormat(new BTime(td.Time)), td.Price, td.Volume, td.TickItaRelation.ToString()));
                tag.value.UpdateBySingleTick(td); //̐DataSubsciberManagerւ̒ʒm܂ł
            }
        }
        void ITickDataHost.OnInitialPriceInfo(Stock stock, int tickCount, int lastCloseDate, int lastCloseValue, int creditLong, int creditShort, int currentPrice, int currentPriceTime, int currentVolume, double currentVWAP, double currentTotalAmount) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);
            CurrentPriceInfo cp = tag.value.CurrentPriceInfo;

            if(lastCloseDate!=0) {
                BDate dt = new BDate(lastCloseDate);
                cp.LastCloseDate.SetValue(dt, BDate.FormatYYMMDD(dt));
            }
            if(lastCloseValue!=0)
                cp.LastClose.SetValue(lastCloseValue);
            if(creditLong!=0)
                cp.CreditLongVolume.SetValue(creditLong);
            if(creditShort!=0)
                cp.CreditShortVolume.SetValue(creditShort);
            if(currentPrice!=0)
                cp.CurrentPrice.SetValue(currentPrice);
            if(currentPriceTime!=0)
                cp.CurrentPriceTime.SetValue(new BTime(currentPriceTime));
            if(currentVolume!=0)
                cp.Volume.SetValue(currentVolume);
            if(currentVWAP!=0)
                cp.VWAP.SetValue(currentVWAP);
            if(currentTotalAmount!=0)
                cp.TotalAmount.SetValue(currentTotalAmount, MarketUtil.FormatTotalAmount(currentTotalAmount));

            cp.AskPrice.SetEmpty();
            cp.AskVolume.SetEmpty();
            cp.BidPrice.SetEmpty();
            cp.BidVolume.SetEmpty();

            //荞݌^̂Ƃ͕KvɉbZ[W
            if(tag.value.DataStatus==PrimaryDataStatus.OK) {
                if(tickCount > tag.value.TimeAndSales.Count) {
                    tag.value.WarningStatus |= IntraDayTradeWarnings.TickSpilled;
                    tag.value.FireEventToSubscriberManager();
                }
            }
            else {
                tag.initializer.Initialize(tickCount);
                if(tag.initializer.IsCompleted) {
                    tag.initializer.CompleteInitialization();
                    base.CompleteLoading(stock);
                }
            }
        }

        //nlElEl̍XV
        void ITickDataHost.OnOpenHighLow(Stock stock, OpenHighLow ohl, int time, int price) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);
            CurrentPriceInfo cp = tag.value.CurrentPriceInfo;
            BTime bt = time==0? null : new BTime(time);
            string expr = time==0? null : BTime.FormatHHMM(bt);

            //͍XVȂꍇ邱Ƃɒ
            if(ohl==OpenHighLow.Open) {
                cp.OpenPrice.SetValue(price);
                if(bt!=null) cp.OpenTime.SetValue(bt, expr);
            }
            else if(ohl==OpenHighLow.High) {
                cp.HighPrice.SetValue(price);
                if(bt!=null) cp.HighTime.SetValue(bt, expr);
            }
            else if(ohl==OpenHighLow.Low) {
                cp.LowPrice.SetValue(price);
                if(bt!=null) cp.LowTime.SetValue(bt, expr);
            }

            //Open/High/Low̕ω̂Ƃ͕KTICKĂ͂Ȃ̂SubscriberManagerɂ͒ʒmȂBߒ̂ƂlB
        }
        //ŗǋCzP{̍XV
        void ITickDataHost.OnBestBidAsk(Stock stock, AskBid ask_bid, int price, int volume, SashineRemark remark) {
            StockTag tag = base.GetTagOrNull(stock);
            Debug.Assert(tag!=null);
            CurrentPriceInfo cp = tag.value.CurrentPriceInfo;

            //TODO flag̃T|[g
            if(ask_bid==AskBid.Ask) {
                if(price!=0) cp.AskPrice.SetValue(price);
                if(volume!=0) cp.AskVolume.SetValue(volume);
            }
            else {
                if(price!=0) cp.BidPrice.SetValue(price);
                if(volume!=0) cp.BidVolume.SetValue(volume);
            }

            if(tag.value.DataChangeTag.IsOK)
                BellagioRoot.DataSubscriberManager.ReserveExecByStock(stock, SubscribeDataKind.Ticks);
        }

        //IZbg
        void ITickDataHost.DynamicReset(Stock stock) {
            IntraDayTrade q = base.Lookup(stock);
            Debug.Assert(q!=null);
            CurrentPriceInfo cp = q.CurrentPriceInfo;

            //sׂƂ́ATimeAndSalesƑ̃NAAݒlOIlɃRs[
            int price = q.TimeAndSales.LastPrice;
            q.Reset();
            if(price!=-1)
                cp.LastClose.SetValue(price);
            
            RealtimeIta ita = BellagioRoot.ItaProvider.Lookup(stock);
            if(ita!=null) {
                ita.Reset();
                IItaDataHost itahost = BellagioRoot.ItaProvider;
                itahost.SetItaAll(q.Stock, ita.AskWork, ita.BidWork);
            }

            BellagioRoot.DataSubscriberManager.ReserveExecByStock(stock, SubscribeDataKind.Ticks);
        }



        IAdaptable IAdaptable.GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        private void EnsureDataSource() {
            if(_dataSource!=null) return;
            _dataSource = BellagioRoot.DataSourceHost.ExternalEnvironment.TickDataSource;
            _dataSource.Prepare(this);
        }


#if UNITTEST
        //eXgp̃C^tF[X𒼐ڃZbgł悤
        public void SetTickDataSource(ITickDataSource ds) {
            _dataSource = ds;
            if(ds!=null) ds.Prepare(this);
        }
        public IntraDayTradeInitTick MapInitializer(Stock stock) {
            return this.GetTagOrNull(stock).initializer;
        }
#endif
    }


#if UNITTEST
    //rIGITickDatanC^tF[X̃vgReXg
    [TestFixture]
    public class TickDataProtocolCheck : ITickDataSource {
        private Stock _stock;
        private ITickDataHost _host;

        [TestFixtureSetUp]
        public void Setup() {
            BellagioEnvironmentParam p = new BellagioEnvironmentParam();
            p.SetupForUnitTest();
            p.UNITTEST_STOCK_COUNT = 1;
            p.RUN_DATA_SUBSCRIBER_ENGINE = true;

            BellagioRoot.Init(p);
            BellagioRoot.BootForTest();

            BellagioRoot.IntraDayTradeProvider.SetTickDataSource(this);
            _stock = BellagioRoot.GlobalStockCollection.FindExact("1").Primary;
        }
        [TestFixtureTearDown]
        public void TearDown() {
            BellagioRoot.DataSubscriberManager.Terminate();
        }
        private void RunAndJoin(ThreadStart st) {
            Thread th = new Thread(st);
            th.Start();
            th.Join();
        }

        [Test]
        public void NoInitialTicks() { //InitialTicksv̏ꍇ
            string message = "";
            IntraDayTrade dt = null;
            BellagioRoot.IntraDayTradeProvider.Open(_stock, ref dt, ref message);
            RunAndJoin(new ThreadStart(NoInitialTicksBody));
            BellagioRoot.IntraDayTradeProvider.AddRef(_stock);
            dt =BellagioRoot.IntraDayTradeProvider.Lookup(_stock);
            Assert.IsNotNull(dt);
            Assert.AreEqual(0, dt.TimeAndSales.Count);
            BellagioRoot.IntraDayTradeProvider.Release(_stock);
        }
        private void NoInitialTicksBody() {
            _host.OnInitialPriceInfo(_stock, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
        }

        [Test]
        public void WaitTicks() {
            string message = "";
            IntraDayTrade dt = null;
            BellagioRoot.IntraDayTradeProvider.Open(_stock, ref dt, ref message);
            RunAndJoin(new ThreadStart(WaitTicksBody1));
            BellagioRoot.IntraDayTradeProvider.AddRef(_stock);
            dt = BellagioRoot.IntraDayTradeProvider.Lookup(_stock);
            Assert.IsNull(dt); //̒iKł͖
            RunAndJoin(new ThreadStart(WaitTicksBody2));
            dt = BellagioRoot.IntraDayTradeProvider.Lookup(_stock);
            Assert.IsNotNull(dt);
            Assert.AreEqual(3, dt.TimeAndSales.Count);
            Assert.AreEqual(2, dt.Minutely.Count);
            BellagioRoot.IntraDayTradeProvider.Release(_stock);
        }
        private void WaitTicksBody1() {
            _host.OnInitialPriceInfo(_stock, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0);
            _host.OnTickData(_stock, 0, 3600*9, 1, 1, 0);
        }
        private void WaitTicksBody2() {
            IntraDayTradeInitTick init = BellagioRoot.IntraDayTradeProvider.MapInitializer(_stock);
            int time = new BTime(9, 2, 0).AsInt();
            _host.OnTickData(_stock, 1, time, 1, 1, 0);
            Assert.IsFalse(init.IsCompleted);
            _host.OnTickData(_stock, 2, time, 1, 1, 0);
            Assert.IsTrue(init.IsCompleted);
        }

        //TODO ̃G[NAbZ[W{bNX\nP̂݌Ă΂邱ƁACloseTickDataP̂݌Ă΂邱ƁAꂼmF

        //TODO OpenTickData^[ɔ񓯊Xbhf[^uɂƂubN邱ƂmF

        public AsyncOpResult OpenTickData(Stock stock, ref string message) {
            return AsyncOpResult.Async;
        }

        public void CloseTickData(Stock stock) {
        }

        public void Prepare(IPrimaryStreamDataHostBase site) {
            _host = site as ITickDataHost;
        }

        public void Terminate() {
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

#endif
}