/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: TabBar.cs,v 1.7 2006/11/03 09:17:54 okajima Exp $
 */
using System;
using System.Diagnostics;
using System.Drawing;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;

using Poderosa.Util.Drawing;
using Poderosa.Util;

namespace Poderosa.Forms {

    //e^uvfɊ֘AtIuWFNgBPoderosȁꍇIPoderosaDocumentɂȂB
    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
    public abstract class TabKey {
        protected object _element;

        public TabKey(object element) {
            Debug.Assert(element!=null);
            _element = element;
        }
        public object Element {
            get {
                return _element;
            }
        }
        public abstract string Caption { get; }
        public abstract Image Icon { get; } //must be 16*16


        //̗vfŔr
        public static bool operator==(TabKey key1, TabKey key2) {
            return _Equals(key1, key2);
        }
        public static bool operator!=(TabKey key1, TabKey key2) {
            return !_Equals(key1, key2);
        }
        public override int GetHashCode() {
            return _element.GetHashCode();
        }
        public override bool Equals(object obj) {
            return _element.Equals(((TabKey)obj)._element);
        }

        private static bool _Equals(TabKey key1, TabKey key2) {
            if(Object.ReferenceEquals(key1, null))
                return Object.ReferenceEquals(key2, null);
            else
                return !Object.ReferenceEquals(key2, null) && key1._element==key2._element;
        }
    }
    
    internal enum DrawState {
		Normal,
		Hover,
		Selected
	}

	//^u̕`ɕKvȐFEtHg^
	internal class TabBarDrawing {
		private Color _backgroundColor;
		private DrawUtil.RoundRectColors _roundRectColors;
		private Font _font;
		private int _omittedMarkWidth;

		public Color BackgroundColor {
			get {
				return _backgroundColor;
			}
		}	 
		public DrawUtil.RoundRectColors RoundRectColors {
			get {
				return _roundRectColors;
			}
		}
		public Font Font {
			get {
				return _font;
			}
		}
		public int OmittedMarkWidth {
			get {
				return _omittedMarkWidth;
			}
		}

		public static TabBarDrawing CreateNormalStyle(Graphics g) {
			TabBarDrawing t = new TabBarDrawing();
			t._font = new Font("MS UI Gothic", 9);
			t._omittedMarkWidth = 6;

			Color c = SystemColors.Control;
			t._backgroundColor = c;
		
			t._roundRectColors = new DrawUtil.RoundRectColors();
			t._roundRectColors.border_color = DrawUtil.ToCOLORREF(DrawUtil.DarkColor(Color.Orange));
			t._roundRectColors.inner_color = DrawUtil.ToCOLORREF(SystemColors.Control);
			t._roundRectColors.outer_color = DrawUtil.ToCOLORREF(SystemColors.Control);
			t._roundRectColors.lightlight_color = DrawUtil.MergeColor(t._roundRectColors.border_color, t._roundRectColors.outer_color);
			t._roundRectColors.light_color = DrawUtil.MergeColor(t._roundRectColors.lightlight_color, t._roundRectColors.border_color);

			return t;
		}
		public static TabBarDrawing CreateActiveStyle(TabBarDrawing normal, Graphics g) {
			TabBarDrawing t = new TabBarDrawing();
			t._font = new Font(normal.Font, normal.Font.Style|FontStyle.Bold);
			t._omittedMarkWidth = 9;

			Color c = SystemColors.ControlLightLight;
			t._backgroundColor = c;

			t._roundRectColors = new DrawUtil.RoundRectColors();
			t._roundRectColors.border_color = DrawUtil.ToCOLORREF(SystemColors.ControlDarkDark);
			t._roundRectColors.inner_color = DrawUtil.ToCOLORREF(c);
			t._roundRectColors.outer_color = DrawUtil.ToCOLORREF(SystemColors.Control);
			t._roundRectColors.lightlight_color = DrawUtil.MergeColor(t._roundRectColors.border_color, t._roundRectColors.outer_color);
			t._roundRectColors.light_color = DrawUtil.MergeColor(t._roundRectColors.lightlight_color, t._roundRectColors.border_color);

			return t;
		}

		public void Dispose() {
			_font.Dispose();
		}
	}

	//TabBar{
    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
	public class TabBar : UserControl {
		
		private static IComparer<TabBarButton> _widthComparer; //̕ёւ
		private static DragAndDropSupport _dragAndDrop; //{staticł͂ȂʓrĂH
		private static TabBarDrawing _normalDrawing; //`ɂĂ̏
		private static TabBarDrawing _activeDrawing;

		private TabBarTable _parentTable; //Rei
		private List<TabBarButton> _buttons;       //\Ă鏇̃{^
        private List<TabBarButton> _sortedButtons; //Ƀ\[ĝꎞIɕۑ
		private ToolTip _tabToolTip;      //{^ToolTip

		//NbNMouseMoveĂ܂̂ŁAhbOJn𔻒łȂ
		private int _dragStartPosX;
		private int _dragStartPosY;

		private AdjustEachWidthResult _lastAdjustmentResult;

		//CAEgp萔
		private const int UNITHEIGHT = 21;
		private const int MINIMUM_BUTTON_WIDTH = 24;
		private const int BUTTON_MARGIN = 4;
		private const int EXTRA_FOR_ACTIVE = 16;
		private const int BUTTON_Y = 3;

		//
		static TabBar() {
			_widthComparer = new TabBarButton.WidthComparer();
			_dragAndDrop = new DragAndDropSupport();
		}

		public TabBar(TabBarTable parent) {
			_parentTable = parent;
            _buttons = new List<TabBarButton>();
			_tabToolTip = new ToolTip();
            this.AllowDrop = true;
            this.Height = UNITHEIGHT;
		}
		protected override void OnLoad(EventArgs e) {
			base.OnLoad (e);
			if(_normalDrawing==null) {
				Graphics g = this.CreateGraphics();
				_normalDrawing = TabBarDrawing.CreateNormalStyle(g);
				_activeDrawing = TabBarDrawing.CreateActiveStyle(_normalDrawing, g);
				g.Dispose();
			}
		}

		//vpeB
	    internal TabBarDrawing NormalDrawing {
			get {
				return _normalDrawing;
			}
		}
		internal TabBarDrawing ActiveDrawing {
			get {
				return _activeDrawing;
			}
		}
		public int TabCount {
			get {
				return _buttons.Count;
			}
		}
		public TabBarTable ParentTable {
			get {
				return _parentTable;
			}
		}
		internal TabBarButton ButtonAt(int index) {
			return _buttons[index];
		}
        internal IEnumerable<TabBarButton> Buttons {
            get {
                return _buttons;
            }
        }
		internal AdjustEachWidthResult LastAdjustmentResult {
			get {
				return _lastAdjustmentResult;
			}
		}
        internal TabBarButton FindButton(TabKey key) {
			foreach(TabBarButton button in _buttons)
				if(button.TabKey==key) return button;
			return null;
		}

		private int GetTabAreaWidth() {
			return this.Width-4;
		}


		public void AddTab(TabKey key, int index, int indexInBar) {
			TabBarButton b = CreateButton(key, index, indexInBar);
			_buttons.Add(b);
			this.Controls.Add(b);
			TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(this);
		}

		public void RemoveTab(TabKey key) {
			TabBarButton b = FindButton(key);
			Debug.Assert(b!=null);

			Controls.Remove(b);
			_buttons.Remove(b);
            TabBarUpdateState.GetCurrent().MarkIndexAssignmentChanged(this);
		}

		private TabBarButton CreateButton(TabKey key, int index, int indexInBar) {
			TabBarButton b = new TabBarButton(this, key, index, indexInBar);
			//TODO _tabToolTip.SetToolTip(b, document.Description);

			b.Visible = true;
			b.Top = 1;
			b.Height = UNITHEIGHT-2;
			b.TabStop = false;
			b.Click += new EventHandler(RootActivator);
            b.DoubleClick += new EventHandler(OnDoubleClick);
			b.MouseDown += new MouseEventHandler(OnMouseDown);
			b.MouseUp += new MouseEventHandler(OnMouseUp);
			b.MouseMove += new MouseEventHandler(OnMouseMove);
            b.KeyDown += new KeyEventHandler(OnButtonKeyDown);
			return b;
		}


		private void OnMouseUp(object sender, MouseEventArgs args) {
            if(args.Button==MouseButtons.Right) {
                TabKey key = ((TabBarButton)sender).TabKey;
                _parentTable.DoRightButtonAction(key);
            }
            else if(args.Button==MouseButtons.Middle) {
                TabKey key = ((TabBarButton)sender).TabKey;
                _parentTable.DoMiddleButtonAction(key);
            }
		}

		private void OnMouseDown(object sender, MouseEventArgs args) {
			if(args.Button!=MouseButtons.Left) return;
			_dragStartPosX = args.X;
			_dragStartPosY = args.Y;
		}
		private void OnMouseMove(object sender, MouseEventArgs args) {
			if(args.Button!=MouseButtons.Left) return;

			if(Math.Abs(_dragStartPosX-args.X) + Math.Abs(_dragStartPosY-args.Y) >= 3) {
				TabBarButton btn = sender as TabBarButton;
				Debug.Assert(btn!=null);
				btn.ClearMouseTrackingFlags();
				_dragAndDrop.StartDrag(btn);
			}
		}
        private void RootActivator(object sender, EventArgs args) {
            try {
                TabKey key = ((TabBarButton)sender).TabKey;
                //ɃANeBuȂłAΉr[ɃtH[JXĂȂǂ̌ʂKvȂƂB

                TabBarUpdateState state = TabBarUpdateState.BeginUpdate("ui-activate");
                _parentTable.Activate(key, true);
                state.Commit();
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
        }

        private void OnDoubleClick(object sender, EventArgs args) {
            try {
                TabBarButton btn = sender as TabBarButton;
                Debug.Assert(btn!=null);
                _parentTable.CaptureState.StartCapture(this, btn.TabKey);
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
        }
        protected override void OnMouseDown(MouseEventArgs e) {
            base.OnMouseDown(e);
            try {
                if(_parentTable.CaptureState.IsCapturing)
                    _parentTable.CaptureState.EndCapture(this.PointToScreen(new Point(e.X, e.Y)));
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
        }
        protected override void OnMouseCaptureChanged(EventArgs e) {
            base.OnMouseCaptureChanged(e);
            if(_parentTable.CaptureState.IsCapturing) _parentTable.CaptureState.CancelCapture();
        }
        private void OnButtonKeyDown(object sender, KeyEventArgs e) {
            if(e.KeyCode==Keys.Enter)
                RootActivator(sender, e); //EnterŃANeBx[gBNbNƓľ
        }

		//e{^SufficientWidthvZ
		internal void CalcSuffientWidths() {
			Graphics g = this.CreateGraphics();
			foreach(TabBarButton b in _buttons) {
				int t = b.CalcSufficientWidth(g);
			}
			g.Dispose();

			//kKvƂ͂܂\[g@
			_sortedButtons = new List<TabBarButton>(_buttons);
			_sortedButtons.Sort(_widthComparer);
		}

		//{^̍Ĕzu
		internal void ArrangeButtons() {
			int total = 0;
            if(_buttons.Count==0) return;

			Graphics g = this.CreateGraphics();
			foreach(TabBarButton b in _buttons) {
				int t = b.SufficientWidth; //͕KvȂ̂vZς݂łƂO
				b.TemporaryWidth = t;
				total += t;
			}

			int x = BUTTON_MARGIN;
			int remaining = this.ClientSize.Width - x*2/*[}[W*/ - (_buttons.Count-1)*BUTTON_MARGIN; 
			AdjustEachWidthResult adjust_result = AdjustEachWidth(total, remaining); 
			_lastAdjustmentResult = adjust_result;

			foreach(TabBarButton  button in _buttons) {
				button.AdjustTextAndWidth(g, button.TemporaryWidth);
				button.Left = x;
				x += button.Width + BUTTON_MARGIN;
				button.Top = BUTTON_Y;
				button.Visible = true;
			}

			g.Dispose();

			this.Invalidate(true);
		}

        //indexUȂ
        internal void AllocateIndex(int index) {
            foreach(TabBarButton button in _buttons) button.Index = index++;
            Invalidate(true);
        }

		internal enum AdjustEachWidthResult {
			Sufficient,
			Adjusted,
			Overflow
		}

		//o[ŜŃ{^̍vsufficient_widthKvAactual_widthۂ̃TCYłƂ܂ۂߍ
		private AdjustEachWidthResult AdjustEachWidth(int sufficient_width, int actual_width) {
			int to_be_shrinked = sufficient_width - actual_width;
			if(to_be_shrinked <= 0) return AdjustEachWidthResult.Sufficient; //Ȃď\TCY

			int order = 0;
			while(to_be_shrinked > 0) {
				//Kv̑ order ʂɒ
				TabBarButton b = _sortedButtons[order];
				//ʂƂ̍A͌El܂ł̍̒iKŏkł郊~bg
				int max_shrink = order==_sortedButtons.Count-1?
					b.TemporaryWidth - MINIMUM_BUTTON_WIDTH :
					b.TemporaryWidth-_sortedButtons[order+1].TemporaryWidth;
				Debug.Assert(max_shrink >= 0);

				if((order+1)*max_shrink < to_be_shrinked) { //̏ʂ܂őSkĂԂɍȂƂ
					for(int i=0; i<=order; i++)
						_sortedButtons[i].TemporaryWidth -= max_shrink;
				}
				else { //order܂łŏk\
					int shrink = to_be_shrinked / (order+1);
					int mod = to_be_shrinked % (order+1); //ŏmod͂1shrinkĂ҂ύtƂ
					Debug.Assert(mod * (shrink+1) + (order+1-mod) * shrink == to_be_shrinked);
					for(int i=0; i<=order; i++)
						_sortedButtons[i].TemporaryWidth -= i<mod? shrink+1 : shrink;
				}

				to_be_shrinked -= (order+1)*max_shrink;
				order++;

				if(order==_sortedButtons.Count && to_be_shrinked>0) return AdjustEachWidthResult.Overflow; //Ō܂ŏkă_Ȃ炠߂
			}
			//܂łāAe_tempWidth͒̒lĂ邱ƂɂȂ
			return AdjustEachWidthResult.Adjusted;
		}

        //I̎s static constructorftHgRXgN^Ɠ\Ȃ̂ŎgȂB肦ȂB
        public static void Init() {
            Application.ApplicationExit += new EventHandler(OnAppExit);
        }

		private static void OnAppExit(object sender, EventArgs args) {
            if(_normalDrawing!=null) {
                _normalDrawing.Dispose();
                _activeDrawing.Dispose();
            }
		}

		protected override void OnPaint(PaintEventArgs arg) {
            base.OnPaint(arg);
            if(TabBarUpdateState.IsActive) return;

			TabBarUpdateState.AssertInactive(); //CommitȌԂŕ`悷邱Ƃ͂܂Ȃ
			//ɋ؂
			Graphics g = arg.Graphics;
			Pen p = SystemPens.WindowFrame;
			g.DrawLine(p, 0, 0, Width, 0);
			p = SystemPens.ControlLight;
			g.DrawLine(p, 0, 1, Width, 1);
			
			//DropPoint Effect
			if(_dragAndDrop.OwnsDropPoint(this)) {
				DrawDropPointEffect(g, _dragAndDrop.CurrentDropPoint.PosX(2)-1, BUTTON_Y);
			}
		}

		protected override void OnResize(EventArgs e) {
			base.OnResize (e);
			this.ArrangeButtons();
		}

		//Drag & Drop֌W
        protected override void OnDragEnter(DragEventArgs drgevent) {
            base.OnDragEnter(drgevent);
            _parentTable.ByPassDragEnter(drgevent);
        }
		protected override void OnDragOver(DragEventArgs drgevent) {
			base.OnDragOver (drgevent);
          
			DragAndDropSupport.DropPoint point = null;
			if(_dragAndDrop.CanDrop(drgevent.Data, this)) {
				const int DROP_CAPACITY_WIDTH = 6; //{^̋Eǂ̈ʒu܂łhbv\Ƃ邩
				if(_buttons.Count==0) { //{^ȂƂ͓
					point = new DragAndDropSupport.DropPoint(this, null, true);
				}
				else {
					Point pt = this.PointToClient(new Point(drgevent.X, drgevent.Y));
					for(int i=0; i<_buttons.Count; i++) {
						TabBarButton btn = _buttons[i];
						if(Math.Abs(btn.Top+btn.Height/2-pt.Y) < btn.Height/2) {
							if(Math.Abs(btn.Left-BUTTON_MARGIN/2-pt.X) < DROP_CAPACITY_WIDTH) {
								point = new DragAndDropSupport.DropPoint(this, btn, true);
								break;
							}
							else if(i==_buttons.Count-1 && btn.Right+BUTTON_MARGIN/2<pt.X) { //E[ݒ肪ł͍̂Ō̃Rg[
								point = new DragAndDropSupport.DropPoint(this, btn, false);
								break;
							}
						}
					}
				}

				if(point!=null)
					drgevent.Effect = DragDropEffects.Link;
				else
					drgevent.Effect = DragDropEffects.Move;
			}
           	_dragAndDrop.SetDropPoint(point);
		}

		protected override void OnDragDrop(DragEventArgs drgevent) {
			base.OnDragDrop (drgevent);
            if(!_dragAndDrop.OwnsDropPoint(this)) {
                _parentTable.ByPassDragDrop(drgevent);
                return; //Drop\ɂȂĂȂ΃_
            }

            try {
                TabBarUpdateState state = TabBarUpdateState.BeginUpdate("drop");
                DragAndDropSupport.DropResult r = _dragAndDrop.Drop();
                state.Commit();
                if(r==DragAndDropSupport.DropResult.Ignored) {
                    Invalidate();
                }
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
            }
		}

		protected override void OnQueryContinueDrag(QueryContinueDragEventArgs qcdevent) {
			base.OnQueryContinueDrag (qcdevent);
            //Debug.WriteLine("QCD " + qcdevent.Action.ToString() + MousePosition.ToString());
			if(qcdevent.EscapePressed) {
				_dragAndDrop.ClearDropPoint();
			}
		}

        //index̐UȂFadd/removenȂǂŎs
        internal void AssignmentIndex() {
            for(int i=0; i<_buttons.Count; i++) {
                TabBarButton b = _buttons[i];
                b.Index = i;
            }
        }

		//Drag&Drop̌ʂ܂̓oX̌ʂɊÂăATCg蒼
		internal void AssignDocuments(TabKey[] keys, int offset, int length) {
			while(_buttons.Count > length) { //]vȂ̂΍폜
				TabBarButton t = _buttons[_buttons.Count-1];
				this.Controls.Remove(t);
				_buttons.RemoveAt(_buttons.Count-1);
			}

			//ǉ
			for(int i=0; i<_buttons.Count; i++) {
				TabBarButton b = _buttons[i];
				b.Init(this, keys[offset+i], offset+i, i);
			}

			for(int i=_buttons.Count; i<length; i++) { //sΒǉ
				TabBarButton b = CreateButton(keys[offset+i], offset+i, i);
				_buttons.Add(b);
				this.Controls.Add(b);
			}

			TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(this);

		}
		internal void AssignDocuments(TabKey[] docs) {
			AssignDocuments(docs, 0, docs.Length);
		}

		private void DrawDropPointEffect(Graphics g, int x, int y) {
			//
			int height = UNITHEIGHT-4;
			g.DrawLine(SystemPens.ControlText, x-3, y, x+4, y);
			g.DrawLine(SystemPens.ControlText, x-3, y+height, x+4, y+height);

			g.DrawLine(SystemPens.ControlText, x  , y, x  , y+height);
			g.DrawLine(SystemPens.ControlDark, x+1, y, x+1, y+height);
		}

	}


	internal class TabBarButton : UserControl {
		//̑傫ɕׂ
		public class WidthComparer : IComparer<TabBarButton> {
            public int Compare(TabBarButton x, TabBarButton y) {
				return y._sufficientWidth - x._sufficientWidth;
			}
		}

		private const int TEXT_MARGIN = 2;

		//Container
		private TabBar _parent;

		private TabKey _tabKey;
		private int _index;
		private string _indexText;
        private int _indexInBar;
		private Image _image;
		private bool _selected;
		private bool _mouseDown;
		private bool _mouseEnter;

		private int _sufficientWidth; //{^̑SeLXg\̂ɏ\ȕ

		//vZɃZbg
		private bool _showIndexText;
		private bool _showOmittedMark;
		private SizeF _partialTextSize; //\Ƃ͂̈ʒu "..."o

		private int _temporaryWidth;

		public TabBarButton(TabBar parent, TabKey key, int index, int indexInBar) {
			Init(parent, key, index, indexInBar);
		}
		public void Init(TabBar parent, TabKey key, int index, int indexInBar) {
			SetStyle(ControlStyles.AllPaintingInWmPaint|ControlStyles.UserPaint|ControlStyles.DoubleBuffer, true);
			//č쐬邱Ƃ͂
			_parent = parent;
			_tabKey = key;
			_index = index;
            _indexInBar = indexInBar;
			_image = key.Icon;
            
			_indexText = (index+1).ToString();
		}


		public TabBar ParentTabBar {
			get {
				return _parent;
			}
		}
		public TabKey TabKey {
			get {
				return _tabKey;
			}
		}
		public int Index {
			get {
				return _index;
			}
            set {
                _index = value;
                _indexText = (_index+1).ToString();
            }
		}
        public int IndexInBar {
            get {
                return _indexInBar;
            }
            set {
                _indexInBar = value;
            }
        }
		public int SufficientWidth {
			get {
				return _sufficientWidth;
			}
		}
		public bool Selected {
			get {
				return _selected;
			}
		}
		public Image Image {
			get {
				return _image;
			}
		}
		public int TemporaryWidth {
			get {
				return _temporaryWidth;
			}
			set {
				_temporaryWidth = value;
			}
		}

		//TabBar炵Ăł͂
		public void SetSelectedInternal(bool value) {
			_selected = value;
			this.BackColor = (value? _parent.ActiveDrawing : _parent.NormalDrawing).BackgroundColor;
		}
		public void ClearMouseTrackingFlags() {
			_mouseDown = false;
			_mouseEnter = false;
		}

		public int CalcSufficientWidths() {
			using(Graphics g = this.CreateGraphics()) {
				return CalcSufficientWidth(g);
			}
		}

        public void UpdateDescription() {
            _image = _tabKey.Icon;
            TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(_parent);
        }

		internal int CalcSufficientWidth(Graphics g) {
			int img_width = _image==null? 0 : _image.Width;
			TabBarDrawing d = GetDrawing();
			float index_width = g.MeasureString(_indexText, d.Font).Width;
			float text_width  = g.MeasureString(_tabKey.Caption,  d.Font).Width;

			_sufficientWidth = img_width + (int)(index_width + text_width) + TEXT_MARGIN*4; //AEAimg-index, index-bodyłS
			if(_selected) _sufficientWidth += 2; //I͍L߂悳
			return _sufficientWidth;
		}

		//SwidthɎ܂悤ɍlăeLXgݒ肷
		internal void AdjustTextAndWidth(Graphics g, int width) {
			int minimum = (_image==null? 0 : _image.Width+TEXT_MARGIN) - TEXT_MARGIN*2 - 2; //[̃}[Wƃ{[_[
			int remaining = width - minimum;

			TabBarDrawing drawing = GetDrawing();

			int index_text_width = (int)g.MeasureString(_indexText, drawing.Font).Width;
			if(index_text_width*3 <= remaining) { //index_text3荞ނ悤
				remaining -= index_text_width + TEXT_MARGIN;
				_showIndexText = true;
			}
			else
				_showIndexText = false;
	
			//{
			string full_text = _tabKey.Caption;
			int full_width = (int)g.MeasureString(full_text, drawing.Font).Width;
			if(full_width <= remaining) { //S\\
				this.Text = full_text;
				_showOmittedMark = false;
			}
			else if(drawing.OmittedMarkWidth <= remaining) {
				this.Text = full_text;
				_showOmittedMark = true;
				//ToDo Ȃ߂̕w肵Ȃ"..."Eɂ߂Ă܂B{KvȂ͂ǂɌ肪悤
				_partialTextSize = g.MeasureString(full_text, drawing.Font, remaining-drawing.OmittedMarkWidth-5);
			}
			else { //؂
				this.Text = "";
				_showOmittedMark = false;
			}

			this.Width = width;
		}

		protected override void OnPaint(PaintEventArgs e) {
			//ToDo ƂőI𐫂ɂ
			Graphics g = e.Graphics;
#if false //Poderosa`
			base.OnPaint(e);
			//border
			if(_selected)
				DrawUtil.DrawRoundRect(g, 0, 0, this.Width-1, this.Height-1, _parent.ActiveDrawing.RoundRectColors);
			else if(_mouseEnter)
				DrawUtil.DrawRoundRect(g, 0, 0, this.Width-1, this.Height-1, _parent.NormalDrawing.RoundRectColors);
#else //FireFox
			if(_selected) {
				base.OnPaint(e); //wih
				DrawUtil.DrawRoundRect(g, 0, 0, this.Width-1, this.Height-1, _parent.ActiveDrawing.RoundRectColors);
				DrawOrangeBar(g);
			}
			else {
				DrawUtil.FillHorizontalGradation(g, 0, 0, this.Width-1, this.Height-1, SystemColors.ControlLightLight, SystemColors.Control);
				//Ƀ{^ɂ邽ߎɕ`
				DrawUtil.DrawRoundRect(g, 0, 0, this.Width-1, this.Height-1, _parent.NormalDrawing.RoundRectColors);
				if(_mouseEnter) DrawOrangeBar(g);
			}
#endif

			DrawButtonInternal(g);	
		}
		private void DrawOrangeBar(Graphics g) {
			//IWLvV
			Brush b = new SolidBrush(Color.Orange);
			g.FillRectangle(b, 1, 0, this.Width-2, 2);
			b.Dispose();
		}


		public void Reset() {
			_mouseDown = false;
			_mouseEnter = false;
			Debug.Assert(!this.InvokeRequired);
			Invalidate();
		}
		protected override void OnMouseEnter(EventArgs e) {
			base.OnMouseEnter(e);
			_mouseEnter = true;
			Invalidate();
		}

		protected override void OnMouseLeave(EventArgs e) {
			base.OnMouseLeave(e);
			_mouseEnter = false;
			Invalidate();
		}

		protected override void OnMouseDown(MouseEventArgs e) {
			base.OnMouseDown(e);
			_mouseDown = true;
			Invalidate();
		}

		protected override void OnMouseUp(MouseEventArgs e) {
			base.OnMouseUp(e);
			_mouseDown = false;
			Invalidate();
		}

		private void DrawButtonInternal(Graphics g) {
			int x, y;
			x = TEXT_MARGIN;
			if(_image!=null) {
                y = (this.Height - _image.Height) / 2;
                DrawImage(g, DrawState.Normal, _image, x, y);
				x += _image.Width + TEXT_MARGIN;
			}

			TabBarDrawing drawing = GetDrawing();

			y = 4; //TODO tHgč
			if(_showIndexText) {
				g.DrawString(_indexText, drawing.Font, SystemBrushes.ControlDark, x, y);
				x += (int)g.MeasureString(_indexText, drawing.Font).Width + TEXT_MARGIN;
			}

			Brush text_brush = _mouseDown? SystemBrushes.ControlLightLight : SystemBrushes.ControlText;
			if(_mouseDown) y++; //傢ɕ`

			if(_showOmittedMark) {
				g.DrawString(_tabKey.Caption, drawing.Font, text_brush, new RectangleF(x, y, _partialTextSize.Width, this.ClientSize.Height-y-2));
				DrawUtil.DrawOmittedMark(g, x + (int)_partialTextSize.Width, this.ClientSize.Height-5, SystemColors.ControlText, _selected);
			}
			else {
				g.DrawString(_tabKey.Caption, drawing.Font, text_brush, new Point(x, y));
			}
		}

		private TabBarDrawing GetDrawing() {
			return _selected? _parent.ActiveDrawing : _parent.NormalDrawing;
		}


		private static void DrawImage(Graphics g, DrawState state, Image image, int x, int y) {
			if (state == DrawState.Normal)
				g.DrawImage(image, x, y, image.Width, image.Height);
			else if (state == DrawState.Selected || state == DrawState.Hover) {
				ControlPaint.DrawImageDisabled(g, image, x+1, y, SystemColors.Control);
				g.DrawImage(image, x-1, y-1, image.Width, image.Height);        
			}
		}
	}

	
	internal class DragAndDropSupport {
		internal class DropPoint {
			private TabBar _tabBar;
			//Ŝ̉E[̂rightƂȂ
			private TabBarButton _target;
			private bool _left; //E̋

			//buttonnull̂ƂB
			public DropPoint(TabBar tabbar, TabBarButton button, bool left) {
				_tabBar = tabbar;
				_target = button;
				_left = left;
			}

			public TabBarButton Target {
				get {
					return _target;
				}
			}
			public bool ForLeft {
				get {
					return _left;
				}
			}
			public bool ForRight {
				get {
					return !_left;
				}
			}
			//Rg[̒[distanceꂽʒuԂ
			public int PosX(int distance) {
				if(_target==null)
					return distance;
				else
					return _left? _target.Left-distance : _target.Right+distance;
			}
			//DropPointwCfbNX
			public int IndexInBar {
				get {
					return _target==null? 0 : (_target.IndexInBar + (_left? 0 : 1));
				}
			}
			public TabBar TabBar {
				get {
					return _tabBar;
				}
			}


			public override bool Equals(object obj) {
				DropPoint p = obj as DropPoint;
				return p!=null && p._target==_target && p._left==_left;
			}
			//RpCG[
			public override int GetHashCode() {
				return base.GetHashCode();
			}


			//nullnull͓AƂvfꂽr
			public static bool Equals(DropPoint p1, DropPoint p2) {
				if(p1==null)
					return p2==null;
				else
					return p1.Equals(p2); //p2nullȂLɂfalseԂ
			}

		}

		//DragAndDropSupport{
		private TabBarButton _draggingButton;
		private DropPoint _dropPoint;

		public void StartDrag(TabBarButton btn) {
			_draggingButton = btn;
            btn.ParentTabBar.ParentTable.OnStartButtonDragByUI(_draggingButton.TabKey);

			//DoDragDropÑRg[ɑ΂QuerynĂ΂炵
			btn.ParentTabBar.DoDragDrop("poderosa.tabkey", DragDropEffects.Move|DragDropEffects.Link); //TODO ̕񂢂
		}
		public bool CanDrop(IDataObject data, TabBar target) {
			string dragging = data.GetData(typeof(string)) as string;
            //TablełhbvłȂ悤ɂBEBhEقȂĂr[փhbvłȂƂ߂Ȃ̂ŁB
			return "poderosa.tabkey"==dragging && _draggingButton.ParentTabBar.ParentTable==target.ParentTable;
		}
		public DropPoint CurrentDropPoint {
			get {
				return _dropPoint;
			}
		}
		public bool OwnsDropPoint(TabBar tabbar) {
			if(_dropPoint==null)
				return false;
			else
				return _dropPoint.TabBar==tabbar;
		}
		public void SetDropPoint(DropPoint point) {
			if(!DropPoint.Equals(_dropPoint, point)) { //null̂Ƃ̂Œ
				if(_dropPoint!=null) _dropPoint.TabBar.Invalidate();
				_dropPoint = point;
				if(point!=null) point.TabBar.Invalidate();
			}
		}
		public void ClearDropPoint() {
			if(_dropPoint!=null) {
				_dropPoint.TabBar.Invalidate();
				_dropPoint = null;
			}
		}

        /// <summary>
        /// 
        /// </summary>
        /// <exclude/>
		public enum DropResult {
			Ignored,
			Reordered,
			Moved
		}

		public DropResult Drop() {
			DropPoint point = _dropPoint;
			Debug.Assert(point!=null);
			_dropPoint = null; //PointƂnullɖ߂

			if(_draggingButton.ParentTabBar==point.TabBar) { //o[łDrop
				return DropToSameTabBar(point);
			}
			else { //قȂo[ւ̈ړ
				return DropToDifferentTabBar(point);
			}
		}

		private DropResult DropToSameTabBar(DropPoint point) {
			TabKey dragging_key = _draggingButton.TabKey;
			TabBar tabbar = _draggingButton.ParentTabBar;
			int ti = point.Target.IndexInBar;
			int di = _draggingButton.IndexInBar;
			int newindex = point.ForLeft? ti : ti+1;
			if(newindex==di || newindex==di+1) return DropResult.Ignored;

			//ւԂdocs
			TabKey[] keys = new TabKey[tabbar.TabCount];
			int cursor = 0;
			for(int i=0; i<=keys.Length; i++) { //E[ւDrop܂ōliCount܂ł܂킷
				if(i==di) continue; //skip

				if(i==newindex)
					keys[cursor++] = dragging_key;
				if(i<keys.Length)
					keys[cursor++] = tabbar.ButtonAt(i).TabKey;
			}
			Debug.Assert(cursor==keys.Length);

			tabbar.AssignDocuments(keys); //č\z
			tabbar.ParentTable.Activate(dragging_key, true);
			return DropResult.Reordered;
		}
		private DropResult DropToDifferentTabBar(DropPoint point) {
			TabKey dragging_key = _draggingButton.TabKey;
			//ړ̍폜
			TabBar bar = _draggingButton.ParentTabBar;
			TabKey[] keys_src = new TabKey[bar.TabCount-1];
			int cursor = 0;
			for(int i=0; i<bar.TabCount; i++) {
				TabBarButton b = bar.ButtonAt(i);
				if(b!=_draggingButton) keys_src[cursor++] = b.TabKey;
			}
			Debug.Assert(cursor==keys_src.Length);
			bar.ParentTable.Deactivate(true);
			bar.AssignDocuments(keys_src);
            TabBarUpdateState.GetCurrent().MarkIndexAssignmentChanged(bar);

			//ړւ̒ǉ
			bar = point.TabBar;
			TabKey[] docs_dst = new TabKey[bar.TabCount+1];
			cursor = 0;
			int newindex = point.IndexInBar;
			for(int i=0; i<=bar.TabCount; i++) {
				if(i==newindex) docs_dst[cursor++] = dragging_key;
				if(i<bar.TabCount)
					docs_dst[cursor++] = bar.ButtonAt(i).TabKey;
			}
			Debug.Assert(cursor==docs_dst.Length);
			bar.AssignDocuments(docs_dst);

			bar.ParentTable.Activate(dragging_key, true);

            TabBarUpdateState.GetCurrent().MarkIndexAssignmentChanged(bar);
			return DropResult.Moved;
		}
	}

	//i\ɂƂɎgATabBar̃RNVBActiveȂ̂TableňɂȂ
    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
	public class TabBarTable : UserControl {
        public interface IUIHandler {
            void ActivateTab(TabKey key);
            void MouseMiddleButton(TabKey key);
            void MouseRightButton(TabKey key);
            void StartTabDrag(TabKey key);
            void AllocateTabToControl(TabKey key, Control target);
            void BypassDragEnter(DragEventArgs args);
            void BypassDragDrop(DragEventArgs args);
        }
        private IUIHandler _uiHandler;

        /*
        public delegate void ButtonActionHandler(TabKey key);
        public delegate void ButtonDragByUIHandler(TabKey key);
        public delegate void ButtonActionAtBarHandler();
        public delegate void SideButtonClickHandler(Point pt);
        public delegate void ButtonAllocationToControlHandler(TabKey key, Control target);
        */
		private TabBarButton _activeButton; //nullɂȂ邱Ƃ
		private List<TabBar> _bars;
        private CaptureStyleReplaceState _captureState;
        /*
        private ButtonActionHandler _activateByUIHandler;
        private ButtonActionHandler _mouseMiddleButtonHandler;
        private ButtonActionHandler _mouseRightButtonHandler;
        private ButtonActionAtBarHandler _mouseRightButtonAtBarHandler;
        private ButtonDragByUIHandler _buttonDragByUIHandler;
        private SideButtonClickHandler _sideButtonHandler;
        private ButtonAllocationToControlHandler _buttonAllocationToControlHandler;
        */

		public const int ROW_HEIGHT = 26;

		public TabBarTable() {
			_bars = new List<TabBar>();
            _captureState = new CaptureStyleReplaceState(this);
            this.Enabled = true;
		}
        public TabKey ActiveTabKey {
            get {
                return _activeButton==null? null : _activeButton.TabKey;
            }
        }
        public IUIHandler UIHandler {
            get {
                return _uiHandler;
            }
            set {
                _uiHandler = value;
            }
        }

        internal CaptureStyleReplaceState CaptureState {
            get {
                return _captureState;
            }
        }

        //indexx[X̏
        public TabKey GetAtOrNull(int index) {
            foreach(TabBar bar in _bars) {
                if(index < bar.TabCount) return bar.ButtonAt(index).TabKey;
                index -= bar.TabCount;
            }
            return null;
        }
        public int TabCount {
            get {
                int c = 0;
                foreach(TabBar bar in _bars) {
                    c += bar.TabCount;
                }
                return c;
            }
        }
        public int IndexOf(TabKey key) {
            int count = 0;
            foreach(TabBar bar in _bars) {
                foreach(TabBarButton button in bar.Buttons) {
                    if(button.TabKey==key) return count + button.IndexInBar;
                }
                count += bar.TabCount;
            }
            return -1;
        }

        public void Create(int rowcount) {
            SetTabBarCount(rowcount);
		}
        //͏dɂȂ肩˂Ȃ
        public void SetTabBarCount(int count) {
            if(count==_bars.Count) return;

            TabBarUpdateState state = TabBarUpdateState.BeginUpdate("set-tabbarcount");
            if(_bars.Count < count) { //
                for(int i = _bars.Count; i<count; i++) {
                    TabBar bar = new TabBar(this);
                    bar.Location = new System.Drawing.Point(0, ROW_HEIGHT*i);
                    bar.Height = ROW_HEIGHT;
                    bar.Width = this.Width;
                    bar.Anchor = AnchorStyles.Left|AnchorStyles.Right;
                    this.Controls.Add(bar);
                    _bars.Add(bar);
                }
            }
            else if(count < _bars.Count) { //
                TabKey[] docs = GetAllDocuments();
                while(_bars.Count>count) {
                    TabBar bar = _bars[_bars.Count-1];
                    _bars.Remove(bar);
                    this.Controls.Remove(bar);
                }
                Rebalance(docs);
            }

            this.Height = ROW_HEIGHT * count;
            for(int i=0; i<_bars.Count; i++)
                _bars[i].Top = ROW_HEIGHT * i; //this.HeightύX炱ĒĂƂ悤
            state.Commit();
        }

        public int TabBarCount {
            get {
                return _bars.Count;
            }
        }
        public int GetAllTabCount() {
            int t =0;
            foreach(TabBar bar in _bars) t += bar.TabCount;
            return t;
        }

        public bool ContainsKey(TabKey key) {
            return FindButton(key)!=null;
        }

        public void Activate(TabKey key, bool fire_external_event) {
			ActivateInternal(FindButton(key));

            if(fire_external_event)
                _uiHandler.ActivateTab(key);
        }
        public void DoMiddleButtonAction(TabKey key) {
            _uiHandler.MouseMiddleButton(key);
        }
        public void DoRightButtonAction(TabKey key) {
            _uiHandler.MouseRightButton(key);
        }
       
		private void ActivateInternal(TabBarButton button) {
			Debug.Assert(button.ParentTabBar.ParentTable==this);
			Debug.Assert(button!=null);

			if(_activeButton!=null) {
				_activeButton.SetSelectedInternal(false);
				TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(_activeButton.ParentTabBar);
			}

			if(button==null)
				_activeButton = null;
			else {
				_activeButton = button;
				_activeButton.SetSelectedInternal(true);
				TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(button.ParentTabBar);
			}

		}


		public void Deactivate(bool fire_external_event) {
			if(_activeButton!=null) {
				_activeButton.SetSelectedInternal(false);
				TabBarUpdateState.GetCurrent().MarkSufficientWidthIsChanged(_activeButton.ParentTabBar);
				_activeButton = null;
			}

		}

        //[UC^tF[XɂActivate
        public void OnActivatedByUI(TabKey key) {
            _uiHandler.ActivateTab(key);
        }
        public void OnMouseMiddleButton(TabKey key) {
            DoMiddleButtonAction(key);
        }
        public void OnMouseRightButton(TabKey key) {
            DoRightButtonAction(key);
        }
        //[UC^tF[XɂDragAndDrop
        public void OnStartButtonDragByUI(TabKey key) {
            _uiHandler.StartTabDrag(key);
        }

		public void AddTab(TabKey key) {
			TabBar bar = FindFirstFreeTabBar();
            if(bar!=null) {
                //indexvZʓ|
                int tabcount = 0;
                int barindex = 0;
                for(int i=0; i<_bars.Count; i++) {
                    tabcount += _bars[i].TabCount;
                    if(_bars[i]==bar) {
                        barindex = i;
                        break;
                    }
                }

                bar.AddTab(key, tabcount++, bar.TabCount); //̖Ȃ̂
                for(int i=barindex+1; i<_bars.Count; i++) {
                    _bars[i].AllocateIndex(tabcount);
                    tabcount += _bars[i].TabCount;
                }
            }
            else {
                bar = _bars[_bars.Count-1]; //dȂŌg
                bar.AddTab(key, GetAllTabCount(), bar.TabCount);
                Rebalance(GetAllDocuments());
            }
		}

        public void RemoveTab(TabKey key, bool fire_external_event) {
            if(_activeButton!=null && _activeButton.TabKey==key) Deactivate(fire_external_event);

            foreach(TabBar bar in _bars) {
                TabBarButton b = bar.FindButton(key);
                if(b!=null) {
                    bar.RemoveTab(key);
                    break;
                }
            }
        }

        public void UpdateDescription(TabKey key) {
            foreach(TabBar bar in _bars) {
                TabBarButton b = bar.FindButton(key);
                if(b!=null) {
                    bar.FindButton(key).UpdateDescription();
                    break;
                }
            }
        }

        public void AssignIndex() {
            int t = 0;
            foreach(TabBar bar in _bars) {
                bar.AllocateIndex(t);
                t += bar.TabCount;
            }
        }

		public void Rebalance(TabKey[] keys) {
			//^ǔɉċϓɂȂ悤ɃoX
			int count = keys.Length/_bars.Count;
			int mod   = keys.Length%_bars.Count;
			int index = 0;

			for(int i=0; i<_bars.Count; i++) {
				TabBar bar = _bars[i];
				int length = i<mod? count+1 : count; //]l
				bar.AssignDocuments(keys, index, length);
				index += length;
			}
		}

        public TabKey[] GetAllDocuments() {
            List<TabKey> r = new List<TabKey>();
            foreach(TabBar bar in _bars) {
                foreach(TabBarButton btn in bar.Buttons)
                    r.Add(btn.TabKey);
            }
            return r.ToArray();
        }

		private TabBar FindFirstFreeTabBar() {
			foreach(TabBar bar in _bars) {
				if(bar.LastAdjustmentResult==TabBar.AdjustEachWidthResult.Sufficient || bar.TabCount==0)
					return bar;
			}
			return null;
		}
		private TabBarButton FindButton(TabKey key) {
			foreach(TabBar bar in _bars) {
				TabBarButton b = bar.FindButton(key);
				if(b!=null) return b;
			}
			return null;
		}

        //q̃{^ŃnhłȂDragDrop
        public void ByPassDragEnter(DragEventArgs args) {
            _uiHandler.BypassDragEnter(args);
        }
        public void ByPassDragDrop(DragEventArgs args) {
            _uiHandler.BypassDragDrop(args);
        }

        /*
        protected override void OnPaint(PaintEventArgs e) {
            base.OnPaint(e);
            Rectangle sidebutton_rect = new Rectangle(this.Width-24, this.Height-24, 24, 24);
            if(e.ClipRectangle.IntersectsWith(sidebutton_rect)) {
                VisualStyleRenderer renderer = new VisualStyleRenderer(VisualStyleElement.Rebar.Chevron.Hot);
                renderer.DrawBackground(e.Graphics, sidebutton_rect);
            }
        }
         */
    }

	//KvȍXVǗBTabBar#ArrangeButtonߏɌĂ΂Ȃ悤ɂ邽߂̎dg
    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
	public class TabBarUpdateState {
		private enum UpdateState {
			NeedToDoNothing,
			AdjustmentIsRequired,
			SufficientWidthIsChanged //̏Ԃ̕
		}

		private class Entry {
			public TabBar tabBar;
			public UpdateState state;
			public Entry(TabBar bar) {
				tabBar = bar;
			}
		}

		private List<Entry> _entries;
        private bool _indexAssignmentChanged;

		public TabBarUpdateState() {
			_entries = new List<Entry>();
		}
		public void MarkAdjustmentRequired(TabBar bar) {
			Entry e = FindOrCreate(bar);
			if(e.state!=UpdateState.SufficientWidthIsChanged)
				e.state = UpdateState.AdjustmentIsRequired;
		}
		public void MarkSufficientWidthIsChanged(TabBar bar) {
			Entry e = FindOrCreate(bar);
			e.state = UpdateState.SufficientWidthIsChanged;
		}
        public void MarkIndexAssignmentChanged(TabBar bar) {
            Entry e = FindOrCreate(bar);
            e.state = UpdateState.SufficientWidthIsChanged; //lςΕKvȕω
            _indexAssignmentChanged = true; //SBARɉe
        }

		//ArrangeButtonsĂׂ̂͂
		public void Commit() {

			foreach(Entry e in _entries) {
                if(_indexAssignmentChanged) {
                    e.tabBar.ParentTable.AssignIndex(); //So[œ
                    _indexAssignmentChanged = false;
                }
                
                if(e.state==UpdateState.SufficientWidthIsChanged) {
					e.tabBar.CalcSuffientWidths();
					e.tabBar.ArrangeButtons();
				}
				else if(e.state==UpdateState.AdjustmentIsRequired) {
					e.tabBar.ArrangeButtons();
				}
			}


            _indexAssignmentChanged = false;
			_entries.Clear();
			_activeFlag = false;
		}

		private Entry FindOrCreate(TabBar bar) {
			foreach(Entry e in _entries) {
				if(e.tabBar==bar) return e;
			}
			Entry newentry = new Entry(bar);
			_entries.Add(newentry);
			return newentry;
		}

		//staticł̂͂Ƌ^₾
		private static bool _activeFlag;
		private static TabBarUpdateState _instance;
		private static string _lastAction;

		public static TabBarUpdateState BeginUpdate(string action) { //͖{Debugɂ̂ݕKv
			Debug.Assert(!_activeFlag, "nested action: last="+_lastAction);
			_activeFlag = true;
			_lastAction = action;
			if(_instance==null) _instance = new TabBarUpdateState();
			return _instance;
		}

		public static void AssertInactive() {
			Debug.Assert(!_activeFlag);
		}

		//TabBarXVƂƂ͂Ă
		public static TabBarUpdateState GetCurrent() {
			Debug.Assert(_activeFlag);
			return _instance;
		}

        public static bool IsActive {
            get {
                return _activeFlag;
            }
        }
	}

    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
    internal class CaptureStyleReplaceState {
        //}EXLv`^Doc-View蓖Ă̏ԊǗ
        private TabKey _key; //蓖Ē̃ucBnullȂ疢gp
        private TabBarTable _table;

        public CaptureStyleReplaceState(TabBarTable table) {
            _table = table;
        }


        public bool IsCapturing {
            get {
                return _key!=null;
            }
        }
        public void StartCapture(TabBar tabbar, TabKey key) {
            _key = key;
            tabbar.Capture = true;
            Cursor.Current = Cursors.Cross;
        }
        public void CancelCapture() {
            _key = null;
        }
        public void EndCapture(Point screen_pt) {
            if(_key!=null) {
                Form f = _table.FindForm();
                Control c = WinFormsUtil.FindTopControl(f, screen_pt); //̃EBhE֎ĂƂłĂȂ
                if(c!=null) {
                    _table.UIHandler.AllocateTabToControl(_key, c);
                }
            }
            _key = null;
        }
    }
}