/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/StockRef.cs#12 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * IuWFNg
 */
using System;
using System.IO;
using System.Collections;
using System.Diagnostics;
using System.Text;
using System.Reflection;
using Poderosa.Util.Collections;
using System.Collections.Generic;

using Bellagio.Data;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Values
{
    public class StockRef : BV {
        private Stock _stock;

        public StockRef() {
            _stock = null;
        }
        public StockRef(Stock stock) {
            _stock = stock;
        }

        public override bool IsNil {
            get {
                return _stock == null;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.Stock(this);
        }
        public override void Let(BV value) {
            _stock = ((StockRef)value)._stock;
        }
        public override BT BT {
            get { return StockType.instance; }
        }

        public Stock Stock {
            get {
                return _stock;
            }
            set {
                _stock = value;
            }
        }
    }

    public class StockType : BObjectT {
        private StockType()
            : base("Stock", new StockRef()) {
        }
        public override BV CreateInstance() {
            return new StockRef();
        }
        public static StockType instance;

        private static StockProperties _stockProperties;

        static StockType() {
            instance =  new StockType();
            _stockProperties = new StockProperties();
        }

        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT isStock = instance;
            BT[] noArg = BuiltInFunctionLibrary.noArg;
            lib.DefineInternalFunction("code", isStock, BT.String, noArg, new BInternalExecution(GetCode));
            lib.DefineInternalFunction("name", isStock, BT.String, noArg, new BInternalExecution(GetName));
            lib.DefineInternalFunction("market", isStock, BT.Bool, new BT[] { BT.String }, new BInternalExecution(GetMarket));
            lib.DefineInternalFunction("unit", isStock, BT.Int, noArg, new BInternalExecution(GetVolumeUnit));
            lib.DefineInternalFunction("taisyaku", isStock, BT.Bool, noArg, new BInternalExecution(GetTaisyaku));
            lib.DefineInternalFunction("yobine", null, BT.Int, new BT[] { BT.Int }, new BInternalExecution(GetYobine));
        }
        private static bool IsStock(BT target) {
            return target==instance;
        }
        private static ExecResult GetCode(BV target, EvalContext extra, BV[] args, BV result) {
            StockRef stock = (StockRef)target;
            if(stock.IsNil)
                return ExecResult.Nil;
            else
                ((BString)result).Value = stock.Stock.Profile.Code;
            return ExecResult.OK;
        }
        private static ExecResult GetName(BV target, EvalContext extra, BV[] args, BV result) {
            StockRef stock = (StockRef)target;
            if(stock.IsNil)
                return ExecResult.Nil;
            else
                ((BString)result).Value = stock.Stock.Profile.Name;
            return ExecResult.OK;
        }
        private static ExecResult GetMarket(BV target, EvalContext extra, BV[] args, BV result) {
            StockRef stock = (StockRef)target;
            BString t = (BString)args[0];
            if(stock.IsNil || t.IsNil)
                return ExecResult.Nil;
            else 
                ((BBoolean)result).Value = _stockProperties.CheckMarket(stock.Stock, t.Value);
            return ExecResult.OK;
        }
        private static ExecResult GetVolumeUnit(BV target, EvalContext extra, BV[] args, BV result) {
            StockRef stock = (StockRef)target;
            if(stock.IsNil)
                return ExecResult.Nil;
            else
                ((BInt)result).Value = stock.Stock.VolumeUnit;
            return ExecResult.OK;
        }
        private static ExecResult GetTaisyaku(BV target, EvalContext extra, BV[] args, BV result) {
            StockRef stock = (StockRef)target;
            if(stock.IsNil)
                return ExecResult.Nil;
            else
                ((BBoolean)result).Value = (stock.Stock.StockFlags & StockFlags.Taisyaku)!=StockFlags.None;
            return ExecResult.OK;
        }
        private static ExecResult GetYobine(BV target, EvalContext extra, BV[] args, BV result) {
            BInt bi = args[0] as BInt;
            if(bi.IsNil)
                return ExecResult.Nil;
            else
                ((BInt)result).Value = MarketUtil.YobineIncrement(bi.Value) - bi.Value;
            return ExecResult.OK;
        }
    }

    //̕ނ𔻒肷@\
    public class StockProperties {
        public delegate bool Check(Stock s);
        //{
        private Dictionary<string, Check> _primitives;
        //Rrl[V '|'łȂꂽŃp[X̂LbV
        private Dictionary<string, Check[]> _cache;

        public StockProperties() {
            _primitives = new Dictionary<string, Check>();
            _primitives.Add("T", delegate(Stock s) { return s.Market==StockExchange.T;  });
            _primitives.Add("T1", delegate(Stock s) { return s.Market==StockExchange.T && s.MarketSubType==StockExchangeSubType.Ichibu; });
            _primitives.Add("T2", delegate(Stock s) { return s.Market==StockExchange.T && s.MarketSubType==StockExchangeSubType.Nibu; });
            _primitives.Add("TM", delegate(Stock s) { return s.Market==StockExchange.T && s.MarketSubType==StockExchangeSubType.Mothers; });
            _primitives.Add("O", delegate(Stock s) { return s.Market==StockExchange.O; });
            _primitives.Add("O1", delegate(Stock s) { return s.Market==StockExchange.O && s.MarketSubType==StockExchangeSubType.Ichibu; });
            _primitives.Add("O2", delegate(Stock s) { return s.Market==StockExchange.O && s.MarketSubType==StockExchangeSubType.Nibu; });
            _primitives.Add("J", delegate(Stock s) { return s.Market==StockExchange.J; });
            _primitives.Add("N", delegate(Stock s) { return s.Market==StockExchange.Nagoya; });
            _primitives.Add("F", delegate(Stock s) { return s.Market==StockExchange.Fukuoka; });
            _primitives.Add("S", delegate(Stock s) { return s.Market==StockExchange.Sapporo; });
            _primitives.Add("H", delegate(Stock s) { return s.Market==StockExchange.Hercules; });
            _primitives.Add("ETF", delegate(Stock s) { return s.MarketSubType==StockExchangeSubType.ETF; });
            _primitives.Add("REIT", delegate(Stock s) { return s.MarketSubType==StockExchangeSubType.REIT; });
            _primitives.Add("EM", delegate(Stock s) { return s.MarketSubType==StockExchangeSubType.Mothers || s.Market==StockExchange.Hercules || s.MarketSubType==StockExchangeSubType.Ambitious || s.MarketSubType==StockExchangeSubType.Centrex || s.MarketSubType==StockExchangeSubType.QBoard; });

            _cache = new Dictionary<string, Check[]>();
        }

        public bool CheckMarket(Stock stock, string cond) {
            Check[] parsed = GetParsedCondition(cond);
            for(int i=0; i<parsed.Length; i++)
                if(parsed[i](stock)) return true;
            return false;
        }

        private Check[] GetParsedCondition(string cond) {
            Check[] r;
            if(!_cache.TryGetValue(cond, out r)) {

                List<Check> l = new List<Check>();
                foreach(string e in cond.Split('|')) {
                    //TODO p[XG[ʒm
                    l.Add(_primitives[e]);
                }
                r = l.ToArray();
                _cache.Add(cond, r);
            }
            return r;
        }
    }
}
