/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Ordering/OrderDataProvider.cs#11 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Poderosa;
using Travis.Collections;

using Bellagio.Environment;
using Bellagio.Data;

namespace Bellagio.Ordering {

    //|WVƉAƉ̃f[^܂Ƃ߂
    //E񓯊ǂ̃p^[łf[^Ԃ邪Ãf[^ɑ΂񓯊NGXg邱ƂȂ悤ɂubNOrderDataProviderōs
    public interface IOrderDataSource : IPrimaryStreamDataBase {
        IOrderItemFormatter OrderItemFormatter { get; }

        //ƉńA񓯊M̃p^[BʂIOrderDataHostɒʒm
        AsyncOpResult GetStockPositionList(ref string message);
        AsyncOpResult GetStockOrderList(ref string message);
        AsyncOpResult GetDerivativePositionList(ref string message);
        AsyncOpResult GetDerivativeOrderList(ref string message);
        AsyncOpResult GetStockPowerInfo(ref string message);
        AsyncOpResult GetDerivativePowerInfo(ref string message);
        AsyncOpResult GetStockSnapshot(AbstractStockProfile prof, ref string message);
        AsyncOpResult GetDerivativeSnapshot(DerivativeStockProfile prof, ref string message);

        //P̒Ɖ
        AsyncOpResult GetOrderDetailInfo(string orderID, OrderStatusCodeCategory category, out TradeOrderItem result);

        //ńA󗝂ǂ͂̏ŊmF̂Ōʔ܂ŃubNBs͗OB
        //ʂƂĂǂɂȂo͈Ŏ󂯎BꂪsvȃP[XA񓯊Ŏ󂯂P[XA邩Ȃڐۗ
        void SendNewOrCloseOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage);
        void SendChangeOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage);
        void SendCancelOrder(TradeOrderItem item, string pincode, out TradeOrderItem result, out string successfulMessage);
        void SendSpecifiedElementCloseOrder(TradeOrderItem item, PositionAndVolumeList list, string pincode, out TradeOrderItem result, out string successfulMessage);
    }
    public interface IOrderDataHost : IPrimaryStreamDataHostBase {
        void SetStockOrderList(DataResponseCode dt, TradeOrderList data, string message);
        void SetStockPositionList(DataResponseCode dt, TradePositionList data, string message);
        void SetDerivativeOrderList(DataResponseCode dt, TradeOrderList data, string message);
        void SetDerivativePositionList(DataResponseCode dt, TradePositionList data, string message);
        void SetStockPowerInfo(DataResponseCode dt, StockBuyingPowerInfo data, string message);
        void SetDerivativePowerInfo(DataResponseCode dt, DerivativeBuyingPowerInfo data, string message);
        void SetStockSnapshot(DataResponseCode dt, OrderDataStockSnapshot data, string message);
        void SetDerivativeSnapshot(DataResponseCode dt, OrderDataStockSnapshot data, string message);
    }

    //ñeXg邽߂̃C^tF[X
    public interface IOrderDataTestUtil : IAdaptable {
        //XML̕ϊ
        TradePositionList TextToStockPositionList(string text);
        TradeOrderList TextToStockOrderList(string text);
    }

    //ʎw̏ڍ
    public class PositionAndVolumeList {
        private List<Pair<string, int>> _data;
        public PositionAndVolumeList() {
            _data = new List<Pair<string,int>>();
        }
        public void Add(string key, int vol) {
            _data.Add(new Pair<string,int>(key, vol));
        }
        public int Count {
            get {
                return _data.Count;
            }
        }
        public string KeyAt(int index) {
            return _data[index].first;
        }
        public int VolumeAt(int index) {
            return _data[index].second;
        }
        public int Find(string key) {
            foreach(Pair<string, int> e in _data) {
                if(e.first==key) return e.second;
            }
            return 0;
        }
        public PositionAndVolumeList Clone() {
            PositionAndVolumeList r = new PositionAndVolumeList();
            foreach(Pair<string, int> e in _data)
                r._data.Add(new Pair<string, int>(e.first, e.second));
            return r;
        }
    }

    //sO
    public class IllegalOrderException : Exception {
        public IllegalOrderException(string msg)
            : base(msg) {
        }
    }


    //񓯊MIuWFNg̃zXeBOɁAʎ擾Jn܂߂
    public class OrderDataElement<T> : AsyncLoadingDataSourceObjectHostT<T> where T:class {
        public delegate AsyncOpResult RefreshDelegate(ref string message);
       
        private OrderDataProvider _parent;
        private RefreshDelegate _refreshDelegate;

        public OrderDataElement(OrderDataProvider parent, RefreshDelegate refresh) {
            _parent = parent;
            _refreshDelegate = refresh;
        }

        public void Refresh() {
            try {
                _parent.EnsureDataSource();

                if(_status==BellagioDocumentStatus.Loading) return; //񓯊NGXg̏dh~

                string message = null;
                AsyncOpResult result = _refreshDelegate(ref message);
                if(result==AsyncOpResult.Async) //Ă͂
                    this.InternalSetValueAndStatus(null, BellagioDocumentStatus.Loading);
                else if(result==AsyncOpResult.Failed)
                    BUtil.ShowWarningMessageBox(message);
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        public void SetResponse(DataResponseCode dt, T data, string message) {
            OrderingUtil.AssertCorrectDataResponseCode(dt);

            DataResponseCode result = dt & DataResponseCode.ResultMask;
            if(result==DataResponseCode.Failed) {
                this.InternalSetValueAndStatus(null, BellagioDocumentStatus.Empty);
                BUtil.ShowWarningMessageBox(message);
                BellagioRoot.ExitApplication();
            }
            else {
                DataResponseCode type = dt & DataResponseCode.SyncTypeMask;
                if(type==DataResponseCode.AsyncResponse) {
                    Debug.Assert(_status==BellagioDocumentStatus.Loading);
                }
                this.InternalSetValueAndStatus(data, BellagioDocumentStatus.Complete);
            }
        }
    }


    public class OrderDataProvider : IOrderDataHost {
        private IOrderDataSource _dataSource;
        private DefaultOrderItemFormatter _defaultFormatter;

        private TradeOrderList _reservedStockOrderList; //ۗ
        private OrderDataElement<TradeOrderList> _stockOrderList;
        private OrderDataElement<TradePositionList> _stockPositionList;
        
        private TradeOrderList _reservedDerivativeOrderList; //ۗ
        private OrderDataElement<TradeOrderList> _derivativeOrderList;
        private OrderDataElement<TradePositionList> _derivativePositionList;

        private OrderDataElement<StockBuyingPowerInfo> _stockPowerInfo;
        private OrderDataElement<DerivativeBuyingPowerInfo> _derivativePowerInfo;

        private AccountInfo _accountInfo;

        //͖ƂɂȂ̂OrderDataElementgȂ
        public delegate void StockSnapshotDelegate(AsyncOpResult result, OrderDataStockSnapshot snapshot);

        public OrderDataProvider() {
            _defaultFormatter = new DefaultOrderItemFormatter();
            _reservedStockOrderList = new TradeOrderList();
            _stockOrderList = new OrderDataElement<TradeOrderList>(this, delegate(ref string m) { return _dataSource.GetStockOrderList(ref m); });
            _stockPositionList = new OrderDataElement<TradePositionList>(this, delegate(ref string m) { return _dataSource.GetStockPositionList(ref m); });
            
            _reservedDerivativeOrderList = new TradeOrderList();
            _derivativeOrderList = new OrderDataElement<TradeOrderList>(this, delegate(ref string m) { return _dataSource.GetDerivativeOrderList(ref m); });
            _derivativePositionList = new OrderDataElement<TradePositionList>(this, delegate(ref string m) { return _dataSource.GetDerivativePositionList(ref m); });

            _stockPowerInfo = new OrderDataElement<StockBuyingPowerInfo>(this, delegate(ref string m) { return _dataSource.GetStockPowerInfo(ref m); });
            _derivativePowerInfo = new OrderDataElement<DerivativeBuyingPowerInfo>(this, delegate(ref string m) { return _dataSource.GetDerivativePowerInfo(ref m); });

            _accountInfo = new AccountInfo();
            _accountInfo.CreditAccountEnabled = true; //蔲
        }

        public OrderDataElement<TradeOrderList> StockOrder {
            get {
                return _stockOrderList;
            }
        }
        public OrderDataElement<TradePositionList> StockPosition {
            get {
                return _stockPositionList;
            }
        }
        public OrderDataElement<TradeOrderList> DerivativeOrder {
            get {
                return _derivativeOrderList;
            }
        }
        public OrderDataElement<TradePositionList> DerivativePosition {
            get {
                return _derivativePositionList;
            }
        }
        
        public IOrderItemFormatter OrderItemFormatter {
            get {
                //tH[}b^Orderingpsłق
                if(OrderingPlugin.Instance==null)
                    return _defaultFormatter;
                else {
                    EnsureDataSource();
                    return _dataSource.OrderItemFormatter;
                }
            }
        }

        public TradeOrderList ReservedStockOrderList {
            get {
                return _reservedStockOrderList;
            }
        }
        public TradeOrderList ReservedDerivativeOrderList {
            get {
                return _reservedDerivativeOrderList;
            }
        }

        public AccountInfo AccountInfo {
            get {
                return _accountInfo;
            }
        }
        public OrderDataElement<StockBuyingPowerInfo> StockBuyingPowerInfo {
            get {
                return _stockPowerInfo;
            }
        }
        public OrderDataElement<DerivativeBuyingPowerInfo> DerivativeBuyingPowerInfo {
            get {
                return _derivativePowerInfo;
            }
        }

        //MTCh ͕ʃXbhŎs\
        public void SetStockOrderList(DataResponseCode dt, TradeOrderList data, string message) {
            _stockOrderList.SetResponse(dt, data, message);
        }
        public void SetStockPositionList(DataResponseCode dt, TradePositionList data, string message) {
            _stockPositionList.SetResponse(dt, data, message);
        }
        public void SetDerivativeOrderList(DataResponseCode dt, TradeOrderList data, string message) {
            _derivativeOrderList.SetResponse(dt, data, message);
        }
        public void SetDerivativePositionList(DataResponseCode dt, TradePositionList data, string message) {
            _derivativePositionList.SetResponse(dt, data, message);
        }
        public void SetStockPowerInfo(DataResponseCode dt, StockBuyingPowerInfo data, string message) {
            _stockPowerInfo.SetResponse(dt, data, message);
        }
        public void SetDerivativePowerInfo(DataResponseCode dt, DerivativeBuyingPowerInfo data, string message) {
            _derivativePowerInfo.SetResponse(dt, data, message);
        }
        public void SetStockSnapshot(DataResponseCode dt, OrderDataStockSnapshot data, string message) {
            //Debug.Assert(_stockSnapshotDelegate!=null);
            DataResponseCode result = dt & DataResponseCode.ResultMask;
            if(result==DataResponseCode.Failed) {
                CallStockSnapshotDelegate(AsyncOpResult.Failed, null);
                BUtil.ShowWarningMessageBox(message);
            }
            else {
                CallStockSnapshotDelegate(AsyncOpResult.Succeeded, data);
            }
        }
        public void SetDerivativeSnapshot(DataResponseCode dt, OrderDataStockSnapshot data, string message) {
            //Debug.Assert(_stockSnapshotDelegate!=null);
            DataResponseCode result = dt & DataResponseCode.ResultMask;
            if(result==DataResponseCode.Failed) {
                CallStockSnapshotDelegate(AsyncOpResult.Failed, null);
                BUtil.ShowWarningMessageBox(message);
            }
            else {
                CallStockSnapshotDelegate(AsyncOpResult.Succeeded, data);
            }
        }
        private void CallStockSnapshotDelegate(AsyncOpResult result, OrderDataStockSnapshot data)  {
            try {
                if(BUtil.IsExecutingInMainThread)
                    _stockSnapshotDelegate(result, data);
                else
                    BellagioPlugin.Instance.MainWindow.Invoke(_stockSnapshotDelegate, result, data);
            }
            finally {
                _stockSnapshotDelegate = null;
            }
        }


        public void OnDataSourceError(string message) {
            ShowFailMessage(message);
        }

        //ԍXV(Xg[̒ʒm)
        public void UpdateOrderStatus(string uniqueString, OrderStatusEvent eventCode) {
            TradeOrderItem item = FindTradeOrderItemByID(uniqueString);
            if(item==null) return;

            switch(eventCode) {
                case OrderStatusEvent.Accepted:
                    item.OrderStatus = OrderStatus.Ordered;
                    item.TransactionStatus = OrderTransactionStatus.Stable;
                    break;
                case OrderStatusEvent.Cancelled:
                    item.OrderStatus = OrderStatus.Cancelled;
                    item.TransactionStatus = OrderTransactionStatus.Stable;
                    break;
                case OrderStatusEvent.Changed:
                    item.OrderStatus = OrderStatus.Changed;
                    item.TransactionStatus = OrderTransactionStatus.Stable;
                    break;
                case OrderStatusEvent.Expired:
                    item.OrderStatus = OrderStatus.Expired;
                    item.TransactionStatus = OrderTransactionStatus.Stable;
                    break;
                case OrderStatusEvent.Failed: //ŝƂ́Aŝ͂킩Ȃ
                    item.TransactionStatus = OrderTransactionStatus.Rejected;
                    break;
            }

            GetOrderDataElementFor(item).CallEventDelegates();

        }

        //̑M@ԍXVʒmZbg
        public void SendNewOrCloseOrder(TradeOrderItem item, string pinCode) {
            lock(this) {
                TradeOrderItem result;
                string successfulMessage;
                _dataSource.SendNewOrCloseOrder(item, pinCode, out result, out successfulMessage);

                OrderDataElement<TradeOrderList> e = GetOrderDataElementFor(item);
                if(e.Value!=null) e.Value.AddFirst(result); 
                e.CallEventDelegates();

                if(BellagioRoot.FixedPreferences.AutoPinCode)
                    BellagioRoot.FixedPreferences.SavedPinCode = pinCode;
                if(BellagioRoot.FixedPreferences.Ordering.ShowSuccessfulOrderMessage)
                    BUtil.ShowInformationMessageBox(successfulMessage);
                if(BellagioRoot.FixedPreferences.Ordering.PositionRefreshByTrade) {
                    OrderDataElement<TradePositionList> pe = GetPositionDataElementFor(item);
                    if(pe.Value!=null) pe.Refresh();
                }
            }
        }
        public void SendSpecifiedElementCloseOrder(TradeOrderItem item, string pinCode) {
            lock(this) {
                TradeOrderItem result;
                string successfulMessage;
                PositionAndVolumeList pos_list;
                if(item.IsClosePositionDetailAvailable)
                    pos_list = item.ClosePositionDetail;
                else {
                    pos_list = new PositionAndVolumeList();
                    pos_list.Add(item.TargetPositionElement.UniqueID, item.Volume);
                }
                _dataSource.SendSpecifiedElementCloseOrder(item, pos_list, pinCode, out result, out successfulMessage);

                OrderDataElement<TradeOrderList> e = GetOrderDataElementFor(item);
                if(e.Value!=null) e.Value.AddFirst(result); 
                e.CallEventDelegates();

                if(BellagioRoot.FixedPreferences.AutoPinCode)
                    BellagioRoot.FixedPreferences.SavedPinCode = pinCode;
                if(BellagioRoot.FixedPreferences.Ordering.ShowSuccessfulOrderMessage)
                    BUtil.ShowInformationMessageBox(successfulMessage);
                if(BellagioRoot.FixedPreferences.Ordering.PositionRefreshByTrade) {
                    OrderDataElement<TradePositionList> pe = GetPositionDataElementFor(item);
                    if(pe.Value!=null) pe.Refresh();
                }
            }
        }
        public void SendChangeOrder(TradeOrderItem item, string pinCode) {
            lock(this) {
                TradeOrderItem result;
                string successfulMessage;
                _dataSource.SendChangeOrder(item, pinCode, out result, out successfulMessage);

                OrderDataElement<TradeOrderList> e = GetOrderDataElementFor(item);
                if(e.Value!=null) e.Value.Replace(item.UniqueString, result);
                e.CallEventDelegates();

                if(BellagioRoot.FixedPreferences.AutoPinCode)
                    BellagioRoot.FixedPreferences.SavedPinCode = pinCode;
                if(BellagioRoot.FixedPreferences.Ordering.ShowSuccessfulOrderMessage)
                    BUtil.ShowInformationMessageBox(successfulMessage);
                //ύXł̓|WVɉeȂ̂Ń|WV͂ȂĂ悢
            }
        }
        public void SendCancelOrder(TradeOrderItem item, string pinCode) {
            lock(this) {
                TradeOrderItem result;
                string successfulMessage;
                _dataSource.SendCancelOrder(item, pinCode, out result, out successfulMessage);

                OrderDataElement<TradeOrderList> e = GetOrderDataElementFor(item);
                if(e.Value!=null) e.Value.Replace(item.UniqueString, result);
                e.CallEventDelegates();

                if(BellagioRoot.FixedPreferences.AutoPinCode)
                    BellagioRoot.FixedPreferences.SavedPinCode = pinCode;
                if(BellagioRoot.FixedPreferences.Ordering.ShowSuccessfulOrderMessage)
                    BUtil.ShowInformationMessageBox(successfulMessage);
                if(BellagioRoot.FixedPreferences.Ordering.PositionRefreshByTrade) {
                    OrderDataElement<TradePositionList> pe = GetPositionDataElementFor(item);
                    if(pe.Value!=null) pe.Refresh();
                }
            }
        }
        //ȂłP̏Ԃ̂ݍXVB|WVɂ͓KpłȂ̂OrderDataElement̃\bhɂ͂Â炢
        public void RefreshOrder(TradeOrderItem item) {
            lock(this) {
                OrderDataElement<TradeOrderList> e = GetOrderDataElementFor(item);
                if(e.DocumentStatus==BellagioDocumentStatus.Loading) return;

                TradeOrderItem result;
                _dataSource.GetOrderDetailInfo(item.UniqueString,
                    item.Stock.Profile is DerivativeStockProfile? OrderStatusCodeCategory.Future : OrderStatusCodeCategory.Stock, out result);

                if(e.Value!=null) e.Value.Replace(item.UniqueString, result);
                e.CallEventDelegates();
            }
        }


        //Ɖ@ɂ͂PƉs\Ƃ
        private StockSnapshotDelegate _stockSnapshotDelegate;
        public void GetStockSnapshot(AbstractStockProfile prof, StockSnapshotDelegate callback) {
            //Debug.Assert(_stockSnapshotDelegate==null); //TODO ȂAssertɂЂ邱Ƃ
            _stockSnapshotDelegate = callback;
            DerivativeStockProfile ds = prof as DerivativeStockProfile;
            AsyncOpResult r;
            string message = "";
            if(ds!=null)
                r = _dataSource.GetDerivativeSnapshot(ds, ref message);
            else
                r = _dataSource.GetStockSnapshot(prof, ref message);

            if(r!=AsyncOpResult.Async) _stockSnapshotDelegate = null;

            if(r==AsyncOpResult.Failed) callback(AsyncOpResult.Failed, null);
        }


        //OrderItem̎ނɂĕԂ
        private OrderDataElement<TradeOrderList> GetOrderDataElementFor(TradeOrderItem item) {
            if(item.Stock.Profile.IsDerivative)
                return _derivativeOrderList;
            else
                return _stockOrderList;
        }
        private OrderDataElement<TradePositionList> GetPositionDataElementFor(TradeOrderItem item) {
            if(item.Stock.Profile.IsDerivative)
                return _derivativePositionList;
            else
                return _stockPositionList;
        }


        public TradeOrderItem FindTradeOrderItemByID(string id) {
            TradeOrderItem result = null;
            TradeOrderList l = _stockOrderList.Value;
            if(l!=null) result = l.FindTradeOrderItemByID(id);
            if(result!=null) return result;

            l = _derivativeOrderList.Value;
            return l==null? null : l.FindTradeOrderItemByID(id);
        }
        //擾ł΂AȂΎ擾ɂ
        public TradeOrderItem FindOrGetTradeOrderItemByID(string id, OrderStatusCodeCategory category) {
            TradeOrderItem item = FindTradeOrderItemByID(id);
            if(item!=null) return item; //

            BellagioRoot.DataSourceHost.ExternalEnvironment.OrderDataSource.GetOrderDetailInfo(id, category, out item);
            return item;
        }

        private void ShowFailMessage(string message) {
            BUtil.ShowWarningMessageBox(message);
        }

        public void EnsureDataSource() {
            if(_dataSource!=null) return;
            _dataSource = BellagioRoot.DataSourceHost.ExternalEnvironment.OrderDataSource;
            _dataSource.Prepare(this);
        }
#if UNITTEST
        public void SetOrderDataSource(IOrderDataSource ds) {
            _dataSource = ds;
            if(ds!=null) ds.Prepare(this);
        }
#endif

        //ۗ̒ǉ
        public void AddReservedOrder(TradeOrderItem item) {
            lock(this) {
                item.OrderStatus = OrderStatus.Preparing; //ۗ̏Ԃ͋IɕύX
                if(item.Stock.Profile.IsDerivative)
                    _reservedDerivativeOrderList.AddItem(item);
                else
                    _reservedStockOrderList.AddItem(item);
                GetOrderDataElementFor(item).CallEventDelegates();
            }
        }
        public void RemoveReservedOrder(TradeOrderItem item) {
            lock(this) {
                if(item.Stock.Profile.IsDerivative)
                    _reservedDerivativeOrderList.RemoveItem(item);
                else
                    _reservedStockOrderList.RemoveItem(item);
                GetOrderDataElementFor(item).CallEventDelegates();
            }
        }

        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

    //̏^ A敨̗LA]͏ȂǂɊgĂ
    public class AccountInfo {
        private bool _creditAccountEnabled; //Mp̉
        private bool _derivativeAccountEnabled; //敨EIvV̉
        private bool _ippanShinyoEnabled; //ʐMp̉
        private bool _tokuteiKouzaEnabled; //J݂ǂ

        public bool CreditAccountEnabled {
            get {
                return _creditAccountEnabled;
            }
            set {
                _creditAccountEnabled = value;
            }
        }
        public bool DerivativeAccountEnabled {
            get {
                return _derivativeAccountEnabled;
            }
            set {
                _derivativeAccountEnabled = value;
            }
        }
        public bool IppanShinyoEnabled {
            get {
                return _ippanShinyoEnabled;
            }
            set {
                _ippanShinyoEnabled = value;
            }
        }
        public bool TokuteiKouzaEnabled {
            get {
                return _tokuteiKouzaEnabled;
            }
            set {
                _tokuteiKouzaEnabled = value;
            }
        }



    }

    //]͏
    public class StockBuyingPowerInfo {
        private long _genbutsu;
        private long _credit;  //MpVK]
        private double _creditRate; //ۏ؋ێ@f[^\[XnꂽlZbgB

        public long Genbutsu {
            get {
                return _genbutsu;
            }
            set {
                _genbutsu = value;
            }
        }
        public long Credit {
            get {
                return _credit;
            }
            set {
                _credit= value;
            }
        }
        public double CreditRate {
            get {
                return _creditRate;
            }
            set {
                _creditRate = value;
            }
        }

        public StockBuyingPowerInfo Clone() {
            StockBuyingPowerInfo r = new StockBuyingPowerInfo();
            r._credit = _credit;
            r._creditRate = _creditRate;
            r._genbutsu = _genbutsu;
            return r;
        }
    }
    public class DerivativeBuyingPowerInfo {
        private long _marginYoryoku; //]
        private long _marginDeficiency; //sz
        private long _totalMargin;
        
        //@ʓ|Ȃ̂public
        public int maxVolumeLargeLong;
        public int maxVolumeLargeShort;
        public int maxVolumeMiniLong;
        public int maxVolumeMiniShort;

        public long MarginYoryoku {
            get {
                return _marginYoryoku;
            }
            set {
                _marginYoryoku = value;
            }
        }
        public long MarginDeficiency {
            get {
                return _marginDeficiency;
            }
            set {
                _marginDeficiency = value;
            }
        }
        public long TotalMargin {
            get {
                return _totalMargin;
            }
            set {
                _totalMargin = value;
            }
        }

        public int GetMaxVolumeFor(DerivativeStockProfile prof, LongShort ls) {
            if(prof.Brand==DerivativeStockProfile.BrandT.NikkeiFuture)
                return ls==LongShort.Long? maxVolumeLargeLong : maxVolumeLargeShort;
            else if(prof.Brand==DerivativeStockProfile.BrandT.NikkeiFutureMini)
                return ls==LongShort.Long? maxVolumeMiniLong : maxVolumeMiniShort;
            else
                return 0;
        }

        public DerivativeBuyingPowerInfo Clone() {
            DerivativeBuyingPowerInfo r = new DerivativeBuyingPowerInfo();
            r._marginYoryoku = _marginYoryoku;
            r._marginDeficiency = _marginDeficiency;
            r._totalMargin = _totalMargin;
            r.maxVolumeLargeLong = maxVolumeLargeLong;
            r.maxVolumeLargeShort = maxVolumeLargeShort;
            r.maxVolumeMiniLong = maxVolumeMiniLong;
            r.maxVolumeMiniShort = maxVolumeMiniShort;
            return r;
        }
    }

    // E敨
    public class OrderDataStockSnapshot {
        private AbstractStockProfile _prof;
        public const int EMPTY_VALUE = -1; //li͎擾łȂƂ
        private int _currentPrice; //ݒl
        private int _lowerLimitPrice; //E
        private int _upperLimitPrice;

        public OrderDataStockSnapshot(AbstractStockProfile prof) {
            _prof = prof;
        }
        public AbstractStockProfile StockProfile {
            get {
                return _prof;
            }
        }
        public int CurrentPrice {
            get {
                return _currentPrice;
            }
            set {
                _currentPrice = value;
            }
        }
        public int LowerLimitPrice {
            get {
                return _lowerLimitPrice;
            }
            set {
                _lowerLimitPrice = value;
            }
        }
        public int UpperLimitPrice {
            get {
                return _upperLimitPrice;
            }
            set {
                _upperLimitPrice = value;
            }
        }
    }

}
