/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Forms/BellagioListControl.cs#6 $
 * $DateTime: 2008/01/15 13:58:47 $
 * 
 * X[YXN[AJX^`XgRg[BI[i[h[ListControlł撣ΎƂ͂ł邪O
 */
using System;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Text;

using Poderosa;
using Poderosa.Commands;
using Bellagio.Common;
using Bellagio.Drawing;
using Bellagio.Environment;

using Travis.Collections;

namespace Bellagio.Forms {
    public abstract class BellagioListControl : UserControl {
        protected static WaitingMark _waitingMark = new WaitingMark(); //Ŝŋ

        protected VScrollBar _scrollbar;
        protected bool _omittedItemExists; //Xy[XȂĕ\̏ȗꂽڂ邩ǂ
        protected ItemPosition[] _itemPositions;


        public BellagioListControl() {
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer|ControlStyles.AllPaintingInWmPaint, true);
            _scrollbar = new VScrollBar();
            _scrollbar.Dock = DockStyle.Right;
            _scrollbar.Enabled = false;
            _scrollbar.ValueChanged += delegate(object sender, EventArgs args) {
                this.Invalidate();
            };
            this.Controls.Add(_scrollbar);
        }

        public bool OmittedItemExists {
            get {
                return _omittedItemExists;
            }
        }

        public void ResetContent() {
            ResetInternalState();
            AdjustScrollBar();
            base.Invalidate();
        }

        protected override void OnResize(EventArgs e) {
            base.OnResize(e);
            AdjustScrollBar();
        }

        protected void AdjustScrollBar() {
            int pos = _scrollbar.Enabled? _scrollbar.Value : 0;
            int max = this.ItemHeight * this.ItemCount;
            int height = this.ClientSize.Height;

            if(height >= max) {
                _scrollbar.Enabled = false;
            }
            else {
                _scrollbar.Enabled = true;
                _scrollbar.LargeChange = this.ItemHeight;
                _scrollbar.Maximum = max - height + this.ItemHeight;
            }
        }

        protected override void OnPaint(PaintEventArgs e) {
            base.OnPaint(e);
            if(PaintIncompleteCases(e)) return;

            try {
                int top = e.ClipRectangle.Top + (_scrollbar.Enabled? _scrollbar.Value : 0); //SACe̍W
                int bottom = e.ClipRectangle.Bottom + (_scrollbar.Enabled? _scrollbar.Value : 0);
                int start = top / this.ItemHeight;
                int end = (bottom / this.ItemHeight) + 1;

                int y = this.ItemHeight*start - (_scrollbar.Enabled? _scrollbar.Value : 0);
                Graphics g = e.Graphics;
                for(int i=start; i<=end; i++) {
                    if(i >=0 && i < this.ItemCount)
                        DrawItem(i, g, y);
                    y += this.ItemHeight;
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }
        //̂ƂAу[ĥƂnh
        protected bool PaintIncompleteCases(PaintEventArgs args) {
            
            switch(GetDocumentStatus()) {
                case BellagioDocumentStatus.Complete:
                    return false;
                case BellagioDocumentStatus.Loading: {
                        int s = _waitingMark.Radius * 2;
                        if(args.ClipRectangle.IntersectsWith(DrawingUtil.CenterPointToRect(DrawingUtil.RectCenterPoint(this.ClientRectangle), s, s))) {
                            Graphics g = args.Graphics;
                            using(HighQualityDrawingSupport dr = new HighQualityDrawingSupport(g)) {
                                _waitingMark.Draw(g, DrawingUtil.RectCenterPoint(this.ClientRectangle), BellagioPlugin.Instance.SansaPlatform.getRootTimer().getTick());
                            }
                        }
                        return true;
                    }
                default:
                    return true; //`悵Ȃ
            }
        }


        protected override void OnMouseWheel(MouseEventArgs e) {
            base.OnMouseWheel(e);
            if(_scrollbar.Enabled) {
                int v = _scrollbar.Value - e.Delta;
                if(v < 0) v = 0;
                else if(v > _scrollbar.Maximum) v = _scrollbar.Maximum; //TODO {Maximum-LargeChange+1Ȃ̂ATCYŗ̈悪LƂɕsRx傫Bʉ
                _scrollbar.Value = v;
            }
        }

        public int ContentWidth {
            get {
                return this.ClientSize.Width - _scrollbar.Width;
            }
        }

        //WACe[J̍WԂ
        protected struct ItemAndPoint {
            public int itemIndex;
            public int x;
            public int y;
        }
        protected ItemAndPoint GetPointInItem(int x, int y) {
            ItemAndPoint p = new ItemAndPoint();
            int value = _scrollbar.Enabled? _scrollbar.Value : 0;
            int i = (value + y) / this.ItemHeight;
            if(i < 0 || i>=this.ItemCount)
                p.itemIndex = -1;
            else {
                p.itemIndex = i;
                p.x = x;
                p.y = (value + y) - i*this.ItemHeight;
            }
            return p;
        }

        


        //vf̈ʒuvZT|[g
        //eڂ̈ʒu@TCYƂɍĕ]
        protected class ItemPosition {
            public int x;
            public int width;
            public int defaultWidth;
            public string text;
            public bool visible; //蓮visibleݒ
            public bool fixedVisible; //VisibleݒύX\

            public ItemPosition(string text_, int defaultWidth_, bool visible_, bool fixedVisible_) {
                defaultWidth = defaultWidth_;
                visible = visible_;
                text = text_;
                fixedVisible = fixedVisible_;
            }

            public int right {
                get {
                    return x+width;
                }
            }
            public int center {
                get {
                    return x + width/2;
                }
            }
        }
        protected struct ItemPositionAlloc {
            public int remain;
            public bool omitted_flag;
        }

        //Œ蕝蓖
        protected void SetFixedPositionWidth(ItemPosition pos, ref ItemPositionAlloc alloc) {
            if(alloc.omitted_flag || !pos.visible)
                pos.width = 0;
            else {
                pos.width = pos.defaultWidth<=alloc.remain? pos.defaultWidth : 0;
                alloc.remain -= pos.width;
                Debug.Assert(alloc.remain >= 0);
                alloc.omitted_flag |= pos.width==0;
            }
        }
        //]̕zBĂ鍀ڂɂ̂ݕ^
        protected void AdjustRemainingWidth2(ItemPosition pos1, ItemPosition pos2, int remain) {
            int count = 0;
            if(pos1.width>0) count++;
            if(pos2.width>0) count++;
            if(count > 0) {
                remain /= count;
                if(pos1.width>0) pos1.width += remain;
                if(pos2.width>0) pos2.width += remain;
            }
        }

        //ICommandTarget擾Ƃ͂΂
        protected ICommandTarget CastCommandTarget() {
            ICommandTarget t = this.ParentForm.Owner as ICommandTarget;
            return t==null? BellagioPlugin.Instance.MainWindow as ICommandTarget : t;
        }

        protected static void DrawDelimiter(Graphics g, Pen pen, int x, int top, int bottom) {
            g.DrawLine(pen, x, top, x, bottom);
        }
        //top,bottomw肵Amarginɏk߂RectԂ
        protected static Rectangle MakeRect(ItemPosition pos, int top, int bottom, int margin) {
            return new Rectangle(pos.x+margin, top+margin, pos.width-margin*2, bottom-top-margin*2);
        }
        //L[p̕\
        protected static string FormatShortcutKey(int index) {
            char ch;
            if(index < 9) //0...8
                ch = (char)('1' + index);
            else if(index < 35)
                ch = (char)('A' + (index - 9));
            else
                return "-";

            return new string(ch, 1);
        }

        //ACRCAEg
        protected struct IconPosition {
            public int iconIndex;
            public int offsetX; //Z̒S_̑΍W
            public IconPosition(int index, int offX, string tooltip) {
                iconIndex = index;
                offsetX = offX;
                toolTipText = tooltip;
            }
            public string toolTipText;
        }
        protected class IconSetInfo {
            private IconList _iconList;
            private IconPosition[] _positions;

            internal IconSetInfo(IconList il, IconPosition[] positions) {
                _iconList = il;
                _positions = positions;
            }
            //`
            public void Draw(Graphics g, int centerX, int centerY) {
                int y = centerY - _iconList.IconSize.Height/2;
                for(int i=0; i<_positions.Length; i++) {
                    g.DrawImage(_iconList.GetBitmap(_positions[i].iconIndex), centerX + _positions[i].offsetX, y);
                }
            }
            //WACRCfNX擾@YȂ-1 ̓ZS̑΍WȂƂɒ
            public int FindHoveringIndex(int x, int y) {
                Size sz = _iconList.IconSize;
                for(int i=0; i<_positions.Length; i++) {
                    if(new Rectangle(_positions[i].offsetX, -sz.Height/2, sz.Width, sz.Height).Contains(x, y))
                        return i;
                }
                return -1;
            }
            public IconPosition GetAt(int index) {
                return _positions[index];
            }
        }

        //ڂvisibleݒReLXgj[
        protected void ShowVisibleConfigContextMenu() {
            ContextMenuStrip cm = new ContextMenuStrip();
            for(int i=0; i<_itemPositions.Length; i++) {
                ItemPosition p = _itemPositions[i];
                ToolStripMenuItem mi = new ToolStripMenuItem();
                mi.Text = p.text;
                mi.Checked = p.visible;
                mi.Enabled = !p.fixedVisible;
                mi.Tag = p;
                mi.Click += delegate(object sender, EventArgs args) {
                    ToolStripMenuItem mi2 = (ToolStripMenuItem)sender;
                    ItemPosition p2 = (ItemPosition)mi2.Tag;
                    p2.visible = !p2.visible;
                    OnVisibleConfigChanged();
                };
                cm.Items.Add(mi);
            }

            cm.Show(this, this.PointToClient(Control.MousePosition));
        }

        //Visible Settings
        protected void AdjustItemPositionVisibility(string value) {
            string[] s = value.Split(',');
            int[] r = new int[s.Length];
            for(int i=0; i<s.Length; i++) {
                r[i] = ParseUtil.ParseInt(s[i], -1);
            }

            for(int i=0; i<_itemPositions.Length; i++) {
                ItemPosition p = _itemPositions[i];
                if(!p.fixedVisible)
                    p.visible = !ArrayUtil.Contains(r, i);
            }
        }
        protected string FormatHiddenColumns() {
            StringBuilder bld = new StringBuilder();
            for(int i=0; i<_itemPositions.Length; i++) {
                if(!_itemPositions[i].visible) {
                    if(bld.Length>0) bld.Append(',');
                    bld.Append(i.ToString());
                }
            }
            return bld.ToString();
        }


        //hNXŒS
        protected abstract int ItemCount { get; }
        protected abstract int ItemHeight { get; }
        protected abstract void DrawItem(int itemIndex, Graphics g, int y); //Pvf`BꕔĂƂƂ̂yɂȂ邱Ƃ
        protected abstract void ResetInternalState();
        protected abstract BellagioDocumentStatus GetDocumentStatus();
        protected abstract void OnVisibleConfigChanged();
        public abstract void ReloadPreference();

        //L[{[hANV
        public virtual void DoActionByNumericIndex(int index) {
        }
    }
}
