/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/ChartDrawing.cs#17 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * `[g̕`
 * OmegaChartChartDrawingɑc悪邪ς
 */
using System;
using System.Collections.Generic;
using System.Collections;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;

#if UNITTEST
using NUnit.Framework;
#endif

using Poderosa.Util;
using Poderosa;

using Sansa.Runtime;
using Travis;
using Travis.LocationOperation;

using Bellagio.Drawing;
using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Forms;

namespace Bellagio.Chart {

    public class ChartDrawing : ViewContentGraphics {
        //private IPlatformBridge _platform;
        private ChartDrawingSettings _settings;

        //private double _scrollProgress; //XN[̐isB01ɍsɂāAE\NP{ɓ QuoteƂ͕ʂ̓

        private IChartDataSource _data; //f[^̂
        private string _detailedStockDescription; //sꂻ̑̏@R[h{bNX̉ɕ\

        private CandleDrawing _candleDrawing;
        private CandleDrawing _highlightCandleDrawing;
        private CandleDrawing _volumeDrawing;
        private IndicatorFigureDrawing _figureDrawing;
        private ChartScaleInfo _scaleInfo;

        private CHARTPOSITION _displayPosition; //`[g\́uE[v̈ʒuB[ł͂Ȃ̂Œ
        private IndicatorFigureDrawingPosition _figureDrawingPosition;

        //zu XN[o[܂܂Ȃ`[gGA番Ă
        private BRect _timeLabelLocation;  //[̓t/\̈
        private BRect _priceLabelLocation; //E[̉i\̈
        private BRect _mainLocation;       //QƂBIV[^AoA{̂ɂɕ
        private BRect _oscillatorLocation;
        private BRect _volumeLocation;
        private BRect _chartBodyLocation;

        private MouseTrackingLinePosition _mousePriceLine; //}EXɒǐ鉿i
        private MouseTrackingLinePosition _mouseTimeLine;  //}EXɒǐt/

        //IV[^O[v̕\ݒ
        public class OscillatorArea {
            public int height;
            public bool visible;
        }

        //ChartDrawing̃CX^XɌŗL̐ݒ
        public class LocalSettings {
            public bool drawHeader;
            public bool enableFigureDrawing;
            public bool enableMouseTrackingLine;
            public List<OscillatorArea> oscillatorArea;
            public CHARTPOSITION lastLockedIndicatorFigurePos;
            public int volumeAreaHeight;

            public LocalSettings() {
                drawHeader = true;
                enableFigureDrawing = true;
                enableMouseTrackingLine = true;
                oscillatorArea = new List<OscillatorArea>();
                lastLockedIndicatorFigurePos = CHARTPOSITION.Empty(null);
            }
        }
        private LocalSettings _localSettings;


        public ChartDrawing(Control control, IPlatformBridge platform, ChartDrawingSettings settings) : base(platform, control) {
            _settings = settings;
            _localSettings = new LocalSettings();
            _scaleInfo = new ChartScaleInfo(this);
            _figureDrawing = new IndicatorFigureDrawing();
            _figureDrawingPosition = new IndicatorFigureDrawingPosition();
            
            _timeLabelLocation = new BRect();
            _priceLabelLocation = new BRect();
            _mainLocation = new BRect();
            _oscillatorLocation = new BRect();
            _volumeLocation = new BRect();
            _chartBodyLocation = new BRect();

            _mousePriceLine = new MouseTrackingLinePosition();
            _mouseTimeLine = new MouseTrackingLinePosition();

            _candleDrawing = new CandleDrawing();
            _highlightCandleDrawing = new CandleDrawing();
            _volumeDrawing = new CandleDrawing();

            _rectangleOperator = new RectangleOperatorBuilder()
                .SplitFixed(SplitDir.Bottom, _settings.timeLabelHeight.V, 0).ProbeBRect(_timeLabelLocation).CloseBranch()
                .SplitFixed(SplitDir.Right, _settings.priceLabelWidth.V, 80).ProbeBRect(_priceLabelLocation).CloseBranch()
                .ProbeBRect(_mainLocation)
                .SplitFixed(SplitDir.Bottom, delegate() { return this.TotalOscillatorHeight(); }, delegate() { return this.TotalOscillatorHeight()*2; }).ProbeBRect(_oscillatorLocation).CloseBranch()
                .SplitFixed(SplitDir.Bottom, delegate() { return _localSettings.volumeAreaHeight; }, delegate() { return _localSettings.volumeAreaHeight*2; }).ProbeBRect(_volumeLocation).CloseBranch()
                .ProbeBRect(_chartBodyLocation)
                .Finish();
            
            Init(_scaleInfo);
        }



        public ChartScaleInfo ChartScaleInfo {
            get {
                return _scaleInfo;
            }
        }
        public ChartDrawingSettings ChartDrawingSettings {
            get {
                return _settings;
            }
        }
        public IndicatorFigureDrawingPosition FigureDrawingPosition {
            get {
                return _figureDrawingPosition;
            }
        }
        public Rectangle FigureAreaRect {
            get {
                Size sz = new Size(_figureDrawing.Settings.width.V, _figureDrawing.LastPaintHeight);
                return new Rectangle(_figureDrawingPosition.GetDrawPoint(_mainLocation, sz), sz);
            }
        }
        public BRect MainLocation {
            get {
                return _mainLocation;
            }
        }
        public LocalSettings ChartLocalSettings {
            get {
                return _localSettings;
            }
        }
        public CHARTPOSITION DisplayPosition {
            get {
                return _displayPosition;
            }
            set {
                _displayPosition = value;
            }
        }
        public int GetMaxCandleCount() {
            if(_chartBodyLocation.IsEmpty)
                return 1;
            else
                return Math.Max(1, ((_chartBodyLocation.Width-_settings.priceLabelWidth.V) / _settings.candlePitch)+1);
        }
        public IChartDataSource ChartDataSource {
            get {
                return _data;
            }
        }

        public int TotalOscillatorHeight() {
            Debug.Assert(_localSettings.oscillatorArea!=null); //ĂԑOResetOscillatorGroupsKv
            int r = 0;
            foreach(OscillatorArea oa in _localSettings.oscillatorArea) {
                if(oa.visible) r += oa.height;
            }
            return r;
        }
        public int VisibleOscillatorGroupCount {
            get {
                int c = 0;
                for(int i=0; i<_localSettings.oscillatorArea.Count; i++)
                    if(_localSettings.oscillatorArea[i].visible) c++;
                return c;
            }
        }


        public void SetChartDataSource(IChartDataSource ds) {
            _data = ds;
            if(ds!=null) {
                ResetOscillatorGroups(ds.OscillatorGroupSet);
                _displayPosition = ds.SectionInfo.LastPosition;
                _detailedStockDescription = ds.Stock==null? "" : FormatDetailedStockDescription(ds.Stock);
                _localSettings.volumeAreaHeight = ParseUtil.ParseInt(ds.SourceSchema.VolumeHeightInfo.Value, _settings.volumeAreaHeight.V);
            }
        }

        public void ResetOscillatorGroups(RuntimeOscillatorGroupSet groups) {
            _localSettings.oscillatorArea.Clear();
            for(int i=0; i<groups.Count; i++) {
                OscillatorArea oa = new OscillatorArea();
                RuntimeOscillatorGroup g = groups[i];
                oa.height = g.Height;
                oa.visible = oa.height>0 && g.VisibleAny;

                _localSettings.oscillatorArea.Add(oa);
            }
        }


        public override void Dispose() {
            if(_candleDrawing!=null)
                _candleDrawing.Dispose();
            if(_highlightCandleDrawing!=null)
                _highlightCandleDrawing.Dispose();
            if(_volumeDrawing!=null)
                _volumeDrawing.Dispose();
        }

        protected override void InvalidateForDataAnimationTick() {
            if(!_chartBodyLocation.IsEmpty && _container!=null) {
                int x = _chartBodyLocation.Right - _settings.candlePitch;
                _container.Invalidate(new Rectangle(x, 0, _settings.candlePitch, _container.Height)); //łẼE\N̂ݕ`
            }
        }

        public override void Paint(Graphics g, Rectangle clip) {
            //`悷܂łɂ̓ZbgĂȂƂ
            Debug.Assert(_data!=null);
            Debug.Assert(_settings!=null);

            /*
            if(_candleDrawing==null) {
                _candleDrawing = new CandleDrawing(_settings.candleColor.col, _settings.backColor.col, _settings.candleWidth.V);
                _volumeDrawing = new CandleDrawing(_settings.volumeGraphColor.col, _settings.backColor.col, _settings.candleWidth.V);
            }
            */
            g.PixelOffsetMode = System.Drawing.Drawing2D.PixelOffsetMode.None;

            //TODO Rectangle܂
            if(_chartBodyLocation.IntersectsWith(ref clip)) {
                DrawPriceLines(g);
                if(_localSettings.drawHeader)
                    g.DrawString(_detailedStockDescription, _settings.stockInfoFont.F, _settings.priceLabelColor.B, _settings.detailedStockInfoX.V, 7);
                DrawVolumeLines(g);
                DrawOscillatorLines(g);
                DrawChartBody(g, ref clip);
            }

            if(_localSettings.enableMouseTrackingLine) DrawMouseTrackingLine(g);
            DrawTimeLines(g);
            if(_localSettings.enableFigureDrawing) DrawIndicatorFigures(g, ref clip);
        }

        private void DrawPriceLines(Graphics g) {
            if(_scaleInfo.PriceTrnanslator==null) return;

            double price_pitch = _scaleInfo.PriceScaleUnit.ToValue();
            double price = price_pitch * Math.Floor(_scaleInfo.MinPrice / price_pitch); //{

            Pen pen = _settings.priceLineStyle.P;
            Brush br = _settings.priceLabelColor.B;
            float str_height = DrawStringUtil.GetFontHeight(_settings.priceLabelFont.F, g);
            float right = _chartBodyLocation.Right;

            while(price < _scaleInfo.MaxPrice + price_pitch) {
                int y = (int)_scaleInfo.PriceTrnanslator.Translate(price);
                if(_chartBodyLocation.ContainsY(y)) {
                    g.DrawLine(pen, _chartBodyLocation.Left, y, right, y);
                    g.DrawString(price.ToString("F0"), _settings.priceLabelFont.F, br, right, y-str_height/2); //li
                }
                price += price_pitch;
            }
        }
        private void DrawVolumeLines(Graphics g) {
            if(_scaleInfo.MaxVolume==0 || _volumeLocation.IsEmpty) return;
            double volume_pitch = _scaleInfo.VolumeScaleUnit.ToValue();
            double volume = 0;

            Pen pen = _settings.volumeLineStyle.P;
            Brush br = _settings.volumeLabelColor.B;
            float str_height = DrawStringUtil.GetFontHeight(_settings.volumeLabelFont.F, g);
            float right = _volumeLocation.Right;

            while(volume < _scaleInfo.MaxVolume + volume_pitch) {
                int y = (int)_scaleInfo.VolumeTranslator.Translate(volume);
                if(_volumeLocation.ContainsY(y)) {
                    if(volume!=0) g.DrawLine(pen, _volumeLocation.Left, y, right, y);
                    g.DrawString(MarketUtil.FormatUnitScale(volume), _settings.volumeLabelFont.F, br, right, y-str_height/2); //li
                }
                volume += volume_pitch;
            }
        }
        private void DrawOscillatorLines(Graphics g) {
            if(_oscillatorLocation.IsEmpty) return;

            float str_height = DrawStringUtil.GetFontHeight(_settings.oscillatorScaleFont.F, g);
            int y = _oscillatorLocation.Top;
            int margin = _settings.oscillatorMargin.V;
            for(int i=0; i<_localSettings.oscillatorArea.Count; i++) {
                OscillatorArea oa = _localSettings.oscillatorArea[i];
                if(_scaleInfo.OscillatorTranslators[i]==null) continue;

                RuntimeOscillatorGroup og = _data.OscillatorGroupSet[i];
                Pen pen = _settings.oscillatorScalePen.P;
                Brush br = new SolidBrush(_settings.oscillatorScalePen.Color);
                float cr = _oscillatorLocation.Right;

                g.DrawLine(pen, _oscillatorLocation.Left, y+margin, cr, y+margin);
                DrawStringUtil.MiddleCenter(g, og.MaxValueString, _settings.oscillatorScaleFont.F, br, (int)cr+12, y+margin);
                g.DrawLine(pen, _oscillatorLocation.Left, y+oa.height/2, cr, y+oa.height/2);
                //DrawStringUtil.MiddleCenter(g, og.MidValueString, _settings.oscillatorScaleFont, br, cr+12, y+oa.height/2); //Ԃɂ͐̂
                g.DrawLine(pen, _oscillatorLocation.Left, y+oa.height-margin, cr, y+oa.height-margin);
                DrawStringUtil.MiddleCenter(g, og.MinValueString, _settings.oscillatorScaleFont.F, br, (int)cr+12, y+oa.height-margin);

                br.Dispose();

                y += oa.height;
            }
        }

        private void DrawTimeLines(Graphics g) {
            //Ԃ̋؂
            ChartSectionInfo section_info = _data.SectionInfo;
            CHARTPOSITION position = _displayPosition;
            if(section_info.IsEmpty || position.IsEmpty) return;

            int time;
            string time_str;
            int x = _chartBodyLocation.Right;
            Pen pen = _settings.timeLineStyle.P;;
            Brush br = _settings.timeLabelColor.B;
            
            //E\N̍ɐ̂{A15:00ȂǏI[͕Kv
            time = _data.GetNextDateOrTimeAt(position);
            if(time!=-1 && _data.IsTimeLabelRequired(time, out time_str)) {
                DrawTimeLineAndLabel(g, pen, br, x, time_str);
            }

            
            //TODO ̌JԂłłĂBȂƂ
            do {
                x -= _settings.candlePitch;
                if(x < _chartBodyLocation.Left - _settings.candlePitch) break; //I

                //xeԂ̓\K{
                time = _data.GetDateOrTimeAt(position);
                if(time!=-1 && _data.IsTimeLabelRequired(time, out time_str))
                    DrawTimeLineAndLabel(g, pen, br, x, time_str);
            } while(section_info.MovePrev(ref position));
        }
        private void DrawTimeLineAndLabel(Graphics g, Pen pen, Brush br, int x, string time) {
            g.DrawLine(pen, x, _mainLocation.Top, x, _mainLocation.Bottom);
            g.DrawString(time, _settings.timeLabelFont.F, br, x, _timeLabelLocation.Top);
        }

        private void DrawChartBody(Graphics g, ref Rectangle clip) {
            g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.Default;

            //`悵悤ƂĂPf[^̍̍WێBindicatorpxW͈قȂ
            int x = _chartBodyLocation.Right /*- ScrollAmount()*/; //XN[CCurrentCandlep̂

            //JgS{l
            ChartSectionInfo section_info = _data.SectionInfo;
            if(section_info.IsEmpty) return;
            CHARTPOSITION position = _displayPosition;
            if(position.IsEmpty) return;

            ChartDrawingStats.CandleCountFullPaint((int)((x-_chartBodyLocation.Left) / _settings.candlePitch));

            bool latest = section_info.LastPosition.FastEq(ref position);
            do {
                x -= _settings.candlePitch;
                CHARTPOSITION position_left = position;
                section_info.MovePrev(ref position_left); //indicator_leftɐs

                if(x+_settings.candlePitch < clip.Left) break; //͈͂Ɏ܂邱Ƃ͂Ȃ
                if(x <= clip.Right) { //`͈̔͊O x͌Ȃ񂷂Ƃɒ
                    ChartDrawingStats.CandleCountActual(1);

                    DrawChartUnit(g, ref position, ref position_left, x, latest);

                }

                latest = false;

            } while(section_info.MovePrev(ref position));

        }
        private void DrawChartUnit(Graphics g, ref CHARTPOSITION position, ref CHARTPOSITION position_left, int x, bool latest) {
            Candle values;
            int hc = (_settings.candlePitch+1)/2;
            ChartSectionInfo section_info = _data.SectionInfo;
            ChartSectionType section_type = position.ToChartSectionType();
            ChartSectionType section_type_left = position_left.IsEmpty? ChartSectionType.Space : position_left.ToChartSectionType(); //p[gȂƂSpace
            RuntimeIndicatorSet inds = _data.Indicators;

            if(section_type==ChartSectionType.Space) {
                //ňقȂwiFœĥ͂肤
                //󔒗̈ł̓CWP[^Ȃ
                if(section_type_left==ChartSectionType.Space) return;
            }

            //ڋύt\̉_Ȃ
            ISpecialIndicatorPaint[] special_paints = inds.GetSpecialIndicatorPaints();
            for(int i=0; i<special_paints.Length; i++) {
                special_paints[i].Draw(_data, this, g, ref position, ref position_left, x);
            }

            if(section_type==ChartSectionType.Data) {
                //CE\N
                values = _data.GetCandleAt(position);
                if(!values.IsNil) {
                    CandleExtraChart e = _data.GetCandleExtraChart(position);
                    DrawCandle(g, x, values, e, latest);
                    if(e!=null)
                        DrawChartExtra(g, x, values, e);

                    //Mpc
                    if(!_volumeLocation.IsEmpty) {
                        LinearTranslator tx = _scaleInfo.VolumeTranslator;
                        if(tx!=null) {
                            Candle l = _data.GetCandleAt(position_left);
                            if(l!=null && !l.IsNil) {
                                int l_cl = l.CreditLong; int r_cl = values.CreditLong;
                                int l_cs = l.CreditShort; int r_cs = values.CreditShort;
                                if(l_cl!=0 && r_cl!=0)
                                    g.DrawLine(_settings.creditLongPen.P, x-_settings.candlePitch+hc, (int)tx.Translate(l_cl), x+hc, (int)tx.Translate(r_cl));
                                if(l_cs!=0 && r_cs!=0)
                                    g.DrawLine(_settings.creditShortPen.P, x-_settings.candlePitch+hc, (int)tx.Translate(l_cs), x+hc, (int)tx.Translate(r_cs));
                            }
                        }
                    }

                }
            }
            
            //Indicators
            //gLbgR
            for(int ii=0; ii<inds.ElementCount; ii++) {
                RuntimeIndicatorElement element = inds.GetElementAt(ii);
                RuntimeIndicator ri = element.Container;
                if(!element.Visible) continue;
                if(ri.SpecialIndicatorPaint!=null && ri.SpecialIndicatorPaint.IsAssociated(element)) continue;
                LinearTranslator tx = GetIndicatorTranslator(element);
                if(tx==null) continue;

                double l = _data.GetIndicatorValue(element, position_left);
                double r = _data.GetIndicatorValue(element, position);
                if(!Double.IsNaN(l) && !Double.IsNaN(r)) { //map[0]-1̂Ƃ͕KRIɂfalseɂȂ
                    element.Pen.DrawLine(g, x - hc, (int)tx.Translate(l), (int)(x + hc), (int)tx.Translate(r));
                }
            }

        }

        private void DrawCandle(Graphics g, int left, Candle values, CandleExtraChart extra, bool is_current_price) {
            int lx = (int)(left + _settings.candlePitch/2);
            Debug.Assert(values!=null);

            LinearTranslator px = _scaleInfo.PriceTrnanslator;
            if(px==null) return;

            AbstractStockProfile prof = _data.Stock.Profile;
            int open = (int)px.Translate(prof.AdjustPriceD(values.Open));
            int close = (int)px.Translate(prof.AdjustPriceD(values.Close));
            int modified_close = close + (is_current_price? CurrentPriceEffect() : 0);
            int high  = (int)px.Translate(prof.AdjustPriceD(values.High));
            int low = (int)px.Translate(prof.AdjustPriceD(values.Low));

            CandleDrawing cd = extra!=null && extra.highlight? _highlightCandleDrawing : _candleDrawing;
            if(open==modified_close)
                cd.DrawCross(g, lx, open, close, high, low);
            else if(open > modified_close) //ȉ㉺ƍW̏㉺͋tBGtFNĝŋt]`[g͓Ȃ
                cd.DrawYosen(g, lx, modified_close, open, high, low);
            else
                cd.DrawInsen(g, lx, open, modified_close, high, low);

            if(!_volumeLocation.IsEmpty) {
                LinearTranslator vx = _scaleInfo.VolumeTranslator;
                if(vx!=null && values.Volume>0) {
                    int v = (int)vx.Translate(values.Volume);
                    int z = (int)vx.Translate(0);
                    _volumeDrawing.DrawUpBar(g, lx, v, z);
                }
            }
        }
        //`[g̓
        private void DrawChartExtra(Graphics g, int x, Candle value, CandleExtraChart extra) {
            if(_scaleInfo.PriceTrnanslator==null) return;

            if(extra.fushiHigh || extra.fushiLow) {
                int ix = (int)(x + _settings.candlePitch/2);
                Brush br = _settings.fushiColor.B;
                int price = extra.fushiHigh? value.High : value.Low;
                int mod = extra.fushiHigh? -5 : 5;
                DrawStringUtil.MiddleCenter(g, _data.Stock.Profile.FormatPrice(price), _settings.priceLabelFont.F, br, ix, (int)_scaleInfo.PriceTrnanslator.Translate(_data.Stock.Profile.AdjustPriceD(price)) + mod);
            }

            //}[N̕\
            SplitInfo split = extra.splitInfo;
            if(split!=null) {
                int ix = (int)(x + _settings.candlePitch/2);
                Brush br = _settings.splitColor.B;
                int y = (int)_scaleInfo.PriceTrnanslator.Translate(value.Low) + 5;
                DrawStringUtil.MiddleCenter(g, "", _settings.priceLabelFont.F, br, ix, y);
                y += (int)DrawStringUtil.GetFontHeight(_settings.priceLabelFont.F, g);
                DrawStringUtil.MiddleCenter(g, String.Format("1:{0}", split.Ratio), _settings.priceLabelFont.F, br, ix, y);
            }
        }

        //}EXǐ
        internal void OnMouseMoveForTrackingLine(int x, int y) {
            if(!_localSettings.enableMouseTrackingLine) return;

            int start, end;

            //t
            if(_chartBodyLocation.ContainsX(x)) {
                int di = GetChartDisplayIndexFromXPos(x);
                if(di < 0) {
                    _mouseTimeLine.NextPaintPosition = -1;
                }
                else {
                    _mouseTimeLine.NextPaintPosition = _chartBodyLocation.Right - _settings.candlePitch * (_displayPosition.ToDisplayIndex()-GetChartDisplayIndexFromXPos(x)) - _settings.candlePitch/2;
                }
            }
            else {
                _mouseTimeLine.NextPaintPosition = -1;
            }
            _mouseTimeLine.GetInvalidateRegion(out start, out end, _settings.mouseTimeLineStyle.Width);
            if(_mouseTimeLine.LastPaintPosition!=_mouseTimeLine.NextPaintPosition) {
                _container.Invalidate(); //IndicatorFigurêłǂĂŜĕ`ɂȂĂ܂BʁXInvalidateĂ`掞̃Nbv̈͂UnionȂ̂ł܂ӖȂ
                _mouseTimeLine.LastPaintPosition = -1; //InvalidateĂ܂Ȃpς
            }

            //li
            if(_chartBodyLocation.ContainsY(y))
                _mousePriceLine.NextPaintPosition = y;
            else
                _mousePriceLine.NextPaintPosition = -1;

            _mousePriceLine.GetInvalidateRegion(out start, out end, 16); //TODO tHg̍ɂ
            _container.Invalidate(new Rectangle(_mainLocation.Left, start, _mainLocation.Width+_settings.priceLabelWidth.V, end-start+1), false);
            _mousePriceLine.LastPaintPosition = -1;

        }
        private void DrawMouseTrackingLine(Graphics g) {
            if(_scaleInfo.PriceTrnanslator==null) return;

            //t
            if(_mouseTimeLine.NextPaintPosition!=-1) {
                CHARTPOSITION pos = GetChartPositionFromXPos(_mouseTimeLine.NextPaintPosition);
                if(!pos.IsEmpty) {
                    int x = _chartBodyLocation.Right - _settings.candlePitch/2 - (_displayPosition.ToDisplayIndex() - pos.ToDisplayIndex()) * _settings.candlePitch;
                    Pen pen = _settings.mouseTimeLineStyle.P;;
                    g.DrawLine(pen, x, _mainLocation.Top, x, _mainLocation.Bottom);
                    _mouseTimeLine.LastPaintPosition = x;
                }
            }

            //li
            if(_mousePriceLine.NextPaintPosition!=-1) {
                int y = _mousePriceLine.NextPaintPosition;
                if(_chartBodyLocation.ContainsY(y)) {
                    Pen pen = _settings.mousePriceLineStyle.P;
                    g.DrawLine(pen, _chartBodyLocation.Left, y, _chartBodyLocation.Right, y);

                    float str_height = DrawStringUtil.GetFontHeight(_settings.priceLabelFont.F, g);

                    int price = (int)_scaleInfo.PriceTrnanslator.Inverse((double)y);
                    int yobine = Yobine(price);
                    if(yobine > 1)
                        price = ((price+yobine/2) / yobine) * yobine; //Ēl̐{ɏC
                    g.DrawString(price.ToString("F0"), _settings.priceLabelFont.F, _settings.priceLabelColor.B, _chartBodyLocation.Right, y - str_height/2);

                    _mousePriceLine.LastPaintPosition = y;
                }
            }
        }

        private void DrawIndicatorFigures(Graphics g, ref Rectangle clip) {
            //MouseTrackingLineĕ\邩߂
            if(_mouseTimeLine.NextPaintPosition!=-1) {
                CHARTPOSITION pos = _localSettings.lastLockedIndicatorFigurePos;
                if(pos.IsEmpty) pos = GetChartPositionFromXPos(_mouseTimeLine.NextPaintPosition);

                if(!pos.IsEmpty)
                    _figureDrawing.Paint(g, _figureDrawingPosition.GetDrawPoint(_mainLocation, new Size(_figureDrawing.Settings.width.V, _figureDrawing.LastPaintHeight)), _figureDrawingPosition.IsDragging, ref clip, _data, ref pos);
            }
        }


        private void FillPriceRange(out bool price_available, out bool volume_available) {
            bool price_available_local = false;
            RuntimeIndicatorElement[] value_inds;
            Debug.Assert(_data!=null);
            Debug.Assert(_data.OscillatorGroupSet!=null);
            RuntimeOscillatorGroup[] value_og = _data.OscillatorGroupSet.GetValueCheckRequiredGroups(out value_inds); //ľvZ̕Kv̂A

            double max_price = Double.MinValue;
            double min_price = Double.MaxValue;
            double max_volume = 0;
            //l̃Zbg
            for(int i=0; i<value_og.Length; i++) {
                RuntimeOscillatorGroup g = value_og[i];
                g.MaxValue = max_price;
                g.MinValue = min_price;
                g.ValidValueExists = false;
            }

            int x = _chartBodyLocation.Right;
            CHARTPOSITION pos = _displayPosition;
            if(!pos.IsEmpty) {
                do {
                    x -= _settings.candlePitch;
                    if(x < _chartBodyLocation.Left - _settings.candlePitch) break; //I

                    //ƂɍōEŒ߂
                    if(pos.ToDataIndexSoft()!=-1) {
                        Candle y = _data.GetCandleAt(pos);
                        if(!y.IsNil) {
                            max_price = Math.Max(y.High, max_price);
                            min_price = Math.Min(y.Low, min_price);
                            max_volume = Math.Max(y.Volume, max_volume);
                            price_available_local = true; //Płtrue
                        }
                        for(int i=0; i<value_inds.Length; i++) {
                            double v = _data.GetIndicatorValue(value_inds[i], pos);
                            if(!Double.IsNaN(v)) {
                                RuntimeOscillatorGroup g = value_inds[i].Container.OscillatorGroup;
                                Debug.Assert(g!=null);
                                g.MaxValue = Math.Max(g.MaxValue, v);
                                g.MinValue = Math.Min(g.MinValue, v);
                                g.ValidValueExists = true;
                            }
                        }
                        
                    }

                } while(_data.SectionInfo.MovePrev(ref pos));
            }

            min_price = _data.Stock.Profile.AdjustPriceD(min_price);
            max_price = _data.Stock.Profile.AdjustPriceD(max_price);

            if(_priceLabelLocation.IsEmpty) price_available_local = false;
            if(_volumeLocation.IsEmpty) max_volume = 0;

            if(price_available_local) {
                int yobine = Yobine((int)min_price);
                //Debug.WriteLine(String.Format("FillP {0}/{1}, y={2}", min_price, max_price, yobine));
                if(max_price-min_price < yobine*5) { //㉺܂ɏƂ́Akڂ𒲐ȂƌÂ炢@Ăђl̋Et߂ƕςȕ\ɂȂ邩
                    double d = (yobine*5 - (max_price-min_price)) / 2;
                    max_price += d;
                    min_price -= d;
                }
            }
            else { //liȂƂ͑OIl𒆐Sɂ
                int prev_close = _data.LastClosePrice;
                int yobine = Yobine((int)prev_close);
                max_price = prev_close + yobine*5;
                min_price = prev_close - yobine*5;
            }

            //IV[^̒
            for(int i=0; i<value_og.Length; i++) {
                RuntimeOscillatorGroup g = value_og[i];
                if(g.ValidValueExists) {
                    if(g.IndicatorLocation.IsSymmetric) {
                        double max = Math.Max(Math.Abs(g.MaxValue), Math.Abs(g.MinValue));
                        if(max==0)
                            g.ValidValueExists = false;
                        else {
                            double mv = GraphScaleUnit.FromFloor(max).ToValue(); //Βl̂ł̗p
                            g.MaxValue = mv;
                            g.MinValue = -mv;
                        }
                    }
                    else {
                        int sign_max = g.MaxValue>0? 1 : -1;
                        int sign_min = g.MinValue>0? 1 : -1;
                        GraphScaleUnit max_scale = GraphScaleUnit.FromFloor(Math.Abs(g.MaxValue));
                        GraphScaleUnit min_scale = GraphScaleUnit.FromFloor(Math.Abs(g.MinValue));
                        if(sign_max < 0) max_scale = max_scale.Decrement();
                        if(sign_min > 0) min_scale = min_scale.Decrement(); //̂̒͂₱BꍇčlΖ炩Ȃ񂾂ǂ
                        g.MaxValue = max_scale.ToValue() * sign_max;
                        g.MinValue = min_scale.ToValue() * sign_min;
                    }
                }
            }

            volume_available = max_volume>0;
            price_available = price_available_local; //outp[^̓fQ[g璼ڕύXłȂ
            _scaleInfo.SetPriceRange(max_price, min_price, max_volume);
        }

        protected override void BuildScaleInfo() {
            bool price_available, volume_available;
            FillPriceRange(out price_available, out volume_available); //BuildScaleInfoOiŁAliƏo͈̔͂܂ScaleInfoɃZbg
            
            int vb = _volumeLocation.Bottom; //͂悭oĂ
            int pb = _chartBodyLocation.Bottom;
            if(_chartBodyLocation.Height < _settings.priceScaleUnitInPixel.V || _chartBodyLocation.Width==0) price_available = false;
            if(_volumeLocation.Height < _settings.volumeScaleUnitInPixel.V || _volumeLocation.Width==0) volume_available = false;

            GraphScaleUnit price_scale =  new GraphScaleUnit();
            if(price_available)
                price_scale = AdjustGraphScaleUnit(GraphScaleUnit.FromFloor((_scaleInfo.MaxPrice - _scaleInfo.MinPrice) / (_chartBodyLocation.Height / _settings.priceScaleUnitInPixel.V)));
            GraphScaleUnit volume_scale = new GraphScaleUnit();
            if(volume_available)
                volume_scale = AdjustGraphScaleUnit(GraphScaleUnit.FromFloor(_scaleInfo.MaxVolume / (_volumeLocation.Height / _settings.volumeScaleUnitInPixel.V)));

            LinearTranslator price_tr = null;
            if(price_available) {
                double v = price_scale.ToValue();
                price_tr = LinearTranslator.Solve(_scaleInfo.MinPrice-v/2, pb, _scaleInfo.MaxPrice+v/2, _chartBodyLocation.Top); //XP[l̔͏㉺ɌԎ
            }
            LinearTranslator volume_tr = null;
            if(volume_available)
                volume_tr = LinearTranslator.Solve(0, vb, volume_scale.ToValue(), vb-_settings.volumeScaleUnitInPixel.V); //o͂܂蓮Ȃ

            //IV[^ : ̓Aj[V͂Ȃ
            int top = _oscillatorLocation.Top;
            _scaleInfo.AdjustOscillatorGroupCount(_localSettings.oscillatorArea.Count);
            for(int i=0; i<_localSettings.oscillatorArea.Count; i++) {
                OscillatorArea oa = _localSettings.oscillatorArea[i];
                RuntimeOscillatorGroup og = _data.OscillatorGroupSet[i]; //CfbNX͈vĂ
                int m = _settings.oscillatorMargin.V;
                LinearTranslator ot = null;
                if(!_oscillatorLocation.IsEmpty && oa.visible) {
                    if(!og.IndicatorLocation.ValueCheckRequired) //llocationɂĒ܂ĂƂ͂g
                        ot = LinearTranslator.Solve(og.IndicatorLocation.MinValue, top+oa.height-m, og.IndicatorLocation.MaxValue, top+m);
                    else if(og.ValidValueExists) //`FbNKvȂƂValidValueExistŝƂ̂݌vZAȊOTranslatornullɂĕ`悵Ȃ
                        ot = LinearTranslator.Solve(og.MinValue, top+oa.height-m, og.MaxValue, top+m);
                    top += oa.height;
                }
                _scaleInfo.OscillatorTranslators[i] = ot;
            }

            _scaleInfo.UpdateScale(price_scale, volume_scale, price_tr, volume_tr, _updateReason);

            //Debug.WriteLine(String.Format("BSI vol={0} osc={1}", _volumeLocation.ToString(), _oscillatorLocation.ToString()));
            _updateReason = GraphUpdateReason.None;

            _candleDrawing.Reset(_settings.candleColor.col, _settings.backColor.col, _settings.candleWidth.V);
            _highlightCandleDrawing.Reset(_settings.highlightCandleColor.col, _settings.backColor.col, _settings.candleWidth.V);
            _volumeDrawing.Reset(_settings.volumeGraphColor.col, _settings.backColor.col, _settings.candleWidth.V);

        }
        //'2.5'ȉƃ`[g\ɕsR̂Œ
        private GraphScaleUnit AdjustGraphScaleUnit(GraphScaleUnit gu) {
            if(gu.ToValue()<=2.5)
                return new GraphScaleUnit(ScaleType._5, 0);
            else
                return gu;
        }

        //TickAj[V
        private int CurrentPriceEffect() {
            //ŐVl̂PO̒l qATickvZƍs߂lpd gāAAj[VB
            //̌vZ͍WPʂȂ̂Œ
            ChartScaleInfo.PriceMoveEffectT me = _scaleInfo.PriceMoveEffect;
            if(!me.available) return 0;

            LinearTranslator px = _scaleInfo.PriceTrnanslator;
            int p = (int)px.Translate(me.lastPrice);
            int q = (int)px.Translate(me.semiLastPrice) - p;
            int pd = (me.isUptick? -3 : 3);
            
            //TODO ̊ԔȌvZȂƂ
            double dv = this.DataAnimationTimer.value();
            if(dv < 0.33)
                return DrawingUtil.ProportionalInt(q, pd, dv*3);
            else
                return DrawingUtil.ProportionalInt(pd, 0, (dv-0.33)*1.5);
        }

        private LinearTranslator GetIndicatorTranslator(RuntimeIndicatorElement ind) {
            IndicatorLocation loc = ind.Container.Location;
            if(loc==IndicatorLocation.Chart)
                return _scaleInfo.PriceTrnanslator;
            else if(loc==IndicatorLocation.Volume)
                return _scaleInfo.VolumeTranslator;
            else if(loc==IndicatorLocation.Figure)
                return null;
            else {
                Debug.Assert(loc.IsOscillator);
                int oi = ind.Container.OscillatorGroup.Index;
                Debug.Assert(oi>=0 && oi<_scaleInfo.OscillatorTranslators.Length);
                return _scaleInfo.OscillatorTranslators[oi];
            }
        }
        /*
        private int ScrollAmount() {
            return (int)Math.Floor(_settings.candlePitch * _scrollProgress);
        }
        */

        //}EXWuĂvRuntimeIndicatorԂBȂnull
        internal RuntimeIndicatorElement FindRuntimeIndicatorFormMousePosition(int x, int y) {
            CHARTPOSITION pos = GetChartPositionFromXPos(x);
            if(pos.IsEmpty) return null;

            foreach(RuntimeIndicatorElement ri in _data.Indicators.Elements) {
                if(!ri.Visible) continue;
                double v = _data.GetIndicatorValue(ri, pos);
                if(Double.IsNaN(v)) continue;
                LinearTranslator tr = GetIndicatorTranslator(ri);
                if(tr==null) continue;
                double d = Math.Abs(y - tr.Translate(v));
                if(d < 3) return ri; //}ɌXĂƂ͂ł͕sRȂƂ邩
            }

            return null;
        }

        private CHARTPOSITION GetChartPositionFromXPos(int x) {
            if(x < _chartBodyLocation.Left) return CHARTPOSITION.Empty(_data.SectionInfo);
            int index = GetChartDisplayIndexFromXPos(x);
            if(index < 0) return CHARTPOSITION.Empty(_data.SectionInfo);

            CHARTPOSITION pos = _data.SectionInfo.GetChartPosition(index);
            return pos;
        }
        private int GetChartDisplayIndexFromXPos(int x) {
            int t = (_chartBodyLocation.Right - x) / _settings.candlePitch;
            if(t < 0) return -1; //`[g̉E[E

            if(_displayPosition.IsEmpty) return -1;
            int di = _displayPosition.ToDisplayIndex();
            if(di-t < 0) return -1;
            return di-t;
        }
        private static string FormatDetailedStockDescription(Stock stock) {
            if(!(stock.Profile is BasicStockProfile)) return ""; //wE敨͓ɕ\̕KvȂ

            StringBuilder credit = new StringBuilder();
            StockFlags seido = stock.StockFlags & (StockFlags.Seido|StockFlags.Taisyaku);
            StockFlags mukigen = stock.StockFlags & StockFlags.Ippan;
            string seido_str = "";
            switch(seido) {
                case StockFlags.Seido:
                    seido_str = "xMp()"; break;
                case StockFlags.Taisyaku:
                    seido_str = "xMp()"; break;
                case StockFlags.Taisyaku|StockFlags.Seido:
                    seido_str = "xMp(E)"; break;
                default:
                    seido_str = ""; break;
            }
            string subtype = stock.MarketSubType==StockExchangeSubType.None? "" : EnumDescAttribute.For(typeof(StockExchangeSubType)).GetDescription(stock.MarketSubType);
            string nikkei = (stock.Profile.Flags & StockProfileFlags.Nikkei225)!=StockProfileFlags.None? "o225̗p" : "";

            return String.Format("{0}{1}    P {2}     {3} {4} {5}",
                EnumDescAttribute.For(typeof(StockExchange)).GetDescription(stock.Market),
                subtype,
                stock.VolumeUnit,
                seido_str,
                mukigen==StockFlags.Ippan? "()" : "",
                nikkei);
        }

        //}EXɒǐē̏
        //MouseMoveCxg : _lastPaintɑΉ̈Invalidate , _nextPaint̐ݒ
        //`@: _nextPaintɏ]Ă̕`A_lastPaint̐ݒ
        //Ƃ{BcEƂɎg
        public class MouseTrackingLinePosition {
            private int _lastPaint;
            private int _nextPaint;
            public MouseTrackingLinePosition() {
                _lastPaint = -1;
                _nextPaint = -1;
            }
            public int LastPaintPosition {
                get {
                    return _lastPaint;
                }
                set {
                    _lastPaint = value;
                }
            }
            public int NextPaintPosition {
                get {
                    return _nextPaint;
                }
                set {
                    _nextPaint = value;
                }
            }
            //InvalidateׂW͈͂ݒ肷B_nextPaint̕܂Ŋ܂߂ȂOnPaint_nextPaint܂񂾕ĂĂȂ̂Œ
            public void GetInvalidateRegion(out int start, out int end, int pen_width) {
                int half_width = pen_width/2+1;
                if(_lastPaint==-1) {
                    if(_nextPaint==-1) {
                        start = 0;
                        end = 0; //sKv
                    }
                    else {
                        start = _nextPaint - half_width;
                        end = _nextPaint + half_width;
                    }
                }
                else if(_nextPaint==-1) {
                    start = _lastPaint - half_width;
                    end = _lastPaint + half_width;
                }
                else {
                    start = Math.Min(_lastPaint, _nextPaint) - half_width;
                    end = Math.Max(_lastPaint, _nextPaint) + half_width;
                }
            }
        }

        public void GetOmittedScreenItemList(ref List<string> list) {
            if(_oscillatorLocation.IsEmpty && _data.OscillatorGroupSet.VisibleAny) {
                if(list==null) list = new List<string>();
                list.Add("IV[^");
            }
            if(_volumeLocation.IsEmpty) {
                if(list==null) list = new List<string>();
                list.Add("o");
            }
        }

        internal int GetCurrentMouseTrackingPrice() {
            int p = _mousePriceLine.NextPaintPosition;
            if(p==-1)
                return -1;
            else if(_scaleInfo.PriceTrnanslator==null)
                return -1;
            else
                return (int)_scaleInfo.PriceTrnanslator.Inverse(p);

        }

        public void LockIndicatorFigurePos() {
            _localSettings.lastLockedIndicatorFigurePos = GetChartPositionFromXPos(_mouseTimeLine.NextPaintPosition);
        }
        public void UnlockIndicatorFigurePos() {
            _localSettings.lastLockedIndicatorFigurePos = CHARTPOSITION.Empty(_data.SectionInfo);
        }

        private int Yobine(int value) {
            DerivativeStockProfile d = _data.Stock.Profile as DerivativeStockProfile;
            if(d!=null)
                return MarketUtil.GetDerivativeYobine(d);
            else
                return MarketUtil.YobineIncrement(value) - value;
        }
    }

    //GrddChartƂ̋ʕ
    public interface IChartDataSourceBase {
        Stock Stock { get; }
        int LastClosePrice { get; }
        ChartSectionInfo SectionInfo { get; } //OEꓙɕĂĂOKȂ悤
        Candle GetCandleAt(CHARTPOSITION position);
        int GetDateOrTimeAt(CHARTPOSITION position);
        bool GetChartPositionAt(int raw_time, out CHARTPOSITION result);
        RuntimeIndicatorSet Indicators { get; }
        double GetIndicatorValue(RuntimeIndicatorElement ind, CHARTPOSITION position); //\ȂƂNaN
    }

    //DocumentȂǂőāA`[g̕`挳f[^ɂȂ //TODO positionn̂ref̂قRXg
    public interface IChartDataSource : IChartDataSourceBase {
        ChartSchema SourceSchema { get; }
        int GetNextDateOrTimeAt(CHARTPOSITION position); //position̂PẼE\Nɑʒudata / time
        RuntimeOscillatorGroupSet OscillatorGroupSet { get; }
        bool IsTimeLabelRequired(int raw_time, out string str);
        string GetTimeLabel(int raw_time);
        CandleExtraChart GetCandleExtraChart(CHARTPOSITION pos); //extra𖄂߂ĕԂB
    }
    //`[gɕ⑫\ׂf[^BIChartDataSourceKvɉĒg𖄂߂ĕԂ
    public class CandleExtraChart {
        public bool fushiHigh;
        public bool fushiLow; //lEl̐
        public bool highlight; //؂ł̊ԂȂǁA\
        public SplitInfo splitInfo;
        //ɕAAej[X̃A肪肤
    }



    //œKʑp
    
    public class ChartDrawingStats {
        public static int valueRequestCount;
        public static int valueCacheMissCount;
        public static int candleCountFull; //ClipRect،ȂƂ́A`悷邱ƂɂȂł낤E\N̖{
        public static int candleCountActual;

        [Conditional("PERFORMANCE_CHECK")]
        public static void CandleCountFullPaint(int c) {
            candleCountFull += c;
        }
        [Conditional("PERFORMANCE_CHECK")]
        public static void CandleCountActual(int c) {
            candleCountActual += c;
        }

        [Conditional("PERFORMANCE_CHECK")]
        public static void IncrValueRequestCount() {
            valueRequestCount++;
        }
        [Conditional("PERFORMANCE_CHECK")]
        public static void IncrValueCacheMissCount() {
            valueCacheMissCount++;
        }

        [Conditional("PERFORMANCE_CHECK")]
        public static void Print() {
            if(valueRequestCount==0 || candleCountFull==0) return;
            Debug.WriteLine(String.Format("ValueCache: miss {0} / request {1} ( {2}% )", valueCacheMissCount, valueRequestCount, valueCacheMissCount*100/valueRequestCount));
            Debug.WriteLine(String.Format("DrawData: actual {0} / request {1} ( {2}% )", candleCountActual, candleCountFull, candleCountActual*100/candleCountFull));
        }
    }


}
