/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Personal/DailyData.cs#6 $
 * $DateTime$
 * 
 * OmegaChartX^C̓f[^[_
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

using Poderosa;
using Bellagio.Environment;
using Bellagio.Data;
using Bellagio.Screening;

namespace Bellagio.Personal {
    internal class OmegaChartStyleDailyDataProvider : IDailyDataLoader, IScreeningDailyDataProvider {
        private const int RECORD_SIZE = 32;
        private FileStream _stream;
        private byte[] _buffer;
        private Stock _currentStock;
        private IDailyDataWriteStream _writeStream;

        public void Load(Stock stock, IDailyDataWriteStream holder, DailyDataLoadOption option) {
            try {
                _writeStream = holder;
                string filename = GetfileName(stock);
                int length = ((int)new FileInfo(filename).Length) / RECORD_SIZE;
                _writeStream.InitializeStream(length, false, (stock.Profile.Flags & StockProfileFlags.IgnoreVolume)==StockProfileFlags.None); //OmegaChart̃f[^͂P32oCg
                using(BinaryReader reader = new BinaryReader(new FileStream(filename, FileMode.Open))) {
                    ReadMain(reader, length);
                }
            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
                _writeStream.FailLoading(ex.Message);
                return;
            }

            _writeStream.Complete();
        }

        public SplitReflection DefaultSplitReflection {
            get {
                return SplitReflection.NotRefrected;
            }
        }

        public AsyncOpResult AsyncLoad(Stock stock, IDailyDataWriteStream holder, DailyDataLoadOption option, ref string error) {
            _writeStream = holder;
            _buffer = new byte[0x40000]; //256KB ܂\̂͂
            try {
                _currentStock = stock;
                _stream = new FileStream(GetfileName(stock), FileMode.Open);
                _stream.BeginRead(_buffer, 0, _buffer.Length, new AsyncCallback(OnRead), null);
                return AsyncOpResult.Async;
            }
            catch(Exception ex) {
                error = ex.Message;
                return AsyncOpResult.Failed;
            }
        }

        public void ClearCache(Stock stock) {
        }

        private string GetfileName(Stock stock) {
            string dir = PersonalPlugin.Instance.BellagioEnvironment.DataHomeDir + "daily";
            string filename = Path.Combine(dir, stock.Profile.CodeForDataSource());

            if(!File.Exists(filename))
                throw new BellagioException(String.Format("{0} {1} ̓f[^ {2} Ɍ܂", stock.Profile.Code, stock.Profile.Name, filename));
            return filename;
        }

        private void OnRead(IAsyncResult ar) {
            //񓯊ǂݍ݂̂߂̃}[NmFƂ͎̃EFCgL
            //System.Threading.Thread.Sleep(1000);
            try {
                int byte_len = _stream.EndRead(ar);
                _stream.Close();
                Debug.Assert(byte_len < _buffer.Length); //ǉǂݍ݂KvȃP[X͖T|[gBʓ|
                int length = byte_len / RECORD_SIZE;

                _writeStream.InitializeStream(length, false, (_currentStock.Profile.Flags & StockProfileFlags.IgnoreVolume)==StockProfileFlags.None); //OmegaChart̃f[^͂P32oCg
                using(BinaryReader reader = new BinaryReader(new MemoryStream(_buffer, 0, byte_len, false))) {
                    ReadMain(reader, length);
                }

            }
            catch(Exception ex) {
                Debug.WriteLine(ex.Message);
                Debug.WriteLine(ex.StackTrace);
                _writeStream.FailLoading(ex.Message);
                return;
            }

            _writeStream.Complete();
            //𑣂
            _buffer = null;

        }

        private void ReadMain(BinaryReader reader, int record_count) {
            for(int i=0; i<record_count; i++) {
                //NOTE l{lRXgłȂ悤Ȃt@C̓eꊇbyte[]Ɏ󂯂ēKɃANZXX^C
                int date = reader.ReadInt32();
                int open = reader.ReadInt32();
                int high = reader.ReadInt32();
                int low  = reader.ReadInt32();
                int close= reader.ReadInt32();
                int vol  = reader.ReadInt32();
                int cl = reader.ReadInt32(); //Mpc
                int cs = reader.ReadInt32();
                _writeStream.AddRecord(date, open, high, low, close, vol, cl, cs);
            }
        }

        public IAdaptable GetAdapter(Type adapter) {
            return PersonalPlugin.Instance.PoderosaWorld.AdapterManager.GetAdapter(this, adapter);
        }
        public void Prepare(IPrimaryStreamDataHostBase site) {
        }
        public void Terminate() {
        }


        //XN[jOpBAI
        public void LoadLatestDailyData(AbstractStockProfile prof, int max_length, IDailyDataWriteStream strm) {
            string filename = GetfileName(prof.Primary);
            FileStream fs = new FileStream(filename, FileMode.Open, FileAccess.Read);
            //납ǂނAt@C̒z邱Ƃ͂Ȃ
            int seek_len = Math.Min(max_length*RECORD_SIZE, (int)fs.Length);
            fs.Seek(-seek_len, SeekOrigin.End);
            int count = seek_len / RECORD_SIZE;
            int skipped_head = 0;

            bool volume_available = (prof.Flags & StockProfileFlags.IgnoreVolume)==StockProfileFlags.None;

            try {
                byte[] buf = new byte[RECORD_SIZE];
                int date, open, high, low, close, vol, cl, cs;
                int last_valid_price = 0;
                bool initialized = false;

                for(int i=0; i<count; i++) {
                    int r = fs.Read(buf, 0, buf.Length);
                    Debug.Assert(r==RECORD_SIZE); //łȂP[X̂

                    unsafe {
                        fixed(byte* pb = buf) {
                            int* h = (int*)pb;
                            date = *h;
                            vol  = *(h+5);

                            if(vol!=0 || !volume_available) {
                                open = *(h+1);
                                high = *(h+2);
                                low  = *(h+3);
                                close= *(h+4);
                                last_valid_price = close;
                            }
                            else {
                                if(last_valid_price==0) { //ǂ݂擪l̏ꏉŏoOƒlił炵
                                    skipped_head++;
                                    continue;
                                }
                                //XN[jOł̓`[g\ƂȂƂȂ
                                open = last_valid_price;
                                high = last_valid_price;
                                low = last_valid_price;
                                close = last_valid_price;
                            }
                            
                            cl = *(h+6);
                            cs = *(h+7);
                        }

                        if(!initialized) {
                            strm.InitializeStream(count-skipped_head, false, true);
                            initialized = true;
                        }

                        strm.AddRecord(date, open, high, low, close, vol, cl, cs);
                    }
                }

                strm.Complete();
            }
            finally {
                fs.Close();
            }
        }
        public bool GuessLatestDate(AbstractStockProfile prof, out int result) {
            result = 0;
            string filename = GetfileName(prof.Primary);
            if(!File.Exists(filename)) return false;
            
            FileStream fs = new FileStream(filename, FileMode.Open, FileAccess.Read);
            try {
                if(fs.Length<RECORD_SIZE) return false;
                fs.Seek(-RECORD_SIZE, SeekOrigin.End);
                byte[] buf = new byte[RECORD_SIZE];
                fs.Read(buf, 0, buf.Length);
                unsafe {
                    fixed(byte* pb = buf) {
                        result = *(int*)pb;
                        return true;
                    }
                }
            }
            catch(Exception) {
                return false;
            }
            finally {
                fs.Close();
            }
        }

        public DateTime GuessLatestTradeDate {
            get {
                DateTime t = DateTime.Today;
                if(DateTime.Now.Hour < 19) t = t.AddDays(-1);

                while(!PublicMarketUtil.IsMarketOpenDate(t))
                    t = t.AddDays(-1);

                return t;
            }
        }
    }
}
