/*
 * Copyright 2004,2006 The Poderosa Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * $Id: PoderosaStartup.cs,v 1.7 2006/11/03 09:17:55 okajima Exp $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.IO;
using System.Reflection;

using Poderosa.Plugins;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Poderosa.Boot {

    //u[gp̃Gg|Cg
    /// <summary>
    /// 
    /// </summary>
    /// <exclude/>
    public static class PoderosaStartup {
        public static IPoderosaApplication CreatePoderosaApplication(string[] args) {
            string home_directory = AppDomain.CurrentDomain.BaseDirectory;
            string preference_home = home_directory;
            string open_file = null;
            PluginManifest pm = PluginManifest.CreateByFileSystem(home_directory);

            //R}hCǂ
            int i = 0;
            while(i < args.Length) {
                string t = args[i];
                string v = i<args.Length-1? args[i+1] : "";
                switch(t) {
                    case "-p":
                    case "--profile":
                        preference_home = ResolveProfileDirectory(v);
                        i+=2;
                        break;
                    case "-a":
                    case "--addasm":
                        pm.AddAssembly(home_directory, v.Split(';'));
                        i+=2;
                        break;
                    case "-r":
                    case "--remasm":
                        pm.RemoveAssembly(home_directory, v.Split(';'));
                        i+=2;
                        break;
                    case "-open":
                        open_file = v;
                        i+=2;
                        break;
                    default:
                        i++;
                        break;
                }
            }

            IntPtr hwnd = FindExistingInstance();
            if(!PoderosaPluginEnv.MULTIPLE_INSTANCE_ALLOWED && hwnd!=IntPtr.Zero) {
                System.Windows.Forms.MessageBox.Show(String.Format("{0}͂łɋNĂ܂", PoderosaPluginEnv.WINDOW_CAPTION_KEY), PoderosaPluginEnv.WINDOW_CAPTION_KEY, System.Windows.Forms.MessageBoxButtons.OK, System.Windows.Forms.MessageBoxIcon.Warning);
                Win32.SetForegroundWindow(hwnd);
                return null;
            }
            if(hwnd!=IntPtr.Zero && open_file!=null && TryToSendOpenFileMessage(hwnd, open_file)) return null; //ʃCX^XɑM

            PoderosaStartupContext ctx = new PoderosaStartupContext(pm, home_directory, preference_home, args, open_file);
            return new InternalPoderosaWorld(ctx);
        }
        //vOC\͕񂩂
        public static IPoderosaApplication CreatePoderosaApplication(string plugin_manifest, string preference_home, string[] args) {
            string home_directory = Directory.GetCurrentDirectory();
            InternalPoderosaWorld w = new InternalPoderosaWorld(new PoderosaStartupContext(PluginManifest.CreateByText(plugin_manifest), home_directory, preference_home, args, null));
            return w;
        }
        public static IPoderosaApplication CreatePoderosaApplication(string plugin_manifest, StructuredText preference, string[] args) {
            string home_directory = Directory.GetCurrentDirectory();
            InternalPoderosaWorld w = new InternalPoderosaWorld(new PoderosaStartupContext(PluginManifest.CreateByText(plugin_manifest), home_directory, preference, args, null));
            return w;
        }

        //w̃pX`FbN
        private static string ResolveProfileDirectory(string value) {
            if(StringComparer.InvariantCultureIgnoreCase.Compare(value, "appdata")==0)
                return ConfirmDirectory(Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData));
            if(StringComparer.InvariantCultureIgnoreCase.Compare(value, "commonappdata")==0)
                return ConfirmDirectory(Environment.GetFolderPath(Environment.SpecialFolder.CommonApplicationData));
            else
                return value;
        }
        private static string ConfirmDirectory(string dir) {
            string r = dir + "\\Poderosa";
            if(!Directory.Exists(r))
                Directory.CreateDirectory(r);
            return r;
        }

        //ʃCX^Xւ̑M݂BV[gJbgJƂ̑dNɊւƂŁB
        private static IntPtr FindExistingInstance() {
            //EBhE
            unsafe {
                //find target
                IntPtr hwnd = Win32.FindWindowEx(IntPtr.Zero, IntPtr.Zero, null, null);
                char[] name = new char[256];
                char[] mf  = new char[256];
                while(hwnd!=IntPtr.Zero) {
                    int len = Win32.GetWindowText(hwnd, name, 256);
                    if(new string(name, 0, len).IndexOf(PoderosaPluginEnv.WINDOW_CAPTION_KEY)!=-1) { //Window ClassmFƂƂׂA
                        /*
                        len = Win32.GetWindowModuleFileName(hwnd, mf, 256);
                        if(new string(mf, 0, len).IndexOf(PoderosaPluginEnv.MODULE_FILENAME_KEY)!=-1)
                            return hwnd;
                        */
                        Process[] pss = Process.GetProcessesByName(PoderosaPluginEnv.MODULE_FILENAME_KEY);
                        for(int i=0; i<pss.Length; i++)
                            if(pss[i].MainWindowHandle==hwnd) return hwnd;
                    }
                    hwnd = Win32.FindWindowEx(IntPtr.Zero, hwnd, null, null);
                }

                return IntPtr.Zero;
            }
        }
        private unsafe static bool TryToSendOpenFileMessage(IntPtr hwnd, string filename) {
            char[] data = filename.ToCharArray();
            char* b = stackalloc char[data.Length+1];
            for(int i=0; i<data.Length; i++) b[i] = data[i];
            b[data.Length] = '\0';

            Win32.COPYDATASTRUCT cddata = new Win32.COPYDATASTRUCT();
            cddata.dwData = Win32.PODEROSA_OPEN_FILE_REQUEST;
            cddata.cbData = (uint)(sizeof(char) * (data.Length+1));
            cddata.lpData = b;

            int lresult = Win32.SendMessage(hwnd, Win32.WM_COPYDATA, IntPtr.Zero, new IntPtr(&cddata));
            Debug.WriteLine("TryToSend " + lresult);
            return lresult==Win32.PODEROSA_OPEN_FILE_OK;
        }
    }


    //Ñp[^@R}hCȂǂ\z
    internal class PoderosaStartupContext {
        private static PoderosaStartupContext _instance;
        private string _homeDirectory;
        private string _profileHomeDirectory;
        private string _preferenceFileName;
        private string _initialOpenFile;
        private PluginManifest _pluginManifest;
        private StructuredText _preferences;
        private string[] _args; //ÑR}hC
        private ITracer _tracer; //ÑG[̒ʒm

        public static PoderosaStartupContext Instance {
            get {
                return _instance;
            }
        }

        public PoderosaStartupContext(PluginManifest pluginManifest, string home_directory, string profile_home, string[] args, string open_file) {
            _instance = this;
            _homeDirectory = AdjustDirectory(home_directory);
            _profileHomeDirectory = AdjustDirectory(profile_home);
            _initialOpenFile = open_file;
            _args = args;
            Debug.Assert(pluginManifest!=null);
            _pluginManifest = pluginManifest;
            _preferenceFileName = Path.Combine(_profileHomeDirectory, "options.conf");
            if(!File.Exists(_preferenceFileName))
                _preferences = new StructuredText("Poderosa"); //ō쐬
            else
                _preferences = BuildPreference(_preferenceFileName);
        }
        public PoderosaStartupContext(PluginManifest pluginManifest, string home_directory, StructuredText preference, string[] args, string open_file) {
            _instance = this;
            _homeDirectory = AdjustDirectory(home_directory);
            _profileHomeDirectory = _homeDirectory;
            _initialOpenFile = open_file;
            _args = args;
            Debug.Assert(pluginManifest!=null);
            _pluginManifest = pluginManifest;
            Debug.Assert(preference!=null);
            _preferenceFileName = null;
            _preferences = preference;
        }
        private static string AdjustDirectory(string value) {
            return value.EndsWith("\\")? value : value+"\\";
        }

        public PluginManifest PluginManifest {
            get {
                return _pluginManifest;
            }
        }
        public StructuredText Preferences {
            get {
                return _preferences;
            }
        }
        public string PreferenceFileName {
            get {
                return _preferenceFileName;
            }
        }
        public string HomeDirectory {
            get {
                return _homeDirectory;
            }
        }
        public string ProfileHomeDirectory {
            get {
                return _profileHomeDirectory;
            }
        }
        public string[] CommandLineArgs {
            get {
                return _args;
            }
        }

        //ŏɃI[vt@CBwȂnull
        public string InitialOpenFile {
            get {
                return _initialOpenFile;
            }
        }



        public ITracer Tracer {
            get {
                return _tracer;
            }
            set {
                _tracer = value;
            }
        }

        private static StructuredText BuildPreference(string preference_file) {
            //TODO OȂǂǂKɒʒmKv
            if(File.Exists(preference_file)) {
                TextReader r = null;
                try {
                    r = new StreamReader(preference_file, Encoding.Default);
                    StructuredText t = new TextStructuredTextReader(r).Read();
                    return t;
                }
                finally {
                    if(r!=null) r.Close();
                }
            }
            else
                return new StructuredText("Poderosa");
        }

    }

    //vOC̍\w肷邽߂StructuredTextbp
    internal class PluginManifest {
        
        //PAZup̃f[^
        public class AssemblyNode {
            private Assembly _assembly;
            private StructuredText _manifest;
            private Type[] _pluginTypes;

            public AssemblyNode(Assembly assembly, StructuredText manifest) {
                _assembly = assembly;
                _manifest = manifest;
            }
            public Assembly Assembly {
                get {
                    return _assembly;
                }
            }
            public Type[] PluginTypes {
                get {
                    Debug.Assert(_pluginTypes!=null); //TryToBind̎sKv
                    return _pluginTypes;
                }
            }

            //AZu錾̂Typeo
            public void TryToBind(ITracer tracer) {
                List<Type> types = new List<Type>();
                IList entries = _manifest.FindMultipleEntries("plugin");
                if(entries.Count==0) { //AZuɖߍ܂ꂽAttribute烍[h
                    PluginDeclarationAttribute[] decls = (PluginDeclarationAttribute[])_assembly.GetCustomAttributes(typeof(PluginDeclarationAttribute), false);
                    foreach(PluginDeclarationAttribute decl in decls) {
                        types.Add(decl.Target);
                    }
                }
                else {
                    foreach(string name in entries) {
                        try {
                            Type t = _assembly.GetType(name);
                            if(t==null)
                                tracer.Trace("PluginManager.Messages.TypeLoadError", _assembly.CodeBase, name);
                            else
                                types.Add(t);
                        } catch(Exception) {
                            tracer.Trace("PluginManager.Messages.TypeLoadError", _assembly.CodeBase, name);
                        }
                    }
                }
                _pluginTypes = types.ToArray();
            }
        }

        private StructuredText _data;

        //O̍쐬֎~Bȉstatic\bhgp̂
        private PluginManifest() {
        }


        public IEnumerable Children {
            get {
                return _data.Children;
            }
        }
        public StructuredText RawData {
            get {
                return _data;
            }
        }

        public AssemblyNode LoadAssemblyNode(StructuredText asm) {
            return new AssemblyNode(Assembly.LoadFrom(asm.Name), asm);
        }

        //ɋNɃAZuǉE폜
        public void AddAssembly(string home, string[] filenames) {
            foreach(string f in filenames) {
                _data.AddChild(Path.Combine(home, f));
            }
        }
        public void RemoveAssembly(string home, string[] filenames) {
            foreach(String f in filenames) {
                StructuredText t = _data.FindChild(Path.Combine(home, f));
                if(t!=null) _data.RemoveChild(t);
            }
        }


        //`쐬
        public static PluginManifest CreateByText(string text) {
            PluginManifest m = new PluginManifest();
            m._data = new TextStructuredTextReader(new StringReader(text)).Read();
            return m;
        }
        
        //t@CVXeǂō쐬
        public static PluginManifest CreateByFileSystem(string base_dir) {
            PluginManifest m = new PluginManifest();
            StructuredText st = new StructuredText("manifest");

            //̃fBNgɂ.dllBAvP[Vłł͕svAJ̃fobOsɂ͕Kv
            string[] dlls = Directory.GetFiles(base_dir, "*.dll");
            foreach(string dll in dlls) st.AddChild(dll);

            //qfBNĝ݌B
            string[] dirs = Directory.GetDirectories(base_dir);
            foreach(string dir in dirs) {
                dlls = Directory.GetFiles(dir, "*.dll");
                foreach(string dll in dlls) st.AddChild(dll);
            }

            m._data = st;
            return m;
        }
    }

#if UNITTEST
    [TestFixture]
    public class PluginManifestTests {

        private StringResource _stringResource;

        [TestFixtureSetUp]
        public void Init() {
            //Core.dll͂ƂȂƃ[hłȂ
            Assembly.LoadFrom(String.Format("{0}\\Plugin.dll", PoderosaAppDir()));
            _stringResource = new StringResource("Plugin.strings", typeof(PluginManifest).Assembly);
        }

        [Test]
        public void Test1_DLLList() {
            PluginManifest pm = PluginManifest.CreateByFileSystem(PoderosaAppDir());
            TextWriter strm = new StringWriter();
            TextStructuredTextWriter wr = new TextStructuredTextWriter(strm);
            wr.Write(pm.RawData);
            strm.Close();
            UnitTestUtil.Trace(strm.ToString());
            //NOTE ͂ɖڎȂ
        }

        [Test]
        public void Test2_NormalLoad() {
            ITracer tracer = CreateDefaultTracer();
            PluginManifest pm = PluginManifest.CreateByText(String.Format("manifest {{\r\n  {0}\\Core\\Core.dll {{\r\n plugin=Poderosa.Preferences.PreferencePlugin\r\n}}\r\n}}\r\n", PoderosaAppDir()));
            int count = 0;
            foreach(StructuredText t in pm.Children) {
                PluginManifest.AssemblyNode node = pm.LoadAssemblyNode(t);
                node.TryToBind(tracer);
                Assert.AreEqual(1, node.PluginTypes.Length); //ɎsƂ͌^Ȃ
                Assert.AreEqual("Poderosa.Preferences.PreferencePlugin", node.PluginTypes[0].FullName);
                count++;
            }
            Assert.AreEqual(1, count); //AZuw͂PȂ̂
        }

        [Test]
        public void Test3_AssemblyLoadError() {
            ITracer tracer = CreateDefaultTracer();
            PluginManifest pm = PluginManifest.CreateByText(String.Format("manifest {{\r\n  {0}\\notexist.dll {{\r\n  }}\r\n}}\r\n", PoderosaAppDir()));
            try {
                foreach(StructuredText t in pm.Children) {
                    PluginManifest.AssemblyNode node = pm.LoadAssemblyNode(t);
                    Assert.Fail("we expect exception");
                }
            }
            catch(Exception ex) {
                tracer.Trace(ex);
                Console.Out.WriteLine(ex.Message);
            }
        }

        [Test]
        public void Test4_TypeNotFound() {
            ITracer tracer = CreateDefaultTracer();
            PluginManifest pm = PluginManifest.CreateByText(String.Format("manifest {{\r\n  {0}\\Core\\Core.dll {{\r\n plugin=NotFoundPlugin\r\n}}\r\n}}\r\n", PoderosaAppDir()));
            foreach(StructuredText t in pm.Children) {
                PluginManifest.AssemblyNode node = pm.LoadAssemblyNode(t);
                node.TryToBind(tracer);
                Assert.AreEqual(0, node.PluginTypes.Length);
                CheckOneErrorMessage(tracer.Document, String.Format(_stringResource.GetString("PluginManager.Messages.TypeLoadError"), node.Assembly.CodeBase, "NotFoundPlugin"));
            }
        }

        //NOTE
        // {͂plugin=...̋Lqȗ`eXgׂÂ܂܂łPluginDeclarationAttributePoderosa.Monolithic.dll̂̂ɂȂĂ
        // eXgpɃ[hPlugin.dll̂̂QƂ悤ɂȂƃeXgsłA͂Ȃ胀ŶŒ߂B
        // rhԂPoderosaƋNłĂ΂̋@\͂ƂĂAƂ݂ȂB

        //ȂAPluginManifetōŝType[hƂ܂łŁAꂪƂvOCł邩ǂ̌PluginManagersB

        private string PoderosaAppDir() {
            return UnitTestUtil.GetUnitTestConfig("poderosa_installed_dir");
        }

        //PoderosaWorldoRȂeXgȂ̂łŗ
        private ITracer CreateDefaultTracer() {
            return new DefaultTracer(_stringResource);
        }

        private void CheckOneErrorMessage(TraceDocument doc, string msg) {
            string actual = doc.GetDataAt(0);
            if(actual!=msg) {
                //΂ΒȂBDebugɏoȂƂ킩Â炢
                Debug.WriteLine("actual="+actual);
            }
            Assert.AreEqual(msg, actual);
        }
    }
#endif
}
