﻿/*
 * Copyright (c) 2008  Lagarto Technology, Inc.
 * 
 * $Id$
 * $DateTime$
 * 
 */
#if UNITTEST
using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.Reflection.Emit;
using System.Windows.Forms;
using System.Security;
using System.Security.Permissions;
using System.Security.Policy;
using System.Diagnostics;
using System.ComponentModel;

using NUnit.Framework;

namespace Travis.ObjectPeepHole {
    /***
    [Building a dynamic class from the user-defined interface]

    Style 1: access private fields
    <MemberType> <MemberName> { get; }

    Style 2: invoke private methods
    <ReturnType> <MethodName>(<Arguments>);

    Style 3: fire events(1) - events of the object itself
    void Fire<EventName>(<EventArg>);

    Style 4: fire events(2) - events of fields: especially controls of WinForms
    void Fire<MemberName>_<EventName>(<EventArg>);
    */

    public class PeepHoleBuilder {
        private static ModuleBuilder _module; //unique

        private Type _ifType; //interface type
        private Type _targetType; //the type of target. we create a dynamic type accessing members of _targetType

        private Type _resultType;
        private FieldInfo _targetRef; //a member of _resultType: a reference to the target object
        private FieldInfo _fieldTableField; //FieldInfo[] for fields
        private FieldInfo _methodTableField; //MethodInfo[] for methods

        private List<FieldInfo> _fieldTable;
        private List<MethodInfo> _methodTable;

        private Dictionary<string, FieldInfo> _normalizedFields; //normalized name to FieldInfo

        internal const BindingFlags AllAccessBindingFlags = BindingFlags.NonPublic|BindingFlags.Public|BindingFlags.Instance|BindingFlags.Static;
        private static MethodInfo _fieldInfo_GetValue = typeof(FieldInfo).GetMethod("GetValue");
        private static MethodInfo _fieldInfo_SetValue = typeof(FieldInfo).GetMethod("SetValue", new Type[] { typeof(object), typeof(object) });
        private static MethodInfo _methodInfo_Invoke = typeof(MethodInfo).GetMethod("Invoke", new Type[] { typeof(object), typeof(object[]) });
        private static MethodInfo _methodInfo_componentToEvent = typeof(PeepHoleBuilder).GetMethod("ComponentToEvent");

        private class EventInfoEx {
            public EventInfo eventInfo;
            public MethodInfo invokeMethod;
            public bool useImplicitSender;

            public EventInfoEx(EventInfo ev) {
                eventInfo = ev;
            }
        }

        public PeepHoleBuilder(Type targetType, Type interfaceType) {
            _targetType = targetType;
            _ifType = interfaceType;
            if(!interfaceType.IsInterface) throw new ArgumentException("'interfaceType' must a type of an interface");

            _normalizedFields = new Dictionary<string, FieldInfo>();
            foreach(FieldInfo fi in _targetType.GetFields(AllAccessBindingFlags)) {
                _normalizedFields.Add(DynamicNUnitUtil.NormalizeFieldName(fi.Name), fi);
            }
        }

        public object CreateAccessor(object target) {
            if(target.GetType()!=_targetType) throw new ArgumentException("'target' must be an instance of targetType");

            if(_resultType==null) BuildResultType();

            return Activator.CreateInstance(_resultType, target, _fieldTable.ToArray(), _methodTable.ToArray());
        }

        private void BuildResultType() {
            if(_module==null) CreateModule();
            TypeBuilder bld = _module.DefineType(_targetType.Name + "_" + _ifType.Name);
            bld.AddInterfaceImplementation(_ifType);
            CreateConstructor(bld);

            _fieldTable = new List<FieldInfo>();
            _methodTable = new List<MethodInfo>();

            foreach(MemberInfo mi in _ifType.GetMembers()) { //we expect the members of the interface are all public
                MemberTypes mt = mi.MemberType;
                if(mt==MemberTypes.Property)
                    ProcessProperty(bld, (PropertyInfo)mi); //style 1: property
                else if(mt==MemberTypes.Method) {
                    MethodInfo method = (MethodInfo)mi;
                    string name = method.Name;
                    FieldInfo event_source;
                    EventInfoEx event_info;
                    if(CheckEventName(method, out event_source, out event_info)) {
                        if(event_source!=null)
                            ProcessFieldEventInvoker(bld, method, event_source, event_info);
                        else
                            ProcessTargetEventInvoker(bld, method, event_info);
                    }
                    else
                        ProcessMethodInvoker(bld, method);
                }
                    
            }
            _resultType = bld.CreateType();

            //set table
            //_fieldTableField.SetValue(null, _fieldTable.ToArray());
            //_methodTableField.SetValue(null, _methodTable.ToArray());
        }

        private static void CreateModule() {
            Evidence evidence = AppDomain.CurrentDomain.Evidence;

            PermissionSet require = new PermissionSet(PermissionState.None);
            ReflectionPermission reflection_permission = new ReflectionPermission(PermissionState.None);
            reflection_permission.Flags = ReflectionPermissionFlag.AllFlags;
            require.AddPermission(reflection_permission);

            PermissionSet refused = new PermissionSet(PermissionState.None);
            AssemblyName asmname = new AssemblyName("Travis.DynamicTypes");
            AssemblyBuilder asmbld = AppDomain.CurrentDomain.DefineDynamicAssembly(asmname, AssemblyBuilderAccess.Run,
                new Evidence(evidence), require, null, refused);
            _module = asmbld.DefineDynamicModule("default_module");
        }

        private void CreateConstructor(TypeBuilder bld) {
            //constructor and field
            _targetRef = bld.DefineField("__t", _targetType, FieldAttributes.NotSerialized|FieldAttributes.Private);
            _fieldTableField  = bld.DefineField("__f", typeof(FieldInfo[]), FieldAttributes.NotSerialized|FieldAttributes.Public|FieldAttributes.Static);
            _methodTableField = bld.DefineField("__m", typeof(MethodInfo[]), FieldAttributes.NotSerialized|FieldAttributes.Public|FieldAttributes.Static);

            ILGenerator g = bld.DefineConstructor(MethodAttributes.Public, CallingConventions.Standard, new Type[] { _targetType, typeof(FieldInfo[]), typeof(MethodInfo[]) }).GetILGenerator();
            g.Emit(OpCodes.Ldarg_0); //'this'
            g.Emit(OpCodes.Ldarg_1); //argument
            g.Emit(OpCodes.Stfld, _targetRef);

            //fieldTable, methodTableはstaticなのでコンストラクタで渡す必要はないんだが、
            //dynamic module内のクラスにはFieldInfo.SetValueでは渡せない模様
            //staticメソッドを作成してそれをInvokeすることができるのかは不明

            g.Emit(OpCodes.Ldarg_0); //'this'
            g.Emit(OpCodes.Ldarg_2); 
            g.Emit(OpCodes.Stfld, _fieldTableField);

            g.Emit(OpCodes.Ldarg_0); //'this'
            g.Emit(OpCodes.Ldarg_3);
            g.Emit(OpCodes.Stfld, _methodTableField);

            g.Emit(OpCodes.Ret);

        }

        private void ProcessProperty(TypeBuilder bld, PropertyInfo prop) {
            //Create Property Implementation to access field
            FieldInfo fi;
            if(!_normalizedFields.TryGetValue(DynamicNUnitUtil.NormalizeFieldName(prop.Name), out fi))
                throw new Exception(String.Format("A field for '{0}' is not found in {1}", prop.Name, _targetType.Name));
            if(prop.PropertyType!=fi.FieldType)
                throw new Exception(String.Format("Type mismatch for '{0}': property type is {1}, actual field is {2}", prop.Name, prop.PropertyType.Name, fi.FieldType.Name));

            ILGenerator g;
            PropertyBuilder pb = bld.DefineProperty(prop.Name, PropertyAttributes.None, prop.PropertyType, null);
            int field_index = _fieldTable.Count;
            _fieldTable.Add(fi);

            if(prop.CanRead) {
                MethodBuilder getter = bld.DefineMethod("get_"+prop.Name, MethodAttributes.Private|MethodAttributes.Virtual, prop.PropertyType, Type.EmptyTypes);

                g = getter.GetILGenerator();
                LoadFieldInfo(g, field_index);
                if(fi.IsStatic)
                    g.Emit(OpCodes.Ldnull);
                else 
                    LoadTarget(g);
                g.EmitCall(OpCodes.Callvirt, _fieldInfo_GetValue, null);
                if(IsBoxingRequiredForParameter(prop.PropertyType))
                    g.Emit(OpCodes.Unbox_Any, prop.PropertyType);
                g.Emit(OpCodes.Ret);

                bld.DefineMethodOverride(getter, prop.GetGetMethod());
                pb.SetGetMethod(getter);
            }

            if(prop.CanWrite) {
                MethodBuilder setter = bld.DefineMethod("set_"+prop.Name, MethodAttributes.Private|MethodAttributes.Virtual, null, new Type[] { prop.PropertyType });

                g = setter.GetILGenerator();
                LoadFieldInfo(g, field_index);
                if(fi.IsStatic)
                    g.Emit(OpCodes.Ldnull);
                else 
                    LoadTarget(g);

                g.Emit(OpCodes.Ldarg_1);
                if(IsBoxingRequiredForParameter(prop.PropertyType))
                    g.Emit(OpCodes.Box, prop.PropertyType);
                g.EmitCall(OpCodes.Callvirt, _fieldInfo_SetValue, null);
                g.Emit(OpCodes.Ret);

                bld.DefineMethodOverride(setter, prop.GetSetMethod());
                pb.SetSetMethod(setter);
            }

        }

        //searches event and returns true if appropriate event is found
        private bool CheckEventName(MethodInfo method, out FieldInfo field, out EventInfoEx event_info) {
            string name = method.Name;
            field = null;
            event_info = null;

            if(!name.StartsWith("Fire")) return false;

            //search event
            EventInfo ev = _targetType.GetEvent(name.Substring(4), AllAccessBindingFlags);
            if(ev!=null && MatchEventArgs(ev, method, out event_info)) {
                return true;
            }

            //search member event
            int underbar = name.IndexOf('_');
            if(underbar!=-1) {
                string field_candidate = name.Substring(4, underbar-4).ToLower();
                if(_normalizedFields.TryGetValue(field_candidate, out field)) {
                    ev = field.FieldType.GetEvent(name.Substring(underbar+1), AllAccessBindingFlags);
                    if(ev!=null && MatchEventArgs(ev, method, out event_info)) {
                        return true;
                    }
                }
            }

            //not found
            return false;
        }

        private void ProcessFieldEventInvoker(TypeBuilder bld, MethodInfo method, FieldInfo field, EventInfoEx event_info) {
            // fireevent(args)
            //   1: control c = field.GetValue(_targetRef);
            //   2: call ComponentToEvent(event_key_field, c)   or   event_key_field.GetValue(c)
            //   3: callvirt event_info.Invoker(c, args)
            EventHack hack = WinFormsEventResolver.GetOrCreate(field.FieldType, event_info.eventInfo);
            Debug.Assert(hack!=null);
            Debug.Assert(hack.flags==EventHackFlags.StaticKey || hack.flags==EventHackFlags.InstanceField);

            ParameterInfo[] parameters = method.GetParameters();
            MethodBuilder mb = bld.DefineMethod(method.Name, MethodAttributes.Private|MethodAttributes.Virtual, CallingConventions.Standard, null, ToTypeArray(parameters));
            foreach(ParameterInfo pi in parameters)
                mb.DefineParameter(pi.Position+1, pi.Attributes, pi.Name);
            int field_index = _fieldTable.Count;
            _fieldTable.Add(field);
            int event_key_index = _fieldTable.Count;
            _fieldTable.Add(hack.keyField);

            ILGenerator g = mb.GetILGenerator();
            g.DeclareLocal(typeof(Control));

            //1:
            LoadFieldInfo(g, field_index);
            LoadTarget(g);
            g.EmitCall(OpCodes.Callvirt, _fieldInfo_GetValue, null);
            g.Emit(OpCodes.Stloc_0);

            //2:
            LoadFieldInfo(g, event_key_index);
            g.Emit(OpCodes.Ldloc_0);
            if(hack.flags==EventHackFlags.StaticKey)
                g.EmitCall(OpCodes.Call, _methodInfo_componentToEvent, null);
            else
                g.EmitCall(OpCodes.Callvirt, _fieldInfo_GetValue, null);

            //3:
            if(event_info.useImplicitSender) 
                g.Emit(OpCodes.Ldloc_0); //'sender'
            for(int i=0; i<parameters.Length; i++)
                LDARG(g, i+1);
            g.EmitCall(OpCodes.Callvirt, event_info.invokeMethod, null);

            g.Emit(OpCodes.Ret);

            bld.DefineMethodOverride(mb, method);
        }
        private void ProcessTargetEventInvoker(TypeBuilder bld, MethodInfo method, EventInfoEx event_info) {
            // fireevent(args) {
            //   ldarg_0 (this)
            //   ldfld <targetevent>
            //   callvirt target, arg
            //   
            ParameterInfo[] parameters = method.GetParameters();
            MethodBuilder mb = bld.DefineMethod(method.Name, MethodAttributes.Private|MethodAttributes.Virtual, CallingConventions.Standard, null, ToTypeArray(parameters));
            foreach(ParameterInfo pi in parameters)
                mb.DefineParameter(pi.Position+1, pi.Attributes, pi.Name);
            int field_index = _fieldTable.Count;
            FieldInfo fi = _normalizedFields[event_info.eventInfo.Name.ToLower()];
            Debug.Assert(fi!=null);
            _fieldTable.Add(fi);

            ILGenerator g = mb.GetILGenerator();
            LoadFieldInfo(g, field_index);
            LoadTarget(g);
            g.EmitCall(OpCodes.Callvirt, _fieldInfo_GetValue, null);
            
            if(event_info.useImplicitSender) 
                LoadTarget(g);
            for(int i=0; i<parameters.Length; i++) 
                LDARG(g, i+1);
            g.EmitCall(OpCodes.Callvirt, event_info.invokeMethod, null);

            g.Emit(OpCodes.Ret);
            
            bld.DefineMethodOverride(mb, method);
        }


        private void ProcessMethodInvoker(TypeBuilder bld, MethodInfo method) {
            //filter getter/setter
            if(IsPropertyGetterOrSetter(method)) return;

            MethodInfo mi = FindMethod(method);
            if(mi==null) 
                throw new Exception(String.Format("Appropriate method is not found for {0}", method.Name));

            ILGenerator g;
            ParameterInfo[] parameters = method.GetParameters();
            MethodBuilder mb = bld.DefineMethod(method.Name, MethodAttributes.Private|MethodAttributes.Virtual, CallingConventions.Standard, method.ReturnType, ToTypeArray(parameters));
            foreach(ParameterInfo pi in parameters)
                mb.DefineParameter(pi.Position+1, pi.Attributes, pi.Name);
            
            int method_index = _methodTable.Count;
            _methodTable.Add(mi);

            g = mb.GetILGenerator();
            g.DeclareLocal(typeof(object[])); //local0: object[] for the argument of MethodInfo.Invoke
            //the arguments of MethodInfo.Invoke
            // [1:MethodInfo, 2:target, 3:arg(object[])]

            //1:
            g.Emit(OpCodes.Ldsfld, _methodTableField);
            LDC(g, method_index);
            g.Emit(OpCodes.Ldelem_Ref); //load the target FieldInfo

            //2:
            if(method.IsStatic)
                g.Emit(OpCodes.Ldnull);
            else {
                LoadTarget(g);
            }

            //3:
            LDC(g, parameters.Length);
            g.Emit(OpCodes.Newarr, typeof(object));
            g.Emit(OpCodes.Dup);
            g.Emit(OpCodes.Stloc_0);
            for(int i=0; i<parameters.Length; i++) {
                Type param_type = parameters[i].ParameterType;
                Type concrete_type = ToConcreteType(param_type);
                g.Emit(OpCodes.Dup);
                // valuetype       box -> invoke
                // valuetype+ref   ldobj, box -> invoke -> unbox,stobj
                // object+ref      ldobj -> invoke -> stobj
                LDC(g, i);
                LDARG(g, i+1);
                if(param_type.IsByRef)
                    g.Emit(OpCodes.Ldobj, concrete_type);
                if(IsBoxingRequiredForParameter(param_type))
                    g.Emit(OpCodes.Box, concrete_type);
                g.Emit(OpCodes.Stelem_Ref);
            }

            //Invoke
            g.EmitCall(OpCodes.Callvirt, _methodInfo_Invoke, null);
            if(method.ReturnType==typeof(void))
                g.Emit(OpCodes.Pop);
            else if(method.ReturnType.IsValueType)
                g.Emit(OpCodes.Unbox_Any, method.ReturnType); //I don't understand what's different between Unbox and Unbox_Any

            //unbox ref paramters
            for(int i=0; i<parameters.Length; i++) {
                Type param_type = parameters[i].ParameterType;
                if(param_type.IsByRef) {
                    Type concrete_type = ToConcreteType(param_type);
                    LDARG(g, i+1);

                    g.Emit(OpCodes.Ldloc_0);
                    LDC(g, i);
                    g.Emit(OpCodes.Ldelem_Ref);
                    if(concrete_type.IsValueType)
                        g.Emit(OpCodes.Unbox_Any, concrete_type);

                    g.Emit(OpCodes.Stobj, concrete_type);
                }
            }

            //g.Emit(OpCodes.Unbox, typeof(int));
            g.Emit(OpCodes.Ret);

            bld.DefineMethodOverride(mb, method);
        }
        private MethodInfo FindMethod(MethodInfo src) {
            MethodInfo[] mis = _targetType.GetMethods(AllAccessBindingFlags);
            foreach(MethodInfo mi in mis) {
                //TODO: check the types of argument also!
                if(mi.Name==src.Name && mi.GetParameters().Length==src.GetParameters().Length)
                    return mi;
            }
            return null;
        }
        private bool IsPropertyGetterOrSetter(MethodInfo mi) {
            string name = mi.Name;
            if(name.StartsWith("get_") || name.StartsWith("set_")) {
                return true;
            }
            else
                return false;
        }
        private static Type[] ToTypeArray(ParameterInfo[] pis) {
            Type[] r = new Type[pis.Length];
            for(int i=0; i<r.Length; i++) {
                r[i] = pis[i].ParameterType;
            }
            return r;
        }

        private static Type ToConcreteType(Type param_type) {
            if(param_type.IsByRef)
                return param_type.GetElementType();
            else
                return param_type;
        }
        private static bool IsBoxingRequiredForParameter(Type param_type) {
            if(param_type.IsValueType)
                return true;
            else if(param_type.IsByRef)
                return param_type.GetElementType().IsValueType;
            else
                return false;
        }
        private static bool MatchEventArgs(EventInfo ev, MethodInfo method, out EventInfoEx result) {
            result = null;
            MethodInfo invoker = ev.EventHandlerType.GetMethod("Invoke");
            Debug.Assert(invoker!=null);
            Debug.Assert(invoker.IsPublic);

            ParameterInfo[] event_args = invoker.GetParameters();
            ParameterInfo[] method_args = method.GetParameters();


            if(event_args.Length==method_args.Length) {
                for(int i=0; i<event_args.Length; i++) {
                    if(event_args[i].ParameterType!=method_args[i].ParameterType) return false;
                }
                result = new EventInfoEx(ev);
                result.invokeMethod = invoker;
                result.useImplicitSender = false;
                return true;
            }
            else if(event_args.Length==method_args.Length+1) {
                if(event_args[0].ParameterType!=typeof(object)) return false; //implicit 'sender' arg
                for(int i=1; i<event_args.Length; i++) {
                    if(event_args[i].ParameterType!=method_args[i-1].ParameterType) return false;
                }
                result = new EventInfoEx(ev);
                result.invokeMethod = invoker;
                result.useImplicitSender = true;
                return true;
            }
            else
                return false;
        }


        //ldc. 0から8は固有のがある。他はbyteを一個
        private static OpCode[] short_ldcs = new OpCode[] {
            OpCodes.Ldc_I4_0, OpCodes.Ldc_I4_1, OpCodes.Ldc_I4_2, OpCodes.Ldc_I4_3, 
            OpCodes.Ldc_I4_4, OpCodes.Ldc_I4_5, OpCodes.Ldc_I4_6, OpCodes.Ldc_I4_7, OpCodes.Ldc_I4_8 };
        private static void LDC(ILGenerator g, int index) {
            Debug.Assert(index >= 0);
            Debug.Assert(index < 256);
            if(index < short_ldcs.Length)
                g.Emit(short_ldcs[index]);
            else {
                byte op = (byte)index;
                g.Emit(OpCodes.Ldc_I4_S, op);
            }
        }
        private static OpCode[] short_ldargs = new OpCode[] {
            OpCodes.Ldarg_0, OpCodes.Ldarg_1, OpCodes.Ldarg_2, OpCodes.Ldarg_3
        };
        private static void LDARG(ILGenerator g, int index) {
            Debug.Assert(index >= 0);
            Debug.Assert(index < 256);

            if(index < short_ldargs.Length) {
                g.Emit(short_ldargs[index]);
            }
            else {
                byte op = (byte)index;
                g.Emit(OpCodes.Ldarg_S, op);
            }
            
        }
        private void LoadTarget(ILGenerator g) {
            g.Emit(OpCodes.Ldarg_0); //'target'
            g.Emit(OpCodes.Ldfld, _targetRef);
        }
        private void LoadFieldInfo(ILGenerator g, int index) {
            g.Emit(OpCodes.Ldsfld, _fieldTableField);
            LDC(g, index);
            g.Emit(OpCodes.Ldelem_Ref); //load the target FieldInfo
        }


        private static PropertyInfo component_event = typeof(Component).GetProperty("Events", BindingFlags.NonPublic|BindingFlags.Instance);
        public static Delegate ComponentToEvent(FieldInfo keyField, Control control) {
            object key = keyField.GetValue(null);
            Debug.Assert(key!=null);
            EventHandlerList list = (EventHandlerList)component_event.GetValue(control, null);
            return list[key];
        }
    }



    internal static class DynamicNUnitUtil {
        public static string NormalizeFieldName(string name) {
            int start = 0;
            while(start<name.Length && name[start]=='_') start++;
            int end = name.Length-1;
            while(end>=0 && name[end]=='_') end--;

            return name.Substring(start, end-start+1).ToLower();
        }

        //returns true if the handler type is the (object sender, ***EventArgs arg) style
        public static bool IsFirstArgSender(EventInfo ev) {
            MethodInfo mi = ev.EventHandlerType.GetMethod("Invoke", BindingFlags.NonPublic|BindingFlags.Public|BindingFlags.Instance);
            if(mi==null) return false;

            ParameterInfo[] infos = mi.GetParameters();
            if(infos.Length <= 1) return false;

            return infos[0].ParameterType==typeof(object);
        }

    }

}
#endif
