using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Diagnostics;
using System.Collections.Generic;
using System.Text;

namespace Bellagio.Forms {
    public class SliderTab : UserControl {
        private class Item {
            public string text;

            public Item(string t) {
                text = t;
            }
        }

        private List<Item> _items;
        private int _itemPitch;
        private int _selectedIndex;
        private int _hoverIndex;

        public event EventHandler SelectedIndexChanged;

        public SliderTab() {
            _selectedIndex = -1;
            _hoverIndex = -1;
            _items = new List<Item>();
            _itemPitch = 48;
            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            //this.BackColor = Color.Black;
        }

        public void AddItem(string text) {
            _items.Add(new Item(text));
        }

        public int SelectedIndex {
            get {
                return _selectedIndex;
            }
            set {
                Debug.Assert(value>=0 && value<_items.Count);
                _selectedIndex = value;
            }
        }
        public int ItemPitch {
            get {
                return _itemPitch;
            }
            set {
                _itemPitch = value;
            }
        }

        protected override void OnMouseMove(MouseEventArgs e) {
            base.OnMouseMove(e);
            int i = e.X / _itemPitch;
            if(i >= _items.Count) i = -1;
            if(i != _hoverIndex) {
                _hoverIndex = i;
                Invalidate();
            }
        }
        protected override void OnMouseDown(MouseEventArgs e) {
            base.OnMouseDown(e);
            if(e.Button==MouseButtons.Left) {
                int i = e.X / _itemPitch;
                if(i < _items.Count && i != _selectedIndex) {
                    _selectedIndex = i;
                    //FireEvent!
                    this.SelectedIndexChanged(this, new EventArgs());
                    Invalidate();
                }
            }
        }
        protected override void OnMouseLeave(EventArgs e) {
            base.OnMouseLeave(e);
            if(_hoverIndex!=-1) {
                _hoverIndex = -1;
                Invalidate();
            }
        }

        protected override void OnPaint(PaintEventArgs e) {
            base.OnPaint(e);
            for(int i=0; i<_items.Count; i++) {
                Rectangle rc = new Rectangle(_itemPitch*i, 0, _itemPitch, this.Height);
                if(rc.IntersectsWith(e.ClipRectangle))
                    DrawItem(e.Graphics, _items[i], i, rc);
            }
        }
        private void DrawItem(Graphics g, Item item, int index, Rectangle rc) {
            //background
            Brush br;
            if(index==_selectedIndex)
                br = new SolidBrush(this.BackColor);
            else
                br = new LinearGradientBrush(rc, Color.White , Color.FromArgb(164, 176, 200), LinearGradientMode.Vertical);
            g.FillRectangle(br, rc);
            br.Dispose();

            //؂̉E
            int right = rc.Right;
            g.DrawLine(Pens.LightGray, rc.Right-2, rc.Y, rc.Right-2, rc.Bottom);
            g.DrawLine(Pens.DarkGray, rc.Right-1, rc.Y, rc.Right-1, rc.Bottom);

            //eLXg
            SizeF szf = g.MeasureString(item.text, this.Font);
            g.DrawString(item.text, this.Font, Brushes.Black, rc.X+(int)((_itemPitch-szf.Width)/2), rc.Y+6 );

            //header
            if(index==_selectedIndex)
                DrawItemHeaderMark(g, Pens.DodgerBlue, rc);
            else if(index==_hoverIndex)
                DrawItemHeaderMark(g, Pens.DarkBlue, rc);
        }
        private void DrawItemHeaderMark(Graphics g, Pen pen, Rectangle rc) {
            g.DrawLine(pen, rc.X+1, rc.Y  , rc.Right-2, rc.Y);
            g.DrawLine(pen, rc.X  , rc.Y+1, rc.Right-1, rc.Y+1);
            /*
            g.DrawLine(pen, rc.X  , rc.Y+1, rc.X  , rc.Bottom-4);
            g.DrawLine(pen, rc.X+1, rc.Y  , rc.X+1, rc.Bottom-3);
            g.DrawLine(pen, rc.X+2, rc.Y  , rc.X+2, rc.Bottom-3);
            */
        }
    }
}
