/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Data/LightDaytimeTrade.cs#11 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * ̂݁A͍ȂAƂ^CvDaytimeTrade
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading;

using Poderosa;
using Poderosa.Util.Collections;
using Bellagio.Values; //NSO
using Bellagio.Environment;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Data {
    public class LightIntraDayTrade : StockPrimaryData {

        private RealtimeInfoString<int> _lastClose; 
        private RealtimeInfoString<int> _currentPrice;
        private RealtimeInfoString<int> _timestamp;
        private double _scale;

        public LightIntraDayTrade(Stock stock) : base(stock) {
            _lastClose = new RealtimeInfoString<int>();
            _currentPrice = new RealtimeInfoString<int>();
            _timestamp = new RealtimeInfoString<int>();
            _scale = stock==null? 0.01 : stock.Profile.PriceScale;
        }
        public void PseudoReset(int price, double scale) {
            _scale = scale;
            _currentPrice.SetValue(price);
            _lastClose.SetValue(price);
        }

        public RealtimeInfoString<int> LastCloseR {
            get {
                return _lastClose;
            }
        }
        public RealtimeInfoString<int> CurrentPriceR {
            get {
                return _currentPrice;
            }
        }
        public RealtimeInfoString<int> Timestamp {
            get {
                return _timestamp;
            }
        }
        public override bool FireEventToSubscriberManager() {
            BellagioRoot.DataSubscriberManager.ReserveExecByStock(_stock, SubscribeDataKind.Ticks);
            return false;
        }
        public override void FireErrorEventToSubscriberManager(string msg) {
            BellagioRoot.DataSubscriberManager.ReserveErrorByStock(_stock, SubscribeDataKind.Ticks, msg);
        }


        public double RealLastClose {
            get {
                return ((double)_lastClose.Value) * _scale;
            }
        }
        public double RealCurrentPrice {
            get {
                return ((double)_currentPrice.Value) * _scale;
            }
        }

    }


    public interface ILightIntraDayTradeSource : IPrimaryStreamDataBase {
        AsyncOpResult OpenLightIntraDayTrade(Stock stock, ref string message);
        void CloseLightIntraDayTrade(Stock stock);
    }
    public interface ILightIntraDayTradeHost : IPrimaryStreamDataHostBase {
        void SetInitialData(Stock stock, int prevClose);
        void OnUpdateData(Stock stock, int timestamp, int price);
    }

    public class LightIntraDayTradeInitializer : StockPrimaryDataInitializer {
        private LightIntraDayTrade _target;
        public LightIntraDayTradeInitializer(LightIntraDayTrade lt)
            : base(lt) {
            _target = lt;
        }
        public override bool IsCompleted {
            get {
                return true;
            }
        }
        public override void CompleteInitialization() {
        }
    }

    public class LightIntraDayTradeProvider : StockDataProviderBaseT<LightIntraDayTrade, LightIntraDayTradeInitializer>, ILightIntraDayTradeHost {
        private ILightIntraDayTradeSource _dataSource;

        public LightIntraDayTradeProvider() {
        }
        protected override LightIntraDayTrade CreateValue(Stock stock) {
            return new LightIntraDayTrade(stock);
        }
        protected override AsyncOpResult OpenValue(LightIntraDayTradeInitializer initializer, ref string error) {
            EnsureDataSource();
            AsyncOpResult r = _dataSource.OpenLightIntraDayTrade(initializer.Stock, ref error);
            return r;
        }
        protected override LightIntraDayTradeInitializer CreateInitializer(LightIntraDayTrade value) {
            return new LightIntraDayTradeInitializer(value);
        }
        protected override void CloseValue(Stock stock, LightIntraDayTrade value) {
            _dataSource.CloseLightIntraDayTrade(stock);
        }

        void ILightIntraDayTradeHost.SetInitialData(Stock stock, int prevClose) {
            LightIntraDayTrade v = base.GetTagOrNull(stock).value;
            v.LastCloseR.SetValue(prevClose);
            base.CompleteLoading(stock);
            //BellagioRoot.DataSubscriberManager.ReserveExecByStock(stock, SubscribeDataKind.Ticks);
        }

        void ILightIntraDayTradeHost.OnUpdateData(Stock stock, int timestamp, int price) {
            LightIntraDayTrade v = base.Lookup(stock);
            v.CurrentPriceR.SetValue(price);
            v.Timestamp.SetValue(timestamp);
            v.FireDataUpdateEvent();
            //BellagioRoot.DataSubscriberManager.ReserveExecByStock(stock, SubscribeDataKind.Ticks);
        }


        

        IAdaptable IAdaptable.GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        private void EnsureDataSource() {
            if(_dataSource!=null) return;
            _dataSource = BellagioRoot.DataSourceHost.ExternalEnvironment.IndexDataSource;
            _dataSource.Prepare(this);
        }

    }

}
