﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Threading.Tasks;
using System.Drawing;
using System.Xml.Serialization;
using System.IO;
using FDK;

namespace StrokeStyleT
{
	[Serializable]
	public class CConfig : ISerializableXML		// XML化対象クラスは public であり、StrokeStyleT.ISerializableXML を実装すること。
	{
		// XML化対象。

		public volatile string SSTVersion = String.Format( "{0}.{1}.{2}.{3}", Global.nメジャー番号, Global.nマイナー番号, Global.nリビジョン番号, Global.nビルド番号 );

		public volatile string KitName = "Default";
		public volatile string ThemeName = "Default";

		public volatile float ScrollSpeed = 2.0f;

		public volatile bool VSyncWait = true;
		public volatile bool Windowed = true;
	
		public volatile bool AutoPlayLeftCymbal = false;
		public volatile bool AutoPlayFootPedal = false;
		public volatile bool AutoPlayHiHat = false;
		public volatile bool AutoPlaySnare = false;
		public volatile bool AutoPlayTom1 = false;
		public volatile bool AutoPlayBass = false;
		public volatile bool AutoPlayTom2 = false;
		public volatile bool AutoPlayTom3 = false;
		public volatile bool AutoPlayRightCymbal = false;

		public volatile bool ChinaLeft = false;		// China は左側？
		public volatile bool RideLeft = false;		// Ride は左側？
		public volatile bool SplashLeft = true;		// Sprash は左側？

		public volatile Eコンボ表示位置 ComboPosition = Eコンボ表示位置.LEFT;

		public volatile int PerfectRange = int.Parse( Properties.Resources.DEFAULT_PERFECT_RANGE );
		public volatile int GreatRange = int.Parse( Properties.Resources.DEFAULT_GREAT_RANGE );
		public volatile int GoodRange = int.Parse( Properties.Resources.DEFAULT_GOOD_RANGE );
		public volatile int PoorRange = int.Parse( Properties.Resources.DEFAULT_POOR_RANGE );

		public volatile bool HitSound = false;

		/// <summary>対象となる動画ファイルの拡張子リスト。</summary>
		public volatile List<string> SongFileExtensions = new List<string>();

		/// <summary>SongList.xml のパス。読み込み直後にマクロを展開し、保存直前にマクロに変換する。</summary>
		public volatile List<string> SongListFiles = new List<string>();

		/// <summary>KitList.xml のパス。読み込み直後にマクロを展開し、保存直前にマクロに変換する。</summary>
		public volatile List<string> KitListFiles = new List<string>();

		/// <summary>演奏時のフレーム背景の不透明度。（完全透明:0～100:完全不透明）</summary>
		public volatile int PlayFrameOpacity = 60;


		// XML化対象外。

		[XmlIgnore]
		internal bool b全オートである
		{
			get
			{
				return (
					this.AutoPlayLeftCymbal &&
					this.AutoPlayFootPedal &&
					this.AutoPlayHiHat &&
					this.AutoPlaySnare &&
					this.AutoPlayTom1 &&
					this.AutoPlayBass &&
					this.AutoPlayTom2 &&
					this.AutoPlayTom3 &&
					this.AutoPlayRightCymbal );
			}
		}

		[XmlIgnore]
		internal volatile Dictionary<E演奏レーン, bool> dicAutoPlay = new Dictionary<E演奏レーン, bool>() {
			{ E演奏レーン.LeftCymbal, false },
			{ E演奏レーン.FootPedal, false },
			{ E演奏レーン.HiHat, false },
			{ E演奏レーン.Snare, false },
			{ E演奏レーン.Tom1, false },
			{ E演奏レーン.Bass, false },
			{ E演奏レーン.Tom2, false },
			{ E演奏レーン.Tom3, false },
			{ E演奏レーン.RightCymbal, false },
			{ E演奏レーン.Unknown, false },
		};

		[XmlIgnore]
		internal volatile Dictionary<E判定, int> dicRange = new Dictionary<E判定, int>() {
			{ E判定.PERFECT, int.Parse( Properties.Resources.DEFAULT_PERFECT_RANGE ) },
			{ E判定.GREAT, int.Parse( Properties.Resources.DEFAULT_GREAT_RANGE ) },
			{ E判定.GOOD, int.Parse( Properties.Resources.DEFAULT_GOOD_RANGE ) },
			{ E判定.POOR, int.Parse( Properties.Resources.DEFAULT_POOR_RANGE ) },
			{ E判定.MISS, 0 },
			{ E判定.AUTO, 0 },
		};

		// XML化対象クラスはパラメータなしの public コンストラクタを持つこと。
		public CConfig()
		{
		}

		internal static CConfig t読み込む( string fileName )
		{
			CConfig config = null;
			bool bFileFound = true;

			// Config.xml ファイルの読み込み
	
			config = CXMLFile<CConfig>.t読み込む( fileName );


			// Config内容の妥当性確認、修正

			#region [ SongExtentions がないならデフォルト拡張子を付与する。]
			//-----------------
			if( config.SongFileExtensions == null ||
				config.SongFileExtensions.Count == 0 )
			{
				config.SongFileExtensions = CConfig.listデフォルト拡張子;
			}
			//-----------------
			#endregion

			if( bFileFound )
			{
				#region [ SongExtensions の先頭が '.' でないなら付与する。また、無効な文字列は排除する。]
				//-----------------
				for( int i = 0; i < config.SongFileExtensions.Count; i++ )
				{
					string ext = config.SongFileExtensions[ i ].Trim();

					if( ext[ 0 ] != '.' )
						config.SongFileExtensions[ i ] = '.' + ext;
				}

				// 空文字列はすべて除去する。

				config.SongFileExtensions.RemoveAll( ( path ) => { return string.IsNullOrEmpty( path ); } );
				//-----------------
				#endregion

				#region [ AutoPlay* を dicAutoPlay に反映する。]
				//-----------------
				config.dicAutoPlay = new Dictionary<E演奏レーン, bool>() {
					{ E演奏レーン.LeftCymbal, config.AutoPlayLeftCymbal },
					{ E演奏レーン.FootPedal, config.AutoPlayFootPedal },
					{ E演奏レーン.HiHat, config.AutoPlayHiHat },
					{ E演奏レーン.Snare, config.AutoPlaySnare },
					{ E演奏レーン.Tom1, config.AutoPlayTom1 },
					{ E演奏レーン.Bass, config.AutoPlayBass },
					{ E演奏レーン.Tom2, config.AutoPlayTom2 },
					{ E演奏レーン.Tom3, config.AutoPlayTom3 },
					{ E演奏レーン.RightCymbal, config.AutoPlayRightCymbal },
					{ E演奏レーン.Unknown, false },
				};
				//-----------------
				#endregion
				#region [ Range* を dicRange に反映する。]
				//-----------------
				config.dicRange = new Dictionary<E判定, int>() {
						{ E判定.PERFECT, config.PerfectRange },
						{ E判定.GREAT, config.GreatRange },
						{ E判定.GOOD, config.GoodRange },
						{ E判定.POOR, config.PoorRange },
						{ E判定.MISS, 0 },
						{ E判定.AUTO, 0 },
					};
				//-----------------
				#endregion

				#region [ SongListFiles 要素のマクロを展開し、パスが間違っている要素や、存在しない要素を削除する。]
				//-----------------
				for( int i = config.SongListFiles.Count - 1; i >= 0; i-- )
				{
					// 空文字列なら除去する。

					if( string.IsNullOrEmpty( config.SongListFiles[ i ] ) )
					{
						config.SongListFiles.RemoveAt( i );
						continue;
					}


					// パスが間違っているものや、存在していない場合には除去する。

					try
					{
						config.SongListFiles[ i ] = Folder.tファイルパス内のマクロを展開する( config.SongListFiles[ i ] );
						config.SongListFiles[ i ] = Folder.tパスの正当性を確認のうえ絶対パスに変換して返す( config.SongListFiles[ i ].Trim(), Folder.stgシステムフォルダ );
					}
					catch
					{
						// 例外が発生したら無効な文字列ということなので無視する。
						config.SongListFiles.RemoveAt( i );
						continue;
					}
					if( !File.Exists( config.SongListFiles[ i ] ) )	// 引数がファイルじゃなくフォルダである場合にも false が返る。
					{
						// SongListファイルが存在しないので無視する。
						config.SongListFiles.RemoveAt( i );
						continue;
					}

				}
				//-----------------
				#endregion
				#region [ KitListFiles 要素のマクロを展開し、パスが間違っている要素や、存在しない要素を削除する。]
				//-----------------
				for( int i = config.KitListFiles.Count - 1; i >= 0; i-- )
				{
					// 空文字列なら除去する。

					if( string.IsNullOrEmpty( config.KitListFiles[ i ] ) )
					{
						config.KitListFiles.RemoveAt( i );
						continue;
					}


					// パスが間違っているものや、存在していない場合には除去する。

					try
					{
						config.KitListFiles[ i ] = Folder.tファイルパス内のマクロを展開する( config.KitListFiles[ i ] );
						config.KitListFiles[ i ] = Folder.tパスの正当性を確認のうえ絶対パスに変換して返す( config.KitListFiles[ i ].Trim(), Folder.stgシステムフォルダ );
					}
					catch
					{
						// 例外が発生したら無効な文字列ということなので無視する。
						config.KitListFiles.RemoveAt( i );
						continue;
					}
					if( !File.Exists( config.KitListFiles[ i ] ) )			// 引数がファイルじゃなくフォルダである場合にも false が返る。
					{
						// KitListファイルが存在しないので無視する。
						config.KitListFiles.RemoveAt( i );
						continue;
					}
				}
				//-----------------
				#endregion

				#region [ ScrollSpeed が 0.0 以下なら 1.0 にする。]
				//-----------------
				if( config.ScrollSpeed <= 0.0f )
					config.ScrollSpeed = 1.0f;
				//-----------------
				#endregion
			}

			#region [ SongListFilesが1つもない場合、システムフォルダにSongList.xmlがあればそれを加える。]
			//-----------------
			if( config.SongListFiles.Count == 0 )
			{
				string path = Path.Combine( Folder.stgシステムフォルダ, Properties.Resources.XMLNAME_SONG_LIST );

				if( File.Exists( path ) )
					config.SongListFiles.Add( path );
			}
			//-----------------
			#endregion
			#region [ KitListFilesが1つもない場合、システムフォルダにKitList.xmlがあればそれを加える。]
			//-----------------
			if( config.KitListFiles.Count == 0 )
			{
				string path = Path.Combine( Folder.stgシステムフォルダ, @"DrumsKits\Default" , Properties.Resources.XMLNAME_KITLIST );

				if( File.Exists( path ) )
					config.KitListFiles.Add( path );
			}
			//-----------------
			#endregion

			return config;
		}
		internal void t保存する( string fileName )
		{
			#region [ SongListFiles 要素のシステム/ユーザフォルダパスを、マクロに置換する。]
			//-----------------
			for( int i = 0; i < this.SongListFiles.Count; i++ )
			{
				if( !string.IsNullOrEmpty( this.SongListFiles[ i ] ) )
					this.SongListFiles[ i ] = Folder.tファイルパスをマクロ付きパスに変換する( this.SongListFiles[ i ] );
			}
			//-----------------
			#endregion
			#region [ KitListFiles 要素のシステム/ユーザフォルダパスを、マクロに置換する。]
			//-----------------
			for( int i = 0; i < this.KitListFiles.Count; i++ )
			{
				if( !string.IsNullOrEmpty( this.KitListFiles[ i ] ) )
					this.KitListFiles[ i ] = Folder.tファイルパスをマクロ付きパスに変換する( this.KitListFiles[ i ] );
			}
			//-----------------
			#endregion

			CXMLFile<CConfig>.t保存する( this, fileName );


			// ここから継続して使う場合があるので、マクロを元の状態に戻しておく。

			#region [ SongListFiles 要素のシステム/ユーザフォルダパスのマクロを展開する。]
			//-----------------
			for( int i = 0; i < this.SongListFiles.Count; i++ )
			{
				if( !string.IsNullOrEmpty( this.SongListFiles[ i ] ) )
					this.SongListFiles[ i ] = Folder.tファイルパス内のマクロを展開する( this.SongListFiles[ i ] );
			}
			//-----------------
			#endregion
			#region [ KitListFiles 要素のシステム/ユーザフォルダパスのマクロを展開する。]
			//-----------------
			for( int i = 0; i < this.KitListFiles.Count; i++ )
			{
				if( !string.IsNullOrEmpty( this.KitListFiles[ i ] ) )
					this.KitListFiles[ i ] = Folder.tファイルパス内のマクロを展開する( this.KitListFiles[ i ] );
			}
			//-----------------
			#endregion
		}

		internal void tすべてのAutoPlayを設定する( bool b設定値 )
		{
			this.AutoPlayLeftCymbal =
				this.AutoPlayFootPedal =
				this.AutoPlayHiHat =
				this.AutoPlaySnare =
				this.AutoPlayTom1 =
				this.AutoPlayBass =
				this.AutoPlayTom2 =
				this.AutoPlayTom3 =
				this.AutoPlayRightCymbal = b設定値;
		}

		public static volatile List<string> listデフォルト拡張子 =	// SSTFEditor でも使うので public。
			new List<string>() { 
//				".mp3", ".wav",	".ogg",								// 音楽ファイルが前		→ 音楽ファイル単体対応は今後の課題(2013.2.13)
				".avi", ".flv", ".mp4", ".wmv", ".mpg", ".mpeg"		// 動画ファイルが後
			};

		// ISerializableXML の実装。

		public void t規定の初期化()
		{
			// 特になし
		}
	}
}
