﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Xml;
using SharpDX;

namespace FDK
{
	public static class Utilities
	{
		/// <summary>
		///		Dispose して、null を代入する。
		/// </summary>
		/// <param name="obj">
		///		IDisposable を実装するオブジェクト。
		///	</param>
		public static void 解放する<T>( ref T obj ) where T : IDisposable
		{
			( obj as IDisposable )?.Dispose();
			obj = default( T );
		}

		/// <summary>
		///		レンダーターゲットに対して描画処理を実行する。
		/// </summary>
		/// <remarks>
		///		描画処理は、レンダーターゲットの BeginDraw() と EndDraw() の間で行われることを保証する。
		///		描画処理中に例外が発生しても EndDraw() の呼び出しは保証する。
		/// </remarks>
		/// <param name="target">
		///		レンダーターゲット。
		///	</param>
		/// <param name="描画処理">
		///		BeginDraw() と EndDraw() の間で行う処理。
		///	</param>
		public static void D2DBatchDraw( SharpDX.Direct2D1.RenderTarget target, Action 描画処理 )
		{
			try
			{
				target.BeginDraw();
				描画処理();
			}
			finally
			{
				// SharpDX の EndDraw() では HRESULT を捨ててしまうので、デバイスロストのチェックは、ここではできない。
				target.EndDraw();
			}
		}

		/// <summary>
		///		深度から射影行列（定数）を計算して返す。
		///		Direct2D 用。
		/// </summary>
		public static Matrix D2DPerspectiveProjection( float depth )
		{
			var mat = Matrix.Identity;
			mat.M34 = ( 0 != depth ) ? -( 1.0f / depth ) : 0.0f;
			return mat;
		}

		public static int 最大公約数を返す( int m, int n )
		{
			if( ( 0 >= m ) || ( 0 >= n ) )
				throw new FDKException( "引数に0以下の数は指定できません。" );

			// ユーグリッドの互除法
			int r;
			while( ( r = m % n ) != 0 )
			{
				m = n;
				n = r;
			}

			return n;
		}

		public static int 最小公倍数を返す( int m, int n )
		{
			if( ( 0 >= m ) || ( 0 >= n ) )
				throw new FDKException( "引数に0以下の数は指定できません。" );

			return ( m * n / Utilities.最大公約数を返す( m, n ) );
		}

		public static double 変換_100ns単位からsec単位へ( long 数値100ns )
		{
			return 数値100ns / 10_000_000.0;
		}

		public static long 変換_sec単位から100ns単位へ( double 数値sec )
		{
			return (long) ( 数値sec * 10_000_000.0 + 0.5 ); // +0.5 で四捨五入できる。
		}

		/// <summary>
		///		このメソッドの 呼び出し元のメソッド名 を返す。デバッグログ用。
		/// </summary>
		public static string 現在のメソッド名
		{
			get
			{
				// 1つ前のスタックフレームを取得。
				var prevFrame = new StackFrame( skipFrames: 1, fNeedFileInfo: false );

				var クラス名 = prevFrame.GetMethod().ReflectedType.ToString();
				var メソッド名 = prevFrame.GetMethod().Name;

				return $"{クラス名}.{メソッド名}()";
			}
		}

		/// <summary>
		///		画像ファイルからシェーダリソースビューを作成して返す。
		/// </summary>
		/// <remarks>
		///		（参考: http://qiita.com/oguna/items/c516e09ee57d931892b6 ）
		/// </remarks>
		public static (SharpDX.Direct3D11.ShaderResourceView srv, Size2F viewSize, SharpDX.Direct3D11.Texture2D texture) CreateShaderResourceViewFromFile(
			SharpDX.Direct3D11.Device d3dDevice,
			SharpDX.Direct3D11.BindFlags bindFlags,
			string 画像ファイルパス )
		{
			var 出力 = (
				srv:(SharpDX.Direct3D11.ShaderResourceView) null,
				viewSize: new Size2F(0,0), 
				texture: (SharpDX.Direct3D11.Texture2D) null );

			using( var image = new System.Drawing.Bitmap( 画像ファイルパス ) )
			{
				var 画像の矩形 = new System.Drawing.Rectangle( 0, 0, image.Width, image.Height );

				using( var bitmap = image.Clone( 画像の矩形, System.Drawing.Imaging.PixelFormat.Format32bppArgb ) )
				{
					var ロック領域 = bitmap.LockBits( 画像の矩形, System.Drawing.Imaging.ImageLockMode.ReadOnly, bitmap.PixelFormat );
					var dataBox = new[] { new DataBox( ロック領域.Scan0, bitmap.Width * 4, bitmap.Height ) };
					var textureDesc = new SharpDX.Direct3D11.Texture2DDescription() {
						ArraySize = 1,
						BindFlags = bindFlags,
						CpuAccessFlags = SharpDX.Direct3D11.CpuAccessFlags.None,
						Format = SharpDX.DXGI.Format.B8G8R8A8_UNorm,
						Height = bitmap.Height,
						Width = bitmap.Width,
						MipLevels = 1,
						OptionFlags = SharpDX.Direct3D11.ResourceOptionFlags.None,
						SampleDescription = new SharpDX.DXGI.SampleDescription( 1, 0 ),
						Usage = SharpDX.Direct3D11.ResourceUsage.Default
					};
					var texture = new SharpDX.Direct3D11.Texture2D( d3dDevice, textureDesc, dataBox );
					bitmap.UnlockBits( ロック領域 );
					出力.srv = new SharpDX.Direct3D11.ShaderResourceView( d3dDevice, texture );
					出力.texture = texture;
				}

				出力.viewSize = new Size2F( 画像の矩形.Width, 画像の矩形.Height );
			}

			return 出力;
		}

		/// <summary>
		///		空のテクスチャとそのシェーダーリソースビューを作成し、返す。
		/// </summary>
		public static (SharpDX.Direct3D11.ShaderResourceView srv, SharpDX.Direct3D11.Texture2D texture) CreateShaderResourceView(
			SharpDX.Direct3D11.Device d3dDevice,
			SharpDX.Direct3D11.BindFlags bindFlags,
			Size2 サイズdpx )
		{
			var textureDesc = new SharpDX.Direct3D11.Texture2DDescription() {
				ArraySize = 1,
				BindFlags = bindFlags,
				CpuAccessFlags = SharpDX.Direct3D11.CpuAccessFlags.None,
				Format = SharpDX.DXGI.Format.B8G8R8A8_UNorm,
				Height = サイズdpx.Height,
				Width = サイズdpx.Width,
				MipLevels = 1,
				OptionFlags = SharpDX.Direct3D11.ResourceOptionFlags.None,
				SampleDescription = new SharpDX.DXGI.SampleDescription( 1, 0 ),
				Usage = SharpDX.Direct3D11.ResourceUsage.Default
			};

			var 出力 = ( srv: (SharpDX.Direct3D11.ShaderResourceView) null, texture: (SharpDX.Direct3D11.Texture2D) null );
			出力.texture = new SharpDX.Direct3D11.Texture2D( d3dDevice, textureDesc );
			出力.srv = new SharpDX.Direct3D11.ShaderResourceView( d3dDevice, 出力.texture );

			return 出力;
		}

		/// <summary>
		///		DataContract オブジェクトをシリアル化してXMLファイルに保存する。
		///		失敗すれば例外を発出。
		/// </summary>
		public static void 保存する( object 保存するオブジェクト, string XMLファイルパス )
		{
			var serializer = new DataContractSerializer( 保存するオブジェクト.GetType() );

			var settings = new XmlWriterSettings() {
				Encoding = new System.Text.UTF8Encoding( false ),   // UTF-8, BOMなし
				NewLineChars = Environment.NewLine,                 // 改行コード
				NewLineHandling = NewLineHandling.Replace,          // 改行コードを統一（上書き）
				Indent = true,                                      // 改行＆インデントを使う（falseにすると改行もされなくなる）
			};

			using( var writer = XmlWriter.Create( XMLファイルパス, settings ) )
			{
				serializer.WriteObject( writer, 保存するオブジェクト );
			}
		}

		/// <summary>
		///		XMLファイルを逆シリアル化して、DataContract オブジェクトを生成する。
		///		失敗すれば例外を発出。
		/// </summary>
		/// <remarks>
		///		DataContract 属性を持つ型の逆シリアル時には、コンストラクタは呼び出されないので注意。
		///		各メンバは 0 または null になるが、それがイヤなら OnDeserializing コールバックを用意して、逆シリアル化の前に初期化すること。
		/// </remarks>
		public static T 復元する<T>( string XMLファイルパス ) where T : class, new()
		{
			T dataContract;

			var serializer = new DataContractSerializer( typeof( T ) );

			using( var reader = XmlReader.Create( XMLファイルパス ) )
			{
				dataContract = ( T ) serializer.ReadObject( reader );
			}

			return dataContract;
		}

		/// <summary>
		///		XMLファイルを逆シリアル化して、DataContract オブジェクトを生成する。
		/// </summary>
		/// <remarks>
		///		復元に失敗すれば、新規にインスタンスを生成し、XML ファイルに保存してから返す。
		///		DataContract 属性を持つ型の逆シリアル時には、コンストラクタは呼び出されないので注意。
		///		各メンバは 0 または null になるが、それがイヤなら OnDeserializing コールバックを用意して、逆シリアル化の前に初期化すること。
		/// </remarks>
		public static T 復元または新規作成する<T>( string XMLファイルパス ) where T : class, new()
		{
			T obj;

			if( File.Exists( XMLファイルパス ) )
			{
				try
				{
					obj = Utilities.復元する<T>( XMLファイルパス );
					Log.Info( $"XMLファイルから{typeof( T ).Name}を復元しました。[{XMLファイルパス}]" );
				}
				catch( Exception e )
				{
					Log.WARNING( $"XMLファイルからの復元に失敗しました。初期状態で生成します。[{e.Message}][{XMLファイルパス}]" );
					obj = new T();
				}
			}
			else
			{
				Log.WARNING( $"{typeof(T).Name}ファイルが存在しません。新規作成します。[{XMLファイルパス}]" );
				obj = new T();

				Utilities.保存する( obj, XMLファイルパス );
			}

			return obj;
		}
	}
}