# -*- coding: utf-8 -*-
class User < ActiveRecord::Base
  has_and_belongs_to_many :projects

  attr_protected :is_admin

  validates :email, presence: true, uniqueness: true
  validates_format_of :email,
    with: /^([^@\s]+)@((?:[-a-z0-9]+\.)+[a-z]{2,})$/i,
    message: ' must be valid'
  has_secure_password
  validates :handle, presence: true
  validates_length_of :handle, :minimum => 1

  before_validation :create_handle_from_email
  before_create :appoint_first_user_to_admin, :set_default_config
  after_create :add_inbox_project
  after_destroy :ensure_an_admin_remains

  private

  def create_handle_from_email
    # email の '@' までの文字列を handle とする
    if self[:handle].nil? and not self.email.nil?
      self[:handle] = self.email.split("@", 2)[0]
    end
  end

  def appoint_first_user_to_admin
    # 最初のユーザの場合、管理者の権限を持たせる
    if User.count == 0
      self[:is_admin] = true
    end
  end

  def set_default_config
    # en のときは、システム設定で初期化する
    # 既に en 以外で登録されている場合は、その設定を用いるようにする
    if self.locale == 'en'
      # 設定を DefaultConfig から読み出して登録する
      default = DefaultConfig.first

      self.locale = default.locale
    end
  end

  def add_inbox_project
    # ユーザ作成時に Inbox プロジェクトとチケットを追加する
    project = Project.create(title: 'Inbox',
                             description: 'Your private project.',
                             is_group: false, is_open: false,
                             created_by_user_id: self.id)

    # locale に応じたメッセージを作成する
    message_1st = {}
    message_2nd = {}

    message_1st[:title] = 'Thank you for your registration'
    message_1st[:comment] = 'First, resolve this ticket.'
    message_2nd[:title] = "Let's create your own project"
    message_2nd[:comment] = 'Go to the Project page, and make your project.'

    if self.locale == :ja
      message_1st[:title] = 'ユーザ登録ありがとうございます！'
      message_1st[:comment] = '最初に、このチケットを Resolved にしてみましょう。'
      message_2nd[:title] = '自分のプロジェクトを作りましょう'
      message_2nd[:comment] = '"プロジェクト" ページから、自分用のプロジェクトを作りましょう。'
    end

    Ticket.create(title: message_1st[:title],
                  comment: message_1st[:comment],
                  project_id: project.id,
                  state_id: 1, priority_id: 1, workload_id: 1,
                  assigned_user_id: self.id,
                  updated_by_user_id: self.id)
    Ticket.create(title: message_2nd[:title],
                  comment: message_2nd[:comment],
                  project_id: project.id,
                  state_id: 1, priority_id: 2, workload_id: 1,
                  assigned_user_id: self.id,
                  updated_by_user_id: self.id)
  end

  def ensure_an_admin_remains
    # 最後の管理者は削除されないようにする
    # TODO 削除されてしまうのを修正する
    if User.where(is_admin: true).count.zero?
      raise t('errors.messages.cant_delete_last_admin')
    end
  end
end
