//using System;
//using System.Collections.Generic;
//using System.Text;
//using SlothLib.Web.Search;
//using System.Text.RegularExpressions;
//using SlothLib.Web.Search.Google;

package slothLib.web.search;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.rpc.ServiceException;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;
import GoogleSearch.GoogleSearchResult;
import GoogleSearch.GoogleSearchServiceLocator;
import GoogleSearch.ResultElement;

/**
     * GoogleAPIを使った検索を担うクラス
 */
    public class GoogleSOAPSearchAPI implements IWebSearch
    {
        
        // privateフィールド

        // APIキー
        private String apiKey;

        // 検索対象とする言語language restrict
        private String defaultLr;

        
        public static final String KEY_PROP = "slothlib.web.search.GoogleSOAPSearchAPI.key";

        static {slothLib.SlothLib.init();}
        
        


        // コンストラクタ

        /**
         * コンストラクタ　検索対象の言語を指定する
         * @param apiKey 検索クエリ
         * @param defaultLanguage 検索対象とする言語
         */
        public GoogleSOAPSearchAPI(String apiKey, Language defaultLanguage)
        {

            this.apiKey = apiKey;
            switch (defaultLanguage)
            {
                case All:
                    this.defaultLr = "";
                    break;
                case Japanese:
                    this.defaultLr = "lang_ja";
                    break;
                case English:
                    this.defaultLr = "lang_en";
                    break;

            }
        }

        /**
         * コンストラクタ　日本語のページを検索対象とする
         * @param apiKey ライセンスキー
         */
        public GoogleSOAPSearchAPI(String apiKey)
        {
        	this(apiKey, Language.Japanese);
        }


        /**
         * コンストラクタ　検索対象の言語を細かく指定する
         * @param apiKey ライセンスキー
         * @param defaultLr 検索対象とする言語(lang_ja等の形式で入力してください)
         */
        public GoogleSOAPSearchAPI(String apiKey, String defaultLr)
        {
            this.apiKey = apiKey;
            this.defaultLr = defaultLr;
        }


        
        /**
         * コンストラクタ 
         * プロパティ slothlib.web.search.GoogleSOAPSearchAPI.key の値をキーとする．
         */
        public GoogleSOAPSearchAPI() throws WebSearchException
        {
        	String tmp = System.getProperty(KEY_PROP);
        	if (tmp == null)
        		throw new WebSearchException("cannot find property: " + KEY_PROP);
        	this.apiKey = tmp.trim();
        }
        
        


        // DoSearch

        /**
         * GoogleAPIを用いて検索を実行する
         * @param query 検索クエリ
         * @param maxResult 検索結果取得数
         * @return GoogleSearchAPIResult型の検索結果
         */
        public GoogleSOAPSearchAPIResult doSearch(String query, int maxResult) throws WebSearchException
        {
            try {
				return doGoogleSearchAPI(query, maxResult, this.defaultLr);
			} catch (RemoteException e) {
				throw new WebSearchException(e);
			} catch (ServiceException e) {
				throw new WebSearchException(e);
			}
        }



        private String modQuery(String query){
			try {
    			ByteArrayOutputStream bos = new ByteArrayOutputStream();
				OutputStreamWriter writer = new OutputStreamWriter(bos, "UTF-8");
				writer.write(query);
				writer.close();
				byte [] tmp = bos.toByteArray();
				char [] ctmp = new char[tmp.length]; 
				for (int index = 0; index < tmp.length; index++)
					ctmp[index] = (char)(tmp[index] & 0xff);
				String tmpStr = new String(ctmp);
				System.err.println(tmpStr);
				return tmpStr;
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
			return query;
        }
        
        
        // DoGoogleSearchAPI

        /**
         * 検索を実行
         * @param query クエリ
         * @param maxResult 最大結果数
         * @param lr language restrict 
         * @return 検索結果
         */
        private GoogleSOAPSearchAPIResult doGoogleSearchAPI(
            //String apiKey,
            String query,
            int maxResult,
            String lr
            //以下の引数の対応は、時間と余裕があったらで
            //boolean filter,	【未実装】
            //String restrict,【未実装】
            //boolean safeSearch,【未実装】
            //String ie,【未実装】
            //String oe【未実装】
            ) throws ServiceException, RemoteException
        {
            // 結果統合用
            List<GoogleSOAPElement> ResultElementList = new ArrayList<GoogleSOAPElement>();

            // Create a Google Search object
            
            
            GoogleSearch.GoogleSearchPort s = (new GoogleSearchServiceLocator()).getGoogleSearchPort(); 
            long totalResult = 0;
            // 結果を10ずつに分けて取得
            for (int i = 0; i <= maxResult - 10; i = i + 10)
            {
                int range = maxResult - i;
                if (range > 10)
                {
                    range = 10;
                }
    			String codedQuery = modQuery(query);

                GoogleSearchResult r = s.doGoogleSearch(apiKey, codedQuery,
                    i, range, false, "", false, lr, "", "");

                int rank = 0;
                for (ResultElement re: r.getResultElements())
                {
                    rank++;
                    String tmp;
                    re.setTitle(re.getTitle().replaceAll("<[^>]*>", ""));
                    re.setSnippet(re.getSnippet().replaceAll("<[^>]*>", "")); 

                    GoogleSOAPElement sere = new GoogleSOAPElement(
                        rank, re.getURL(), re.getTitle(), re.getSnippet());
                    ResultElementList.add(sere);
                }
                totalResult = r.getEstimatedTotalResultsCount();
            }

            // 返す結果を作成してさようなら。
            return new GoogleSOAPSearchAPIResult(query, totalResult, ResultElementList.toArray(new GoogleSOAPElement[0]));

        }



        // OpenResultPage

        /**
         * 検索結果をブラウザで表示する
         * @param query 検索クエリ
         * @param language 検索対象とする言語
         */
        public static void openResultPage(String query, Language language)
        {
            String openURL = "http://www.google.co.jp/search?num=100&hl=ja&q=" ;
			try {
				openURL +=  URLEncoder.encode(query, "UTF8"); 
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
				openURL += query;
			}
            //System.Web.HttpUtility.UrlEncode(query, System.Text.Encoding.UTF8);
            switch (language)
            {
                case All:
                    break;
                case English:
                    openURL += "&lr=lang_en";
                    break;
                case Japanese:
                    openURL += "&lr=lang_ja";
                    break;
            }
            PortUtil.openUrl(openURL);
        }

        /**
         * 検索結果をブラウザで表示する　日本語のページを検索対象とする
         * @param query 検索クエリ
         */
        public static void OpenResultPage(String query)
        {
            openResultPage(query, Language.Japanese);
        }



        // 列挙型

        /**
         * 検索対象とする言語
         */
        public enum Language
        {


            /**
             * 日本語 デフォルト
             */
            Japanese,

            /**
             * 英語
             */
            English,

            /**
             * 全ての言語
             */
            All,
        }



//        //#region IWebSearch メンバ
//
//        /// <summary>
//        /// Google検索エンジンで検索する
//        /// </summary>
//        /// <param name="query">検索クエリ</param>
//        /// <param name="resultNum">取得件数</param>
//        /// <returns>検索結果</returns>
//        IWebSearchResult IWebSearch.DoSearch(String query, int resultNum)
//        {
//            return DoSearch(query, resultNum);
//        }
//
//        //#endregion
//
//        //#region ISearch メンバ
//
//        /// <summary>
//        /// Google検索エンジンで検索する
//        /// </summary>
//        /// <param name="query">検索クエリ</param>
//        /// <param name="resultNum">取得件数</param>
//        /// <returns>検索結果</returns>
//        ISearchResult ISearch.DoSearch(String query, int resultNum)
//        {
//            return DoSearch(query, resultNum);
//        }
//
//        //#endregion
    }

