package slothLib.NLP;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;;

/// <summary>
/// 文字列から文字列に変換するフィルタを集めたクラス
/// </summary>
/// <remarks>
/// 
/// [TODO]コメント書く

public class Filter
{

	/// <summary>
	/// numKanjiOneRegexでマッチする各文字に対応するローマ数字のマップ
	/// </summary>
	private static final Map<String, String> mapNumKanjiToNumRoman = new HashMap<String, String>();
/// <summary>
	/// staticコンストラクタ
	/// </summary>
	static
	{
		mapNumKanjiToNumRoman.put("〇", "0");
		mapNumKanjiToNumRoman.put("一", "1");
		mapNumKanjiToNumRoman.put("二", "2");
		mapNumKanjiToNumRoman.put("三", "3");
		mapNumKanjiToNumRoman.put("四", "4");
		mapNumKanjiToNumRoman.put("五", "5");
		mapNumKanjiToNumRoman.put("六", "6");
		mapNumKanjiToNumRoman.put("七", "7");
		mapNumKanjiToNumRoman.put("八", "8");
		mapNumKanjiToNumRoman.put("九", "9");
	}
	
	

	/// <summary>
	/// ToTitleCaseを利用するためのTextInfo
	/// </summary>
	// private static final TextInfo textInfo = Thread.CurrentThread.CurrentCulture.TextInfo;
	
	/// <summary>
	/// 全角にするためのMatchEvaluator
	/// </summary>
	private static final MatchEvaluator toWideMatchEvaluator = 
		new MatchEvaluator(){public String eval(String in){ return matchToWide(in);}};
	
	/// <summary>
	/// 半角にするためのMatchEvaluator
	/// </summary>
	private static final MatchEvaluator toNarrowMatchEvaluator = 
		new MatchEvaluator(){public String eval(String in){ return matchToNarrow(in);}};
	/// <summary>
	/// 英数にマッチする正規表現
	/// </summary>
	private static final Pattern alphaDigitRegex = Pattern.compile("[ａ-ｚＡ-Ｚ０-９]+");
	
	/// <summary>
	/// 半角カタカナにマッチする正規表現
	/// </summary>
	private static final Pattern halfKanaRegex = Pattern.compile("[。-゜]+");
	
	
	
	private static String matchToWide(String m)
	{
		// 全角にする
		String wide = HankakuZenkaku.hanToZen(m);
		// 置換文字列を返す
            return wide;
	}
	
	private static String matchToNarrow(String m)
	{
		// 半角にする
		String half = HankakuZenkaku.zenToHan(m);
		// 置換文字列を返す
		return half;
	}
	
	
	/// <summary>
	/// 正規表現を使ってフィルタリングする。includeにマッチし、かつexcludeにマッチしないものを残す
	/// </summary>
	/// <param name="str">入力文字列</param>
	/// <param name="include">残す文字列にマッチする正規表現</param>
	/// <param name="exclude">除外する文字列にマッチする正規表現</param>
	/// <returns>フィルタ適用後の文字列。除外された場合はnullが返る</returns>
	public static String regexFilter(String str, Pattern include, Pattern exclude)
	{
		//includeもexcludeもnullならスルーするだけ
		if (include != null)
		{
			if (!include.matcher(str).matches())
			{
				return null;
			}
		}
		if (exclude != null)
		{
			if (!exclude.matcher(str).matches())
			{
				return null;
			}
		}
		// 合格
		return str;
	}


	/// <summary>
	/// 小文字にする
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String toLower(String str)
	{
		if (str != null)
		{
			return str.toLowerCase();
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 大文字にする
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String toUpper(String str)
	{
		if (str != null)
		{
			return str.toUpperCase();
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 各語の一文字目のみを大文字にして、残りを小文字にする。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>matches
	public static String toTitleCase(String str)
	{
		if (str != null)
		{
			//	return textInfo.toTitleCase(str);
			//  Java Library does not have correspoinding capability.
			return str;
		}
		
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 半角文字にする。
	/// カタカナや句読点も半角になるので注意。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String toNarrow(String str)
	{
		if (str != null)
		{
			return HankakuZenkaku.zenToHan(str);
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 全角文字にする。
	/// 半角英数や記号も全角になるので注意。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String toWide(String str)
	{
		if (str != null)
		{
			return HankakuZenkaku.hanToZen(str);
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 半角カタカナ・句読点のみを全角にする。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String halfKanaToWide(String str)
	{
		if (str != null)
		{
			return replace(str, halfKanaRegex, toWideMatchEvaluator);
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 英数のみを半角にする。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String alphaDigitToNarrow(String str)
	{
		if (str != null)
		{
			return replace(str, alphaDigitRegex, toNarrowMatchEvaluator);
		}
		else
		{
			return null;
		}
	}
	
	/// <summary>
	/// 半角カタカナ・句読点を全角に、英数を半角にする。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String alphaDigitToNarrowHalfKanaToWide(String str)
	{
		if (str != null)
		{
			String r = str;
			r = replace(r, alphaDigitRegex, toNarrowMatchEvaluator);
			r = replace(r, halfKanaRegex, toWideMatchEvaluator);
			return r;
		}
		else
		{
			return null;
		}
	}
	
	
	/// <summary>
	/// まっとうな数値表現にマッチする正規表現
	/// 全角でもマッチする。
	/// </summary>
	private static final Pattern numRegex = Pattern.compile("\\d+([,，]\\d+)*([\\.．]\\d+)?");
	
	/// <summary>
	/// 漢数字羅列的表現にマッチする正規表現
	/// </summary>
	private static final Pattern numUseKanjiRegex = Pattern.compile("[〇一二三四五六七八九]+([,，、][〇一二三四五六七八九]+)*");
	
	/// <summary>
	/// 漢数字表現らしきものにマッチする正規表現の一段目
	/// </summary>
	private static final Pattern numKanjiPre1Regex = Pattern.compile("[一二三四五六七八九十百千万億兆京]+");
	
	/// <summary>
	/// 漢数字表現らしきものにマッチする正規表現の二段目
	/// </summary>
	private static final String [] numKanjiPre2Str = new String[5];
	private static final Pattern [] numKanjiPre2Patterns = new Pattern[5];
	static {
		numKanjiPre2Str[4] = "(.+)京";
		numKanjiPre2Str[3] = "(.+)兆";
		numKanjiPre2Str[2] = "(.+)億";
		numKanjiPre2Str[1] = "(.+)万";
		numKanjiPre2Str[0] = "(.+)";
		for (int i = 0; i < 5; i++)
			numKanjiPre2Patterns[i] = Pattern.compile(numKanjiPre2Str[i]);
	}
	
	private static final Pattern numKanjiPre2Regex = 
		Pattern.compile(numKanjiPre2Str[4] + "?" + numKanjiPre2Str[3] + "?" + numKanjiPre2Str[2] + "?" + numKanjiPre2Str[1] + "?" + numKanjiPre2Str[0]);

	
	/// <summary>
	/// 一万未満の漢数字表現にマッチする正規表現
	/// 〇や零にはマッチしないが、とくに問題はない
	/// </summary>
	private static final String [] numKanjiStr = new String[4];
	private static final Pattern [] numKanjiPatterns = new Pattern[4];
	static {
		numKanjiStr[3] = "([一二三四五六七八九]?)千";
		numKanjiStr[2] = "([一二三四五六七八九]?)百";
		numKanjiStr[1] = "([一二三四五六七八九]?)十";
		numKanjiStr[0] = "([一二三四五六七八九]?)";
		for (int i = 0; i < 4; i++)
			numKanjiPatterns[i] = Pattern.compile(numKanjiStr[i]);
	}
	private static final Pattern numKanjiSenRegex =
		Pattern.compile(numKanjiStr[3]+ "?" + numKanjiStr[2]+ "?" +numKanjiStr[1]+ "?" + numKanjiStr[0]+ "?" );

	
	/// <summary>
	/// 漢数字1字にマッチする正規表現
	/// </summary>
	private static final Pattern numKanjiOneRegex = Pattern.compile("[〇一二三四五六七八九]");
	
	/// <summary>
	/// numKanjiOneRegexでマッチする各文字に対応するローマ数字のマップ
	/// </summary>
	/// private static final Map<String, String> mapNumKanjiToNumRoman = new HashMap<String, String>();
	/// defined at the top of the file
	
	/// <summary>
	/// 漢数字1字をローマ数字にするためのMatchEvaluator
	/// </summary>
	private static final MatchEvaluator numKanjiToNumRomanMatchEvaluator =
		new MatchEvaluator(){public String eval(String in){ return matchNumKanjiToNumRoman(in);}};

	/// <summary>
	/// numKanjiToNumRomanMatchEvaluatorのためのメソッド
	/// </summary>
	/// <param name="m"></param>
	/// <returns></returns>
	private static String matchNumKanjiToNumRoman(String m)
	{
		return mapNumKanjiToNumRoman.get(m);
	}
	
	/// <summary>
	/// 漢数字の羅列表現から数値に変換するメソッド
	/// </summary>
	/// <param name="str"></param>
	/// <returns></returns>
	private static long numKanjiToNum(String str)
	{
		return Long.parseLong(replace(str, numKanjiOneRegex, numKanjiToNumRomanMatchEvaluator));
	}
	
	/// <summary>
	/// 漢数字の一万未満の表現を数値に変換するメソッド
	/// </summary>
	/// <param name="str"></param>
	/// <returns></returns>
	private static long kanjiNumSenToNum(String str)
	{
		Matcher numKanjiSenMatch = numKanjiSenRegex.matcher(str);
		if (numKanjiSenMatch.matches())
		{
			String target = numKanjiSenMatch.group();
			long val = 0;
			for (int i = 3; i >= 0; i--){
				System.out.println("target = " + target);
				Matcher ms = numKanjiPatterns[i].matcher(target);
				if (ms.find()){
					String v = ms.group(1);
					System.out.println("- " + v);
					if (v.equals(""))
						v = "一";
					val += Math.pow(10, i) * Long.parseLong(mapNumKanjiToNumRoman.get(v));
					target = target.substring(ms.end());
				}
			}	
			return val;
		}
		
		Matcher numUseKanjiMatch = numUseKanjiRegex.matcher(str);
		if (numUseKanjiMatch.matches())
		{
			return numKanjiToNum(str);
		}
		return 0;
	}
	
	
	/// <summary>
	/// 文字列を数値として解釈し、半角数字の文字列にして返す。
	/// [TODO]テストがまだ不十分であり例外を返す場合があるかもしれない。
	/// [TODO]「万千百」といったものにToWide0を返す。
	/// [TODO]「30万」などに対応していない。
	/// </summary>
	/// <param name="str">フィルタを適用する文字列</param>
	/// <returns>フィルタ適用後の文字列</returns>
	public static String toNumerical(String str)
	{
		// まずはまっとうな数値表現かどうかを調べる。
		Matcher numMatch = numRegex.matcher(str);
		if (numMatch.matches())
		{
			return Double.parseDouble(toNarrow(numMatch.group())) + ""; // convert double to String
		}
		
		// とりあえず、数値の羅列による表現らしいかどうか調べてみる。
		Matcher numUseKanjiMatch = numUseKanjiRegex.matcher(str);
		if (numUseKanjiMatch.find())
		{
			String v = numUseKanjiMatch.group();
			if (v.length() >= 2)
			{
				// 数値の羅列による表現と判定。
				String removeComma = v.replace(",", "").replace("，", "").replace("、", "");
				return numKanjiToNum(removeComma) + ""; //convert long to String
			}
		}

		// 漢数字表現かどうか調べてみる。
		Matcher numKanjiPre1Match = numKanjiPre1Regex.matcher(str);
		if (numKanjiPre1Match.matches())
		{
			Matcher numKanjiPre2Match = numKanjiPre2Regex.matcher(numKanjiPre1Match.group());
			if (numKanjiPre2Match.matches())
			{
				String target = numKanjiPre2Match.group();
				long val = 0;
				for (int i = 4; i >= 0; i--){
					System.out.println("target = " + target);
					Matcher ms = numKanjiPre2Patterns[i].matcher(target);
					if (ms.find()){
						String v = ms.group(1);
						val += Math.pow(10, i * 4) * kanjiNumSenToNum(v);
						target = target.substring(ms.end());
					}
				}	
				return val + "";
			}
		}
		
		return "0";
	}
	
	static String replace(String target, Pattern pattern, MatchEvaluator me){
		StringBuffer sb = new StringBuffer(); 
		int index = 0;
		Matcher m = pattern.matcher(target);
		while (m.find()){
			int start = m.start();
			int end = m.end();
			String str = target.substring(start, end);
			sb.append(target.substring(index, start));
			sb.append(me.eval(str));
			index = end;
		}
		sb.append(target.substring(index));
		return sb.toString();
	}


}

interface MatchEvaluator{
	String eval(String str);
}

