﻿using System;
using System.Collections.Generic;
using System.Text;

using System.IO;
using System.Diagnostics;

namespace SlothLib.NLP
{
    /// <summary>
    /// TreeTaggerを利用するクラス
    /// </summary>
    public class TreeTagger : IMorphologicalAnalyzer
    {
        #region private フィールド

        /// <summary>
        /// tree-tagger.exe のパス
        /// </summary>
        private string treeTaggerPath;

        ///// <summary>
        ///// english.parのパス
        ///// </summary>
        //private string englishParPath;

        /// <summary>
        /// 乱数を生成する
        /// </summary>
        private static Random random = new Random();

        #endregion

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="treeTaggerPath">tree-tagger.exeのパス</param>
        public TreeTagger(string treeTaggerPath)
        {
            TreeTaggerPath=treeTaggerPath;
        }

        ///// <summary>
        ///// TreeTaggerのパスを設定する
        ///// </summary>
        ///// <param name="path">tree-tagger.exeのパス</param>
        //private void setTreeTaggerPath(string path)
        //{
        //    if (File.Exists(path))
        //    {
        //        this.treeTaggerPath = path;
        //    }
        //    else if (path.EndsWith("\\") && File.Exists(path + "tree-tagger.exe"))
        //    {
        //        this.treeTaggerPath = path + "tree-tagger.exe";
        //    }
        //    else if (File.Exists(path + "\\tree-tagger.exe"))
        //    {
        //        this.treeTaggerPath = path + "\\tree-tagger.exe";
        //    }
        //    else
        //    {
        //        throw new FileNotFoundException("TreeTagger is not found", path);
        //    }
        //}

        #endregion

        #region DoAnalyze

        /// <summary>
        /// TreeTaggerで解析を行う
        /// </summary>
        /// <param name="text">解析対象のテキスト</param>
        /// <returns>解析結果</returns>
        public TreeTaggerResult DoAnalyze(string text)
        {
            EnglishTokenizer tokenizer = new EnglishTokenizer();
            string[] words = tokenizer.DoTokenize(text);
            StringBuilder builder = new StringBuilder();
            foreach (string s in words)
            {
                builder.AppendLine(s);
            }
            text = builder.ToString();

            // ファイルパス
            string inputFilePath;
            do
            {
                inputFilePath = /*Path.GetTempPath() + "SlothLib.NLP.TreeTagger.DoAnalyze." + text.GetHashCode() + "." +*/ random.Next().ToString() + "i.tmp";
            } while (File.Exists(inputFilePath));
            string outputFilePath;
            do
            {
                outputFilePath = /*Path.GetTempPath() + "SlothLibNLPTreeTaggerDoAnalyze" + text.GetHashCode() +*/ random.Next().ToString() + "o.tmp";
            } while (File.Exists(outputFilePath));

            //Argumentのパスが長すぎるとダメらしいのでカレントディレクトリを移す
            string binDir = Directory.GetParent(this.treeTaggerPath).FullName;
            string currentDirectory = Directory.GetCurrentDirectory();
            Directory.SetCurrentDirectory(binDir);

            //ファイルに解析対象テキストを書き出し
            using (StreamWriter sw = new StreamWriter(inputFilePath, false, System.Text.Encoding.GetEncoding("shift_jis")))
            {
                sw.Write(text);
            }

            //出力結果を得る
            string parsedText = ParseFile(inputFilePath,outputFilePath);

            //tempファイルを削除
            try
            {
                File.Delete(inputFilePath);
                File.Delete(outputFilePath);
            }
            catch { }

            //カレントディレクトリを戻す
            Directory.SetCurrentDirectory(currentDirectory);
            return new TreeTaggerResult(parsedText);
        }

        /// <summary>
        /// テキストファイルを読み込んで解析する。
        /// </summary>
        /// <param name="inputFilePath">入力用tempファイルのパス</param>
        /// <param name="outputFilePath">出力用tempファイルのパス</param>
        /// <returns></returns>
        private string ParseFile(string inputFilePath, string outputFilePath)
        {
            // プロセス作成とかいろいろ
            System.Diagnostics.Process p = new System.Diagnostics.Process();
            FileInfo fileinfo = new FileInfo(this.treeTaggerPath);
            p.StartInfo.FileName = fileinfo.Name;
            //この引数（パス？）が長いと何故か上手くいかない。よって相対パスで指定
            p.StartInfo.Arguments = @" ..\lib\english.par -token -lemma -sgml -no-unknown " + inputFilePath + " " + outputFilePath;
            p.StartInfo.CreateNoWindow = true;
            p.StartInfo.UseShellExecute = false;
            p.Start();
            p.WaitForExit();
            int ec = p.ExitCode;
            p.Close();
            if (ec != 0)
            {
                File.Delete(inputFilePath);
                if (File.Exists(outputFilePath))
                {
                    File.Delete(outputFilePath);
                }
                throw new Exception("RunTime error while running TreeTagger");
            }

            //出力ファイルから読み出し
            string output = null;
            using (StreamReader sr = new StreamReader(outputFilePath, System.Text.Encoding.GetEncoding("shift_jis")))
            {
                output = sr.ReadToEnd();
            }
            return output;
        }

        #endregion

        #region プロパティ

        /// <summary>
        /// tree-tagger.exeのパス
        /// </summary>
        public string TreeTaggerPath
        {
            get { return treeTaggerPath; }
            set
            {
                string path = value;
                if (File.Exists(path))
                {
                    this.treeTaggerPath = path;
                }
                else if (path.EndsWith("\\") && File.Exists(path + "tree-tagger.exe"))
                {
                    this.treeTaggerPath = path + "tree-tagger.exe";
                }
                else if (File.Exists(path + "\\tree-tagger.exe"))
                {
                    this.treeTaggerPath = path + "\\tree-tagger.exe";
                }
                else
                {
                    throw new FileNotFoundException("TreeTagger is not found", path);
                }
            }
        }

        ///// <summary>
        ///// english.parのパス
        ///// </summary>
        //public string EnglishParPath
        //{
        //    get { return englishParPath; }
        //    set
        //    {
        //        if (!File.Exists(value))
        //        {
        //            throw new FileNotFoundException("english.par is not found", path);
        //        }
        //        englishParPath = value;
        //    }
        //}

        #endregion

        #region IMorphologicalAnalyzer メンバ

        IMorphologicalAnalyzerResult IMorphologicalAnalyzer.DoAnalyze(string text)
        {
            return this.DoAnalyze(text);
        }

        #endregion
    }
}
