package jp.sourceforge.shovel.mobilephone.impl;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import jp.sourceforge.shovel.exception.ApplicationException;
import jp.sourceforge.shovel.mobilephone.IWillcom;
import junit.framework.Assert;

import org.junit.runner.RunWith;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;
import org.seasar.framework.unit.Seasar2;
import org.seasar.framework.unit.TestContext;
import org.seasar.framework.util.ResourceUtil;

import au.com.bytecode.opencsv.CSVReader;

@RunWith(Seasar2.class)
public class WillcomImplTest {
    TestContext context_;
    Map<String, Map<String, String>> profiles_;

    public WillcomImplTest() throws IOException {
        profiles_ = new HashMap<String, Map<String, String>>();
        File file = ResourceUtil.getResourceAsFile("jp.sourceforge.shovel.mobilephone.impl.WillcomImplTest", "csv");
        CSVReader reader = new CSVReader(new FileReader(file));
        String[] keys = reader.readNext();
        String[] line;
        while ((line = reader.readNext()) != null) {
            HashMap<String, String> profile = new HashMap<String, String>();
            //0はUserAgent
            for (int i = 1; i < keys.length; i++) {
                String value = line[i];
                if (value.length() <= 0) {
                    value = null;
                }
                profile.put(keys[i], value);
            }
            profiles_.put(line[0], profile);
        }
        reader.close();
    }

    public void testShouldDetectUserAgentsAsWillcom() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Assert.assertFalse(willcom.isDoCoMo());
            Assert.assertFalse(willcom.isEZweb());
            Assert.assertFalse(willcom.isSoftBank());
            Assert.assertFalse(willcom.isEmobile());
            Assert.assertTrue(willcom.isWillcom());
            Assert.assertFalse(willcom.isNotMobile());
        }
    }
    
    public void testShouldProvideTheModelNameOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            if (willcom.getModel().indexOf(",") < 0) {
                Assert.assertEquals(profile.get("model"), willcom.getModel());
            } else {
                String[] models = willcom.getModel().split(",");
                boolean result = false;
                for (String model : models) {
                    result |= model.compareTo(profile.get("model")) == 0;
                }
                Assert.assertTrue(result);
            }
        }
    }

    public void testShouldProvideTheVendorNameOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("vendor"), willcom.getVendor());
        }
    }
    
    public void testShouldProvideTheModelVersionOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("modelVersion"), willcom.getModelVersion());
        }
    }
    
    public void testShouldProvideTheBrowserVersionOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("browserVersion"), willcom.getBrowserVersion());
        }
    }
    
    public void testShouldProvideTheCacheSizeOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IWillcom willcom = (IWillcom)container.getComponent(IWillcom.class);
            willcom.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(Integer.parseInt(profile.get("cacheSize")), willcom.getCacheSize());
        }
    }
}
