package jp.sourceforge.shovel.mobilephone.impl;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import jp.sourceforge.shovel.exception.ApplicationException;
import jp.sourceforge.shovel.mobilephone.IEZweb;
import junit.framework.Assert;

import org.junit.runner.RunWith;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;
import org.seasar.framework.mock.servlet.MockHttpServletRequest;
import org.seasar.framework.unit.Seasar2;
import org.seasar.framework.unit.TestContext;
import org.seasar.framework.util.ResourceUtil;

import au.com.bytecode.opencsv.CSVReader;

@RunWith(Seasar2.class)
public class EZwebImplTest {
    TestContext context_;
    Map<String, Map<String, String>> profiles_;
    HttpServletRequest request_;
    
    public EZwebImplTest() throws IOException {
        profiles_ = new HashMap<String, Map<String, String>>();
        File file = ResourceUtil.getResourceAsFile("jp.sourceforge.shovel.mobilephone.impl.EZwebImplTest", "csv");
        CSVReader reader = new CSVReader(new FileReader(file));
        String[] keys = reader.readNext();
        String[] line;
        while ((line = reader.readNext()) != null) {
            HashMap<String, String> profile = new HashMap<String, String>();
            //0はUserAgent
            for (int i = 1; i < keys.length; i++) {
                profile.put(keys[i], line[i]);
            }
            profiles_.put(line[0], profile);
        }
        reader.close();
    }
    
    public void testShouldDetectUserAgentsAsEzweb() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Assert.assertFalse(ezweb.isDoCoMo());
            Assert.assertTrue(ezweb.isEZweb());
            Assert.assertFalse(ezweb.isSoftBank());
            Assert.assertFalse(ezweb.isEmobile());
            Assert.assertFalse(ezweb.isWillcom());
            Assert.assertFalse(ezweb.isNotMobile());
        }
    }
    
    public void testShouldProvideTheModelNameOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("model"), ezweb.getModel());
        }
    }
    
    public void testShouldProvideTheUidOfASubscriber() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
        String userAgent = "KDDI-TS3C UP.Browser/6.2.0.12.1.3 (GUI) MMP/2.0')";
        ezweb.parse(userAgent);
        Assert.assertNull(ezweb.getUID());

        String uid = "12345678901234_56.ezweb.ne.jp";
        MockHttpServletRequest request = (MockHttpServletRequest)request_;
        request.addHeader("X-UP-SUBNO", uid);
        ezweb.parse(userAgent);
        Assert.assertEquals(uid, ezweb.getUID());
    }
    
    public void testShouldProvideTheVersionOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("version"), ezweb.getVersion());
        }
    }

    public void testShouldProvideTheDeviceIdOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("model"), ezweb.getDeviceID());
        }
    }

    public void testShouldProvideTheServerOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(profile.get("server"), ezweb.getServer());
        }
    }

    public void testShouldTellWhetherAUserAgentIsXhtmlCompliantModelOrNot() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(Boolean.parseBoolean(profile.get("xhtmlCompliant")), ezweb.isXHTMLCompliant());
        }
    }
    public void testShouldProvideTheCommentOfAUserAgent() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
        ezweb.parse("UP.Browser/3.04-TS14 UP.Link/3.4.4 (Google WAP Proxy/1.0)");
        Assert.assertEquals("Google WAP Proxy/1.0", ezweb.getComment());

        ezweb = (IEZweb)container.getComponent(IEZweb.class);
        ezweb.parse("KDDI-TS3C UP.Browser/6.2.0.12.1.3 (GUI) MMP/2.0");
        Assert.assertNull(ezweb.getComment());
    }
    
    public void testShouldTellWhetherAUserAgentIsWap2ModelOrNot() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(Boolean.parseBoolean(profile.get("wap2")), ezweb.isWAP2());
        }
    }

    public void testShouldTellWhetherAUserAgentIsWap1ModelOrNot() throws ApplicationException {
        S2Container container = SingletonS2ContainerFactory.getContainer();
        for (String userAgent : profiles_.keySet()) {
            IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
            ezweb.parse(userAgent);
            
            Map<String, String> profile = profiles_.get(userAgent);
            Assert.assertEquals(Boolean.parseBoolean(profile.get("wap1")), ezweb.isWAP1());
        }
    }
    
    public void testShouldProvideTheScreenInformationOfAUserAgent() throws ApplicationException {
        MockHttpServletRequest request = (MockHttpServletRequest)request_;
        request.addHeader("X-UP-DEVCAP-SCREENPIXELS", "90,70");
        request.addHeader("X-UP-DEVCAP-SCREENDEPTH", "16,RGB565");
        request.addHeader("X-UP-DEVCAP-ISCOLOR", "1");
        
        S2Container container = SingletonS2ContainerFactory.getContainer();
        IEZweb ezweb = (IEZweb)container.getComponent(IEZweb.class);
        ezweb.parse("KDDI-TS21 UP.Browser/6.0.2.276 (GUI) MMP/1.1");
        
        Assert.assertEquals(90, ezweb.getWidth());
        Assert.assertEquals(70, ezweb.getHeight());
        Assert.assertEquals(65536, ezweb.getDepth());
        Assert.assertTrue(ezweb.isColor());
    }
}
