package jp.sourceforge.shovel.service.impl;

import static jp.sourceforge.shovel.AvailabilityType.*;
import static jp.sourceforge.shovel.DeviceType.*;
import static org.seasar.framework.container.ContainerConstants.*;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.ArrayUtils;
import org.apache.struts.Globals;
import org.apache.struts.util.MessageResources;
import org.seasar.framework.container.S2Container;

import com.sun.syndication.feed.synd.SyndContent;
import com.sun.syndication.feed.synd.SyndContentImpl;
import com.sun.syndication.feed.synd.SyndEntry;
import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.feed.synd.SyndFeedImpl;
import com.sun.syndication.io.FeedException;
import com.sun.syndication.io.SyndFeedOutput;

import jp.sourceforge.shovel.AvailabilityType;
import jp.sourceforge.shovel.DeviceType;
import jp.sourceforge.shovel.FormatType;
import jp.sourceforge.shovel.SortOrderType;
import jp.sourceforge.shovel.SortType;
import jp.sourceforge.shovel.ViewType;
import jp.sourceforge.shovel.entity.IConnectionWrapper;
import jp.sourceforge.shovel.entity.IDedicatedClient;
import jp.sourceforge.shovel.entity.IDevice;
import jp.sourceforge.shovel.entity.IDirectMessage;
import jp.sourceforge.shovel.entity.IFavorite;
import jp.sourceforge.shovel.entity.IFriendship;
import jp.sourceforge.shovel.entity.IServerFile;
import jp.sourceforge.shovel.entity.IStatus;
import jp.sourceforge.shovel.entity.IStatusToken;
import jp.sourceforge.shovel.entity.IUser;
import jp.sourceforge.shovel.exception.ApplicationException;
import jp.sourceforge.shovel.logic.IShovelLogic;
import jp.sourceforge.shovel.service.IDirectoryService;
import jp.sourceforge.shovel.service.IServerFileService;
import jp.sourceforge.shovel.service.IShovelService;

public class ShovelServiceImpl implements IShovelService {
    Map<Long, IStatus> statusMap_;

    public ShovelServiceImpl() {
        statusMap_ = new HashMap<Long, IStatus>();
    }
    void cacheStatus(IStatus status) {
        if (status != null) {
            statusMap_.put(status.getStatusId(), status);
        }
    }
    void cacheStatuses(IStatus[] statuses) {
        for (IStatus status : statuses) {
            cacheStatus(status);
        }
    }
    void purgeStatuses() {
        statusMap_.clear();
    }
    IStatus fetchStatus(long statusId) {
        return statusMap_.get(statusId);
    }
    
    int correctLimit(int limit) {
        IUser user = getDirectoryService().getLoginUser();
        if (user != null && limit <= 0) {
            limit = user.getViewLines();
        }
        //TODO
        if (limit <= 0 || limit > 20) {
            limit = 20;
        }
        return limit;
    }
    
    /////
    public IStatus[] getFavorites(String foreignKey, int page, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getUser(foreignKey);
        if (user == null) {
            throw new ApplicationException("");
        }
        limit = correctLimit(limit);
        IStatus[] statuses = getShovelLogic().getFavorites(user.getUserId(), page * limit, limit + 1);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getReplies(String foreignKey, int page, int limit) throws ApplicationException {
        if (foreignKey == null) {
            //TODO
            throw new ApplicationException("");
        }
        IUser user = getDirectoryService().getLoginUser();
        //1つ多く検索して次ページの有無の判断に
        if (user == null || foreignKey.compareTo(user.getForeignKey()) != 0) {
            user = getDirectoryService().getUser(foreignKey);
        }
        limit = correctLimit(limit);
        IStatus[] statuses = getShovelLogic().getReplies(
                user.getUserId(), page * limit, limit + 1);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus getRecent(String foreignKey) {
        IUser user = getDirectoryService().getUser(foreignKey);
        IStatus status = getShovelLogic().getRecent(user.getUserId());
        cacheStatus(status);
        return status;
    }
    public IStatus getRecent(long userId) {
        IStatus status = getShovelLogic().getRecent(userId);
        cacheStatus(status);
        return status;
    }
    public IStatus getStatus(long statusId) {
        IStatus status = fetchStatus(statusId);
        if (status == null) {
            status = getShovelLogic().getStatus(statusId);
            cacheStatus(status);
        }
        return status;
    }
    public IStatus[] getRecents(long[] userIds) {
        IStatus[] statuses = getShovelLogic().getRecents(userIds);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getStatuses(int page, int limit) {
        limit = correctLimit(limit);
        //1つ多く検索して次ページの有無の判断に
        IStatus[] statuses = getShovelLogic().getStatuses(page * limit, limit + 1);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getStatuses(long sinceId, int limit) {
        limit = correctLimit(limit);
        IStatus[] statuses = getShovelLogic().getStatuses(sinceId, limit);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getStatuses(SortType sortType, SortOrderType sortOrderType, int limit) {
        limit = correctLimit(limit);
        IStatus[] statuses = getShovelLogic().getStatuses(sortType, sortOrderType, limit);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getStatuses(String foreignKey, boolean withFriends, String since, int limit) throws ApplicationException {
        if (foreignKey == null) {
            //TODO
            throw new ApplicationException("");
        }
        limit = correctLimit(limit);
        
        IUser user = getDirectoryService().getLoginUser();
        if (user == null || foreignKey.compareTo(user.getForeignKey()) != 0) {
            user = getDirectoryService().getUser(foreignKey);
        }
        IStatus[] statuses = getShovelLogic().getStatuses(user.getUserId(),
                withFriends, user.getRepliesType(), since, limit);
        cacheStatuses(statuses);
        return statuses;
    }
    public IStatus[] getStatuses(String foreignKey, boolean withFriends, int page, int limit, int correctedOffset, int correctedLimit) throws ApplicationException {
        if (foreignKey == null) {
            //TODO
            throw new ApplicationException("");
        }
        limit = correctLimit(limit);
        int offset = page * limit + correctedOffset;
        limit += correctedLimit;
        
        //1つ多く検索して次ページの有無の判断に
        IUser user = getDirectoryService().getLoginUser();
        if (user == null || foreignKey.compareTo(user.getForeignKey()) != 0) {
            user = getDirectoryService().getUser(foreignKey);
        }
        IStatus[] statuses = getShovelLogic().getStatuses(user.getUserId(),
                withFriends, user.getRepliesType(), offset, limit);
        cacheStatuses(statuses);
        return statuses;
    }
    public int removeStatus(long statusId) throws ApplicationException {
        IUser user = getDirectoryService().getLoginUser();
        IStatus status = getStatus(statusId);
        if (status == null) {
            //TODO
            throw new ApplicationException("");
        }
        if (user.getUserId() != status.getSenderId()) {
            //TODO
            throw new ApplicationException("");
        }
        getDirectoryService().decrementFavoritesByStatus(statusId);
        getShovelLogic().clearStatusReference(statusId);
        return getShovelLogic().updateStatusRemove(statusId);
    }
    public IStatus updateStatus(String status, String source) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IStatusToken[] tokens = getShovelLogic().parseStatus(status);
        boolean open = true;
        long referenceSenderId = 0;
        if (tokens != null && tokens.length > 0) {
            for (IStatusToken token : tokens) {
                switch (token.getType()) {
                case 1:
                    open = false;
                case 2:
                    String foreignKey = tokens[0].getRegs()[1];
                    IUser user = directoryService.getUser(foreignKey);
                    referenceSenderId = user.getUserId();
                default:
                    break;
                }
            }
        }
        IUser user = directoryService.getLoginUser();
        directoryService.incrementStatuses(user.getForeignKey());
        
        return getShovelLogic().updateStatus(status, source, referenceSenderId, open, user.getUserId());
    }
    public IStatus[] truncate(IStatus[] statuses, int limit) {
        limit = correctLimit(limit);
        if (statuses.length > limit) {
            statuses = (IStatus[])ArrayUtils.subarray(statuses, 0, limit);
        }
        return statuses;
    }
    public IUser[] truncate(IUser[] users, int limit) {
        limit = correctLimit(limit);
        if (users.length > limit) {
            users = (IUser[])ArrayUtils.subarray(users, 0, limit);
        }
        return users;
    }
    public IFriendship[] truncate(IFriendship[] friends, int limit) {
        limit = correctLimit(limit);
        if (friends.length > limit) {
            friends = (IFriendship[])ArrayUtils.subarray(friends, 0, limit);
        }
        return friends;
    }
    public IStatus[] prepareForView(IStatus[] statuses, int limit, boolean truncate) {
        if (truncate) {
            statuses = truncate(statuses, limit);
        }
        if (statuses == null || statuses.length <= 0) {
            return new IStatus[0];
        }
        
        Set<Long> userIdSet = new HashSet<Long>();
        Set<String> sourceSet = new HashSet<String>();
        long[] statusIds = new long[statuses.length];
        int i = 0;
        for (IStatus status : statuses) {
            userIdSet.add(status.getReferenceSenderId());
            userIdSet.add(status.getSenderId());
            sourceSet.add(status.getSource());
            statusIds[i++] = status.getStatusId();
        }
        Long[] userIds = userIdSet.toArray(new Long[userIdSet.size()]);
        IUser[] users = getDirectoryService().getUsers(ArrayUtils.toPrimitive(userIds));
        Map<Long, IUser> userMap = new HashMap<Long, IUser>();
        for (IUser user : users) {
            //サムネイル作るので廃止
            IServerFile serverFile = user.getServerFile();
            if (serverFile != null) {
                serverFile.prepareForView();
            }
            userMap.put(user.getUserId(), user);
        }
        
        Map<Long, IFavorite> favoriteMap = new HashMap<Long, IFavorite>();
        IUser user = getDirectoryService().getLoginUser();
        if (user != null) {
            IFavorite[] favorites = getFavorites(statusIds);
            for (IFavorite favorite : favorites) {
                favoriteMap.put(favorite.getStatusId(), favorite);
            }
        }
        
        String[] keys = sourceSet.toArray(new String[sourceSet.size()]);
        IDedicatedClient[] clients = getClients(keys);
        Map<String, IDedicatedClient> clientMap = new HashMap<String, IDedicatedClient>();
        for (IDedicatedClient client : clients) {
            clientMap.put(client.getText(), client);
        }
        
        for (IStatus status : statuses) {
            String source = status.getSource();
            IDedicatedClient client = clientMap.get(source);
            if (client != null) {
                status.setDedicatedClient(client);
            }
            
            long userId = status.getReferenceSenderId();
            status.setReferenceSender(userMap.get(userId));
            userId = status.getSenderId();
            status.setSender(userMap.get(userId));
            long statusId = status.getStatusId();
            status.setFavorite(favoriteMap.get(statusId));
        }
        return statuses;
    }
    public void outputFeed(FormatType formatType, ViewType viewType, IUser user, IStatus[] statuses) throws ApplicationException {
        HttpServletRequest request = (HttpServletRequest)getContainer().getComponent(REQUEST_NAME);
        HttpServletResponse response = (HttpServletResponse)getContainer().getComponent(RESPONSE_NAME);
        MessageResources resources = (MessageResources)request.getAttribute(Globals.MESSAGES_KEY);
        Object[] args = {user.getDisplayName(), formatType.getId()};
        String title = resources.getMessage(viewType.getId() + ".feed.title", args);
        String server = request.getServerName();
        if (request.getServerPort() != 80) {
            server += ":" + String.valueOf(request.getServerPort());
        }
        //args = new Object[] {server, formatType.getId(), user.getUserId()};
        //String link = resources.getMessage(viewType.getId() + ".feed.url", args);
        String link = (String)request.getAttribute("request-url");
        args = new Object[] {user.getDisplayName(), user.getForeignKey()};
        String description = resources.getMessage(viewType.getId() + ".feed.description", args);
        
        SyndFeed feed = new SyndFeedImpl();
        feed.setTitle(title);
        feed.setLink(link);
        feed.setDescription(description);
        
        List<SyndEntry> entries = new ArrayList<SyndEntry>();
        SyndEntry entry;
        SyndContent html;
        for (IStatus s : statuses) {
            IUser sender = s.getSender();
            args = new Object[] {user.getForeignKey(), s.getBody()};
            title = resources.getMessage("statuses.feed.entry.description", args);
            args = new Object[] {server, sender.getForeignKey(), s.getStatusId()};
            link = resources.getMessage("statuses.feed.entry.link", args);
            Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis(s.getSentTime());
            Date publishedDate = calendar.getTime();
            
            entry = new SyndEntryImpl();
            entry.setTitle(title);
            entry.setLink(link);
            entry.setPublishedDate(publishedDate);
            entry.setUpdatedDate(publishedDate);
            html = new SyndContentImpl();
            html.setType("html");
            html.setValue(title);
            entry.setDescription(html);
            entries.add(entry);
        }

        feed.setEntries(entries);

        try {
            feed.setFeedType(formatType.getFeedType());
            String mimeType = formatType.getMimeType();
            response.setContentType(mimeType + "; charset=UTF-8");
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, response.getWriter());
        } catch (FeedException e) {
            //TODO
        } catch (IOException e) {
            //TODO
        }
    }
    
    /////
    
    public IDirectMessage sendDirectMessage(String foreignKey, String text,
            String source) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IUser recipient = directoryService.getUser(foreignKey);
        if (recipient == null) {
            //TODO
            throw new ApplicationException("");
        }
        IUser sender = directoryService.getLoginUser();
        
        IDirectMessage directMessage = getShovelLogic().sendDirectMessage(
                sender.getUserId(), recipient.getUserId(), text, source);
        if (directMessage != null) {
            directoryService.incrementDirectMessages(foreignKey);
        }
        return directMessage;
    }
    public IDirectMessage sendDirectMessage(long userId, String text,
            String source) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IUser recipient = directoryService.getUser(userId);
        if (recipient == null) {
            //TODO
            throw new ApplicationException("");
        }
        return sendDirectMessage(recipient.getForeignKey(), text, source);
    }
    public int removeDirectMessage(long directMessageId) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IUser user = directoryService.getLoginUser();
        IDirectMessage directMessage = getShovelLogic().getDirectMessage(directMessageId);
        
        boolean sent = directMessage.getSenderId() == user.getUserId();
        boolean inbox = directMessage.getRecipientId() == user.getUserId();
        if (sent && !directMessage.isSent()) {
            throw new ApplicationException("");
        } else if (inbox && !directMessage.isInbox()) {
            throw new ApplicationException("");
        }
        if (!sent && !inbox) {
            throw new ApplicationException("");
        }
        int result = getShovelLogic().updateRemoveDirectMessage(directMessageId, sent);
        if (result <= 0) {
            throw new ApplicationException("");
        }
        if (sent) {
            directoryService.decrementDirectMessages(user.getForeignKey());
        }
        return result;
    }
    public IDirectMessage getDirectMessage(long directMessageId) {
        return getShovelLogic().getDirectMessage(directMessageId);
    }
    public IDirectMessage[] getDirectMessages(boolean sent, int page, int limit) {
        IUser user = getDirectoryService().getLoginUser();
        limit = correctLimit(limit);
        
        //1つ多く検索して次ページの有無の判断に
        return getShovelLogic().getDirectMessages(sent, user.getUserId(),
                page * limit, limit + 1);
    }
    public IDirectMessage[] getDirectMessages(boolean sent, long sinceId, int limit) {
        IUser user = getDirectoryService().getLoginUser();
        limit = correctLimit(limit);
        
        //1つ多く検索して次のページの有無の判断に
        return getShovelLogic().getDirectMessages(sent, user.getUserId(), sinceId, limit);
    }
    public IDirectMessage[] getDirectMessages(boolean sent, String since, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getLoginUser();
        limit = correctLimit(limit);
        
        //1つ多く検索して次のページの有無の判断に
        return getShovelLogic().getDirectMessages(sent, user.getUserId(), since, limit);
    }
    public IDirectMessage[] truncate(IDirectMessage[] directMessages) {
        IUser login = getDirectoryService().getLoginUser();
        //TODO
        int limit = 10;
        if (login != null) {
            limit = login.getViewLines();
        }
        if (directMessages.length > limit) {
            directMessages = (IDirectMessage[])ArrayUtils.subarray(directMessages, 0, limit);
        }
        return directMessages;
    }
    public IDirectMessage[] prepareForView(IDirectMessage[] directMessages, boolean truncate) {
        if (truncate) {
            directMessages = truncate(directMessages);
        }
        if (directMessages == null || directMessages.length <= 0) {
            return new IDirectMessage[0];
        }
        
        Set<Long> userIdSet = new HashSet<Long>();
        for (IDirectMessage directMessage : directMessages) {
            userIdSet.add(directMessage.getSenderId());
            userIdSet.add(directMessage.getRecipientId());
        }
        Long[] userIds = userIdSet.toArray(new Long[userIdSet.size()]);
        IUser[] users = getDirectoryService().getUsers(ArrayUtils.toPrimitive(userIds));
        Map<Long, IUser> userMap = new HashMap<Long, IUser>();
        for (IUser user : users) {
            //サムネイル作るので廃止
            IServerFile serverFile = user.getServerFile();
            if (serverFile != null) {
                serverFile.prepareForView();
            }
            userMap.put(user.getUserId(), user);
        }
        
        for (IDirectMessage directMessage : directMessages) {
            long userId = directMessage.getSenderId();
            directMessage.setSender(userMap.get(userId));
            userId = directMessage.getRecipientId();
            directMessage.setRecipient(userMap.get(userId));
        }
        return directMessages;
    }
    public void outputFeed(FormatType formatType, ViewType viewType, IUser user, IDirectMessage[] directMessages) throws ApplicationException {
        HttpServletRequest request = (HttpServletRequest)getContainer().getComponent(REQUEST_NAME);
        HttpServletResponse response = (HttpServletResponse)getContainer().getComponent(RESPONSE_NAME);
        MessageResources resources = (MessageResources)request.getAttribute(Globals.MESSAGES_KEY);
        Object[] args = {user.getForeignKey()};
        String title = resources.getMessage(viewType.getId() + ".feed.title", args);
        String server = request.getServerName();
        if (request.getServerPort() != 80) {
            server += ":" + String.valueOf(request.getServerPort());
        }
        //args = new Object[] {server, formatType.getId(), user.getUserId()};
        //String link = resources.getMessage(viewType.getId() + ".feed.url", args);
        String link = (String)request.getAttribute("request-url");
        String description = resources.getMessage(viewType.getId() + ".feed.description", args);
        
        SyndFeed feed = new SyndFeedImpl();
        feed.setTitle(title);
        feed.setLink(link);
        feed.setDescription(description);
        
        List<SyndEntry> entries = new ArrayList<SyndEntry>();
        SyndEntry entry;
        for (IDirectMessage dm : directMessages) {
            IUser sender = dm.getSender();
            IUser recipient = dm.getRecipient();
            args = new Object[] {sender.getForeignKey(), recipient.getForeignKey()};
            title = resources.getMessage(viewType.getId() + ".feed.entry.title", args);
            args = new Object[] {dm.getText()};
            description = resources.getMessage(viewType.getId() + ".feed.entry.description", args);
            args = new Object[] {server, dm.getDirectMessageId()};
            link = resources.getMessage(viewType.getId() + ".feed.entry.link", args);
            SyndContent content = new SyndContentImpl();
            content.setValue(description);
            Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis(dm.getSentTime());
            Date publishedDate = calendar.getTime();
            
            entry = new SyndEntryImpl();
            entry.setTitle(title);
            if (formatType.isRss()) {
                entry.setDescription(content);
            } else {
                List<SyndContent> contents = new ArrayList<SyndContent>();
                contents.add(content);
                entry.setContents(contents);
            }
            entry.setPublishedDate(publishedDate);
            entry.setLink(link);
            entries.add(entry);
        }
        feed.setEntries(entries);
        try {
            feed.setFeedType(formatType.getFeedType());
            String mimeType = formatType.getMimeType();
            response.setContentType(mimeType + "; charset=UTF-8");
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, response.getWriter());
        } catch (FeedException e) {
            //TODO
        } catch (IOException e) {
            //TODO
        }
    }

    /////
    
    public boolean isFriend(String foreignKey) throws ApplicationException {
        IUser active = getDirectoryService().getLoginUser();
        if (active == null) {
            return false;
        }
        IUser passive = active;
        if (foreignKey != null && foreignKey.length() > 0) {
            passive = getDirectoryService().getUser(foreignKey);
            if (passive == null) {
                //TODO
                throw new ApplicationException("");
            }
        }
        IFriendship friendship = getShovelLogic().getFriendship(active.getUserId(),
                                                                 passive.getUserId());
        return friendship != null;
    }
    public boolean isFriend(long userId) throws ApplicationException {
        IUser user = getDirectoryService().getUser(userId);
        return isFriend(user.getForeignKey());
    }
    public IFriendship[] getFriends(String foreignKey, int offset, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getUser(foreignKey);
        if (user == null) {
            //TODO
            throw new ApplicationException("");
        }
        limit = correctLimit(limit);
        return getShovelLogic().getFriends(user.getUserId(), offset, limit);
    }
    public IFriendship[] getFriends(String foreignKey, boolean withFollower, int page, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getUser(foreignKey);
        limit = correctLimit(limit);
        IFriendship[] friends = getShovelLogic().getFriends(user.getUserId(), page * limit, limit + 1);
        if (friends != null && friends.length > 0) {
            Map<Long, IFriendship> friendMap = new HashMap<Long, IFriendship>();
            long[] passiveIds = new long[friends.length];
            int i = 0;
            for (IFriendship friendship : friends) {
                long passiveId = friendship.getPassiveId();
                passiveIds[i++] = passiveId;
                friendMap.put(passiveId, friendship);
            }
            IFriendship[] mines = getShovelLogic().getFriends(user.getUserId(), passiveIds);
            for (IFriendship mine : mines) {
                IFriendship peoples = friendMap.get(mine.getPassiveId());
                peoples.setFriend(mine);
            }
            if (withFollower) {
                IFriendship[] others = getShovelLogic().getFollowers(user.getUserId(), passiveIds);
                for (IFriendship other : others) {
                    IFriendship peoples = friendMap.get(other.getActiveId());
                    peoples.setFollower(other);
                }
            }
        }
        return friends;
    }
    public IFriendship[] getFollowers(String foreignKey, int offset, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getLoginUser();
        if (user == null) {
            //TODO
            throw new ApplicationException("");
        }
        limit = correctLimit(limit);
        user = getDirectoryService().getUser(foreignKey);
        return getShovelLogic().getFollowers(user.getUserId(), offset, limit);
    }
    public IFriendship[] getFollowers(String foreignKey, boolean withFriends, int page, int limit) throws ApplicationException {
        IUser user = getDirectoryService().getUser(foreignKey);
        long passiveId = user.getUserId();
        user = getDirectoryService().getLoginUser();
        limit = correctLimit(limit);
        IFriendship[] followers = getShovelLogic().getFollowers(passiveId, page * limit, limit);
        if (followers != null && followers.length > 0) {
            Map<Long, IFriendship> followerMap = new HashMap<Long, IFriendship>();
            long[] passiveIds = new long[followers.length];
            int i = 0;
            for (IFriendship friendship : followers) {
                passiveId = friendship.getActiveId();
                passiveIds[i++] = passiveId;
                followerMap.put(passiveId, friendship);
            }
            if (withFriends) {
                IFriendship[] mines = getShovelLogic().getFriends(user.getUserId(), passiveIds);
                for (IFriendship mine : mines) {
                    IFriendship peoples = followerMap.get(mine.getPassiveId());
                    peoples.setFriend(mine);
                }
            }
        }
        return followers;
    }
    public IFriendship createFriendship(String foreignKey) throws ApplicationException {
        if (isFriend(foreignKey)) {
            //TODO
            throw new ApplicationException("");
        }
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(foreignKey);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        directoryService.incrementFriends(active.getForeignKey());
        directoryService.incrementFollowers(foreignKey);
        return getShovelLogic().createFriendship(active.getUserId(), passive.getUserId());
    }
    public IFriendship createFriendship(long userId) throws ApplicationException {
        if (isFriend(userId)) {
            //TODO
            throw new ApplicationException("");
        }
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(userId);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        directoryService.incrementFriends(active.getForeignKey());
        directoryService.incrementFollowers(passive.getForeignKey());
        return getShovelLogic().createFriendship(active.getUserId(), passive.getUserId());
    }
    public int updateFriendshipNotify(long userId, boolean notify) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(userId);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        return getShovelLogic().updateFriendshipNotify(active.getUserId(), userId, notify);
    }
    public int updateFriendshipNotify(String foreignKey, boolean notify) throws ApplicationException {
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(foreignKey);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        return getShovelLogic().updateFriendshipNotify(active.getUserId(), passive.getUserId(), notify);
    }
    public int removeFriendship(String foreignKey) throws ApplicationException {
        if (!isFriend(foreignKey)) {
            throw new ApplicationException("");
        }
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(foreignKey);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        directoryService.decrementFriends(active.getForeignKey());
        directoryService.decrementFollowers(foreignKey);
        int result = getShovelLogic().removeFriendship(active.getUserId(), passive.getUserId());
        if (result != 1) {
            throw new ApplicationException("");
        }
        return result;
    }
    public int removeFriendship(long userId) throws ApplicationException {
        if (!isFriend(userId)) {
            throw new ApplicationException("");
        }
        IDirectoryService directoryService = getDirectoryService();
        IUser active = directoryService.getLoginUser();
        IUser passive = directoryService.getUser(userId);
        if (passive == null) {
            //TODO
            throw new ApplicationException("");
        }
        directoryService.decrementFriends(active.getForeignKey());
        directoryService.decrementFollowers(passive.getForeignKey());
        int result = getShovelLogic().removeFriendship(active.getUserId(), passive.getUserId());
        if (result != 1) {
            throw new ApplicationException("");
        }
        return result;
    }
    public IFriendship[] prepareForView(IFriendship[] friendships, int limit, boolean truncate) {
        if (truncate) {
            friendships = truncate(friendships, limit);
        }
        if (friendships == null || friendships.length <= 0) {
            return null;
        }
        
        Set<Long> userIdSet = new HashSet<Long>();
        for (IFriendship friendship : friendships) {
            userIdSet.add(friendship.getActiveId());
            userIdSet.add(friendship.getPassiveId());
        }
        
        long[] userIds = ArrayUtils.toPrimitive(userIdSet.toArray(new Long[userIdSet.size()]));
        IUser[] users = getDirectoryService().getUsers(userIds);
        Map<Long, IUser> userMap = new HashMap<Long, IUser>();
        for (IUser user : users) {
            //サムネイル作るので廃止
            IServerFile serverFile = user.getServerFile();
            if (serverFile != null) {
                serverFile.prepareForView();
            }
            userMap.put(user.getUserId(), user);
        }
        
        for (IFriendship friendship : friendships) {
            long userId = friendship.getActiveId();
            friendship.setActive(userMap.get(userId));
            userId = friendship.getPassiveId();
            friendship.setPassive(userMap.get(userId));
        }
        
        return friendships;
    }
    
    /////
    
    public IFavorite getFavorite(long statusId) {
        IDirectoryService directoryService = getDirectoryService();
        IUser user = directoryService.getLoginUser();
        return getShovelLogic().getFavorite(statusId, user.getUserId());
    }
    public IFavorite[] getFavorites(long[] statusIds) {
        IDirectoryService directoryService = getDirectoryService();
        IUser user = directoryService.getLoginUser();
        return getShovelLogic().getFavorites(statusIds, user.getUserId());
    }
    public IFavorite createFavorite(long statusId) {
        IDirectoryService directoryService = getDirectoryService();
        IUser user = directoryService.getLoginUser();
        if (getFavorite(statusId) == null) {
            IStatus status = getStatus(statusId);
            if (!status.isRemove()) {
                directoryService.incrementFavorites(user.getForeignKey());
                if (status.getSenderId() != user.getUserId()) {
                    directoryService.incrementGivenFavorites(status.getSenderId());
                }
                getShovelLogic().incrementGivenFavorites(statusId);
                return getShovelLogic().createFavorite(statusId, user.getUserId());
            }
        }
        return null;
    }
    public int removeFavorite(long statusId) {
        IDirectoryService directoryService = getDirectoryService();
        IUser user = directoryService.getLoginUser();
        if (getFavorite(statusId) != null) {
            IStatus status = getStatus(statusId);
            if (!status.isRemove()) {
                directoryService.decrementFavorites(user.getForeignKey());
                if (status.getSenderId() != user.getUserId()) {
                    directoryService.decrementGivenFavorites(status.getSenderId());
                }
                getShovelLogic().decrementGivenFavorites(statusId);
                return getShovelLogic().removeFavorite(statusId, user.getUserId());
            }
        }
        return 0;
    }
    
    /////
    
    public IDevice createDevice(DeviceType deviceType, String address) {
        IUser user = getDirectoryService().getLoginUser();
        IDevice device = user.getDevice();
        device.setAddress(address);
        device.setAvailabilityType(INACTIVATION);
        device.setDeviceType(deviceType);
        device.setUserId(user.getUserId());
        if (device.getDeviceId() > 0) {
            getShovelLogic().updateDevice(device);
        } else {
            getShovelLogic().createDevice(device);
        }
        return device;
    }
    public int activateDevice() {
        IUser user = getDirectoryService().getLoginUser();
        IDevice device = user.getDevice();
        device.setAvailabilityType(ON);
        return getShovelLogic().updateDevice(device);
    }
    public int updateDevice(AvailabilityType availabilityType) {
        IUser user = getDirectoryService().getLoginUser();
        IDevice device = user.getDevice();
        device.setAvailabilityType(availabilityType);
        return getShovelLogic().updateDevice(device);
    }
    public int resetDevice() {
        IUser user = getDirectoryService().getLoginUser();
        IDevice device = user.getDevice();
        device.setAddress("");
        device.setAvailabilityType(INACTIVATION);
        device.setDeviceType(UNKNOWN);
        return getShovelLogic().updateDevice(device);
    }
    
    /////
    
    public IDedicatedClient createClient(String key, String url, String version) {
        return getShovelLogic().createClient(key, url, version);
    }
    public int updateClient(IDedicatedClient client) {
        return getShovelLogic().updateClient(client);
    }
    public IDedicatedClient getClient(String key) {
        return getShovelLogic().getClient(key);
    }
    public IDedicatedClient[] getClients(String[] keys) {
        return getShovelLogic().getClients(keys);
    }
    public int removeClient(String key) {
        return getShovelLogic().removeClient(key);
    }
    
    /////
    
    public IConnectionWrapper[] getConnectionWrappers() {
        return getShovelLogic().getConnectionWrappers();
    }
    public IConnectionWrapper getConnectionWrapper(DeviceType deviceType) {
        return getShovelLogic().getConnectionWrapper(deviceType);
    }
    
    /////
    
    public IUser[] prepareForView(IUser[] users, int limit) {
        users = truncate(users, limit);
        if (users == null || users.length <= 0) {
            return null;
        }
        //サムネイル作るので廃止
        for (IUser user : users) {
            IServerFile serverFile = user.getServerFile();
            if (serverFile != null) {
                serverFile.prepareForView();
            }
        }
        return users;
    }
    
    /////
    
    IDirectoryService directoryService_;
    IServerFileService serverFileService_;
    
    public void setDirectoryService(IDirectoryService directoryService) {
        directoryService_ = directoryService;
    }
    public IDirectoryService getDirectoryService() {
        return directoryService_;
    }
    public void setServerFileService(IServerFileService serverFileService) {
        serverFileService_ = serverFileService;
    }
    public IServerFileService getServerFileService() {
        return serverFileService_;
    }
    
    S2Container container_;
    IShovelLogic shovelLogic_;
    
    public void setContainer(S2Container container) {
        container_ = container;
    }
    S2Container getContainer() {
        return container_;
    }
    public void setShovelLogic(IShovelLogic shovelLogic) {
        shovelLogic_ = shovelLogic;
    }
    IShovelLogic getShovelLogic() {
        return shovelLogic_;
    }
}
