package jp.sourceforge.shovel.service.impl;

import static org.seasar.framework.container.ContainerConstants.*;

import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.SingletonS2ContainerFactory;

import jp.sourceforge.shovel.RepliesType;
import jp.sourceforge.shovel.SortOrderType;
import jp.sourceforge.shovel.SortType;
import jp.sourceforge.shovel.entity.IServerFile;
import jp.sourceforge.shovel.entity.IUser;
import jp.sourceforge.shovel.exception.ApplicationException;
import jp.sourceforge.shovel.interceptor.CsrfInterceptor;
import jp.sourceforge.shovel.logic.IDirectoryLogic;
import jp.sourceforge.shovel.service.IDirectoryService;

public class DirectoryServiceImpl implements IDirectoryService {
    Map<Long, IUser> userMapByPK_;
    Map<String, IUser> userMapByFK_;
    
    public DirectoryServiceImpl() {
        userMapByPK_ = new HashMap<Long, IUser>();
        userMapByFK_ = new HashMap<String, IUser>();
    }
    void cacheUser(IUser user) {
        if (user != null && fetchUser(user.getUserId()) == null) {
            userMapByPK_.put(user.getUserId(), user);
            userMapByFK_.put(user.getForeignKey(), user);
        }
    }
    void cacheUsers(IUser[] users) {
        for (IUser user : users) {
            cacheUser(user);
        }
    }
    void purgeAllUser() {
        userMapByPK_.clear();
        userMapByFK_.clear();
    }
    void purgeUser(long userId) {
        userMapByPK_.remove(userId);
        IUser user = userMapByPK_.get(userId);
        String foreignKey = user.getForeignKey();
        userMapByFK_.remove(foreignKey);
    }
    IUser fetchUser(long userId) {
        return userMapByPK_.get(userId);
    }
    IUser fetchUser(String foreignKey) {
        return userMapByFK_.get(foreignKey);
    }
    
    /////
    
    String once_;
    
    public IUser createUser(String displayName, String foreignKey, String password, String email,
            IServerFile serverFile, String location, String timeZoneId, String description, boolean cache) throws ApplicationException {
        if (cache) {
            IUser user = getUser(foreignKey);
            if (user != null) {
                throw new ApplicationException("");
            }
        }
        long serverFileId = 0;
        if (serverFile != null) {
            serverFileId = serverFile.getServerFileId();
        }
        long creatorId = getLoginUser().getUserId();
        IUser user = getDirectoryLogic().createUser(displayName, foreignKey, password, email,
                serverFileId, location, timeZoneId, description, creatorId);
        return user;
    }
    public IUser createTemporaryUser() {
        return getDirectoryLogic().createTemporaryUser();
    }
    public IUser login(String foreignKey, String password, boolean administrator, boolean once) {
        if (foreignKey == null || foreignKey.length() <= 0) {
            //外部キーが空なら未ログイン
            return null;
        }
        if (password == null) {
            password = "";
        }
        IUser user = getLoginUser();
        if (user == null) {
            user = fetchUser(foreignKey);
            if (user == null) {
                user = getDirectoryLogic().getUserByForeignKey(foreignKey);
                if (user != null && user.isRemove()) {
                    user = null;
                } else {
                    cacheUser(user);
                }
            }
        }
        if (user != null) {
            if (password.compareTo(user.getPassword()) == 0 &&
                (!administrator || (administrator && user.isAdministrator()))) {
                if (once) {
                    once_ = user.getForeignKey();
                } else {
                    getSession().setAttribute("login", user.getForeignKey());
                }
                CsrfInterceptor.setCsrfTicket(user);
            } else {
                user = null;
            }
        }
        return user;
    }
    public void logout() {
        getSession().removeAttribute("login");
    }
    public IUser getLoginUser() {
        String foreignKey = once_;
        if (foreignKey == null) {
            foreignKey = (String)getSession().getAttribute("login");
        }
        IUser user = getUser(foreignKey);
        if (user != null && user.isRemove()) {
            return null;
        }
        return user;
    }
    public IUser getUser(long userId) {
        IUser user = fetchUser(userId);
        if (user == null) {
            user = getDirectoryLogic().getUser(userId);
            cacheUser(user);
        }
        return user;
    }
    public IUser getUser(String foreignKey) {
        if (foreignKey == null || foreignKey.length() <= 0) {
            return null;
        }
        IUser user = fetchUser(foreignKey);
        if (user == null) {
            user = getDirectoryLogic().getUserByForeignKey(foreignKey);
            cacheUser(user);
        }
        return user;
    }
    public IUser[] getUsers(boolean remove, SortType sortType, SortOrderType sortOrderType, int page) {
        IUser user = getLoginUser();
        //TODO
        int limit = 10;
        if (user != null) {
            limit = user.getViewLines();
        }
        IUser[] users = getDirectoryLogic().getUsers(remove, sortType, sortOrderType, page * limit, limit + 1);
        cacheUsers(users);
        return users;
    }
    public IUser[] getUsers(long[] userIds) {
        IUser[] users = getDirectoryLogic().getUsers(userIds);
        cacheUsers(users);
        return users;
    }
    public IUser[] getUsers(String[] foreignKeys) {
        IUser[] users = getDirectoryLogic().getUsers(foreignKeys);
        cacheUsers(users);
        return users;
    }
    public IUser[] getRandUsers() {
        IUser[] users = getDirectoryLogic().getRandUsers(false, 0, 10);
        cacheUsers(users);
        return users;
    }
    int decrementAllFriends(long[] userIds) {
        return getDirectoryLogic().decrementAllFriends(userIds);
    }
    int decrementAllFollower(long[] userIds) {
        return getDirectoryLogic().decrementAllFollower(userIds);
    }
    //TODO 将来はキューでまったり
    public int removeUser(long userId) throws ApplicationException {
        return removeUsers(new long[] {userId});
    }
    //TODO 将来はキューでまったり
    public int removeUsers(long[] userIds) {
        decrementAllFriends(userIds);
        decrementAllFollower(userIds);
        return getDirectoryLogic().updateRemove(userIds);
    }
    public int updateUser(IUser user) throws ApplicationException {
        return getDirectoryLogic().updateUser(user);
    }
    public int updateUserFromImportCsv(IUser user) throws ApplicationException {
        return getDirectoryLogic().updateUserFromImportCsv(user);
    }
    public int updateUserFromSettings(IUser user) throws ApplicationException {
        return getDirectoryLogic().updateUserFromSettings(user);
    }
    public int updatePassword(String password) {
        IUser user = getLoginUser();
        return getDirectoryLogic().updatePassword(user.getUserId(), password);
    }
    public int updatePicture(long serverFileId) {
        IUser user = getLoginUser();
        return getDirectoryLogic().updatePicture(user.getUserId(), serverFileId);
    }
    public int updateReplies(RepliesType repliesType) {
        IUser user = getLoginUser();
        return getDirectoryLogic().updateReplies(user.getUserId(), repliesType);
    }
    public int incrementFavorites(String foreignKey) {
        return getDirectoryLogic().incrementFavorites(foreignKey);
    }
    public int incrementFollowers(String foreignKey) {
        return getDirectoryLogic().incrementFollower(foreignKey);
    }
    public int incrementFriends(String foreignKey) {
        return getDirectoryLogic().incrementFriends(foreignKey);
    }
    public int incrementGivenFavorites(long userId) {
        return getDirectoryLogic().incrementGivenFavorites(userId);
    }
    public int incrementDirectMessages(String foreignKey) {
        return getDirectoryLogic().incrementDirectMessages(foreignKey);
    }
    public int incrementStatuses(String foreignKey) {
        return getDirectoryLogic().incrementStatus(foreignKey);
    }
    public int decrementFavorites(String foreignKey) {
        return getDirectoryLogic().decrementFavorites(foreignKey);
    }
    public int decrementFavoritesByStatus(long statusId) {
        return getDirectoryLogic().decrementFavoritesByStatus(statusId);
    }
    public int decrementFollowers(String foreignKey) {
        return getDirectoryLogic().decrementFollower(foreignKey);
    }
    public int decrementFriends(String foreignKey) {
        return getDirectoryLogic().decrementFriends(foreignKey);
    }
    public int decrementGivenFavorites(long userId) {
        return getDirectoryLogic().decrementGivenFavorites(userId);
    }
    public int decrementStatuses(String foreignKey) {
        return getDirectoryLogic().decrementStatus(foreignKey);
    }
    public int decrementDirectMessages(String foreignKey) {
        return getDirectoryLogic().decrementDirectMessages(foreignKey);
    }
    
    /////
    
    S2Container getContainer() {
        return SingletonS2ContainerFactory.getContainer();
    }
    HttpSession getSession() {
        return (HttpSession)getContainer().getComponent(SESSION_NAME);
    }
    HttpServletRequest getRequest() {
        return (HttpServletRequest)getContainer().getComponent(REQUEST_NAME);
    }
    IDirectoryLogic getDirectoryLogic() {
        return(IDirectoryLogic)getContainer().getComponent(IDirectoryLogic.class);
    }
}
