package jp.sourceforge.shovel.logic.impl;

import static java.util.Locale.*;
import static jp.sourceforge.shovel.SortOrderType.*;
import static jp.sourceforge.shovel.SortType.*;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.seasar.framework.container.S2Container;


import jp.sourceforge.shovel.DeviceType;
import jp.sourceforge.shovel.RepliesType;
import jp.sourceforge.shovel.SortOrderType;
import jp.sourceforge.shovel.SortType;
import jp.sourceforge.shovel.dao.IDedicatedClientDao;
import jp.sourceforge.shovel.dao.IDeviceDao;
import jp.sourceforge.shovel.dao.IDirectMessageDao;
import jp.sourceforge.shovel.dao.IFavoriteDao;
import jp.sourceforge.shovel.dao.IFriendshipDao;
import jp.sourceforge.shovel.dao.IStatusDao;
import jp.sourceforge.shovel.entity.IConnectionWrapper;
import jp.sourceforge.shovel.entity.IDedicatedClient;
import jp.sourceforge.shovel.entity.IDevice;
import jp.sourceforge.shovel.entity.IDirectMessage;
import jp.sourceforge.shovel.entity.IFavorite;
import jp.sourceforge.shovel.entity.IFriendship;
import jp.sourceforge.shovel.entity.IStatus;
import jp.sourceforge.shovel.entity.IStatusToken;
import jp.sourceforge.shovel.exception.ApplicationException;
import jp.sourceforge.shovel.logic.IShovelLogic;

public class ShovelLogicImpl implements IShovelLogic {
    /////
    //status
    static final String URL = "https?://[a-zA-Z0-9/_.?#&;=$+:@%~,\\-]+";
    static final String REPLY = "^@([\\w|-]+)\\s";
    static final String OPEN_REPLY = "^[^\\s]@([\\w|-]+)\\s";
    static final String MULTI_REPLY = "\\s@([\\w|-]+)\\s";
    
    SortedMap<Long, IStatusToken> parseStatus(String data, int type, String pattern, SortedMap<Long, IStatusToken> tokenMap) {
        if (tokenMap == null) {
            tokenMap = new TreeMap<Long, IStatusToken>();
        }
        
        Matcher matcher = Pattern.compile(pattern).matcher(data);
        while (matcher.find()) {
            IStatusToken token = (IStatusToken)getContainer().getComponent(IStatusToken.class);
            token.setType(type);
            token.setStart(matcher.start());
            token.setEnd(matcher.end());
            
            int groupCount = matcher.groupCount();
            List<String> regs = new ArrayList<String>();
            for(int i = 0; i <= groupCount; i++) {
                regs.add(matcher.group(i));
            }
            token.setRegs(regs);
            tokenMap.put((long)matcher.start(), token);
        }
        return tokenMap;
    }
    public IStatusToken[] parseStatus(String status) {
        if (status != null && status.length() > 0) {
            SortedMap<Long, IStatusToken> tokenMap = parseStatus(status, 0, URL, null);
            parseStatus(status, 1, REPLY, tokenMap);
            parseStatus(status, 2, OPEN_REPLY, tokenMap);
            parseStatus(status, 3, MULTI_REPLY, tokenMap);
            return tokenMap.values().toArray(new IStatusToken[tokenMap.size()]);
        }
        return null;
    }
    long parseRFC2822DateTime(String since) throws ApplicationException {
        try {
            DateFormat format = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", ENGLISH);
            Date date = format.parse(since);
            return date.getTime();
        } catch (ParseException e) {
            throw new ApplicationException("");
        }
    }
    public IStatus getStatus(long statusId) {
        return getStatusDao().find(statusId);
    }
    public IStatus getRecent(long userId) {
        return getStatusDao().findRecent(userId);
    }
    public IStatus[] getRecents(long[] userIds) {
        IStatus[] statuses = getStatusDao().findAllRecent(userIds, 0, userIds.length);
        if (statuses == null || statuses.length <= 0) {
            return statuses;
        }
        long[] statusIds = new long[statuses.length];
        int i = 0;
        for (IStatus status : statuses) {
            statusIds[i++] = status.getStatusId();
        }
        return getStatusDao().findAll(statusIds, STATUS_ID, DESC, 0, statusIds.length);
    }
    public IStatus[] getFavorites(long userId, int page, int limit) {
        return getStatusDao().findAllFavorite(userId, page, limit);
    }
    public IStatus[] getStatuses(int offset, int limit) {
        return getStatusDao().findAll(null, STATUS_ID, DESC, offset, limit);
    }
    public IStatus[] getStatuses(long sinceId, int limit) {
        return getStatusDao().findAllBySince(sinceId, limit);
    }
    public IStatus[] getStatuses(SortType sortType, SortOrderType sortOrderType, int limit) {
        return getStatusDao().findAll(null, sortType, sortOrderType, 0, limit);
    }
    public IStatus[] getStatuses(long userId, boolean withFriends, RepliesType repliesType, int offset, int limit) {
        if (withFriends) {
            return getStatusDao().findAllWithFriends(userId, repliesType, offset, limit);
        } else {
            return getStatusDao().findAllSingle(userId, offset, limit);
        }
    }
    public IStatus[] getStatuses(long userId, boolean withFriends, RepliesType repliesType, String since, int limit) throws ApplicationException {
        long sentTime = parseRFC2822DateTime(since);
        if (withFriends) {
            return getStatusDao().findAllWithFriendsSinceTime(userId, repliesType, sentTime, limit);
        } else {
            return getStatusDao().findAllSingleSinceTime(userId, sentTime, limit);
        }
    }
    public IStatus[] getReplies(long userId, int offset, int limit) {
        return getStatusDao().findAllReply(userId, offset, limit);
    }
    public IStatus updateStatus(String body, String source, long referenceSenderId, boolean open, long senderId) throws ApplicationException {
        //まあ、まずないと思うけど
        if (senderId <= 0) {
            throw new ApplicationException("");
        }
        //空POST禁止
        if (body == null || body.length() <= 0 ||
            source == null || source.length() <= 0) {
            //基本、クライアント側で防ぐ。最終防衛線。
            throw new ApplicationException("");
        }
        
        long referenceId = 0;
        if (referenceSenderId > 0) {
            IStatus reference = getStatusDao().findRecent(referenceSenderId);
            if (reference != null) {
                referenceId = reference.getStatusId();
            }
        }
        
        long sentTime = Calendar.getInstance().getTimeInMillis();
        IStatus status = (IStatus)getContainer().getComponent(IStatus.class);
        status.setBody(body);
        status.setSource(source);
        status.setReferenceId(referenceId);
        status.setReferenceSenderId(referenceSenderId);
        status.setSenderId(senderId);
        status.setSentTime(sentTime);
        status.setOpen(open);
        getStatusDao().insert(status);
        
        for (IConnectionWrapper connectionWrapper : connectionWrapperMap_.values()) {
            connectionWrapper.pushMessage(status);
        }
        return status;
    }
    public void recvStatus(String status, String source, long referenceSenderId, boolean open, long senderId) throws ApplicationException {
        updateStatus(status, source, referenceSenderId, open, senderId);
    }
    public int incrementGivenFavorites(long statusId) {
        return getStatusDao().updateIncrementGivenFavorites(statusId);
    }
    public int decrementGivenFavorites(long statusId) {
        return getStatusDao().updateDecrementGivenFavorites(statusId);
    }
    public int updateStatusRemove(long statusId) {
        return getStatusDao().updateRemove(statusId);
    }
    public int clearStatusReference(long referenceId) {
        return getStatusDao().updateReference(referenceId);
    }
    public int removeStatus() {
        return getStatusDao().delete();
    }
    
    /////
    //direct message
    public IDirectMessage getDirectMessage(long directMessageId) {
        return getDirectMessageDao().find(directMessageId);
    }
    public IDirectMessage[] getDirectMessages(boolean sent, long userId, int offset, int limit) {
        if (sent) {
            return getDirectMessageDao().findAllSent(userId, offset, limit);
        } else {
            return getDirectMessageDao().findAllInbox(userId, offset, limit);
        }
    }
    public IDirectMessage[] getDirectMessages(boolean sent, long userId, long sinceId, int limit) {
        if (sent) {
            return getDirectMessageDao().findAllSentSinceId(userId, sinceId, limit);
        } else {
            return getDirectMessageDao().findAllInboxSinceId(userId, sinceId, limit);
        }
    }
    public IDirectMessage[] getDirectMessages(boolean sent, long userId, String since, int limit) throws ApplicationException {
        long sentTime = parseRFC2822DateTime(since);
        if (sent) {
            return getDirectMessageDao().findAllSentSinceTime(userId, sentTime, limit);
        } else {
            return getDirectMessageDao().findAllInboxSinceTime(userId, sentTime, limit);
        }
    }
    public IDirectMessage sendDirectMessage(long senderId, long recipientId,
            String text, String source) throws ApplicationException {
        //まあ、まずないと思うが
        if (senderId <= 0) {
            throw new ApplicationException("");
        }
        //空POST禁止
        if (text == null || text.length() <= 0 ||
            source == null || source.length() <= 0) {
            //基本、クライアント側で防ぐ。最終防衛線。
            throw new ApplicationException("");
        }
        long sentTime = Calendar.getInstance().getTimeInMillis();
        
        
        IDirectMessage directMessage = (IDirectMessage)getContainer().getComponent(IDirectMessage.class);
        directMessage.setText(text);
        directMessage.setRecipientId(recipientId);
        directMessage.setSenderId(senderId);
        directMessage.setSentTime(sentTime);
        directMessage.setSource(source);
        directMessage.setInbox(true);
        directMessage.setSent(true);
        getDirectMessageDao().insert(directMessage);
        return directMessage;
    }
    public int updateRemoveDirectMessage(long directMessageId, boolean sent) {
        return getDirectMessageDao().updateRemove(directMessageId, sent);
    }
    
    /////
    //friendship
    public IFriendship getFriendship(long activeId, long passiveId) {
        return getFriendshipDao().find(activeId, passiveId);
    }
    public IFriendship[] getFriends(long activeId, int offset, int limit) {
        return getFriendshipDao().findAllFriends(activeId, offset, limit);
    }
    public IFriendship[] getFriends(long activeId, long[] passiveIds) {
        return getFriendshipDao().findAllFriendsBy(activeId, passiveIds, 0, passiveIds.length);
    }
    public IFriendship[] getFollowers(long passiveId, int offset, int limit) {
        return getFriendshipDao().findAllFollower(passiveId, offset, limit);
    }
    public IFriendship[] getFollowers(long passiveId, DeviceType deviceType, long friendshipId, int limit) {
        return getFriendshipDao().findAllFollowerByDevice(passiveId, deviceType, friendshipId, limit);
    }
    public IFriendship[] getFollowers(long passiveId, long[] activeIds) {
        return getFriendshipDao().findAllFollowerBy(passiveId, activeIds, 0, activeIds.length);
    }
    public IFriendship createFriendship(long activeId, long passiveId) {
        IFriendship friendship = (IFriendship)getContainer().getComponent(IFriendship.class);
        friendship.setActiveId(activeId);
        friendship.setPassiveId(passiveId);
        //お知らせ機能の初期値はOn
        friendship.setNotify(true);
        getFriendshipDao().insert(friendship);
        return friendship;
    }
    public int updateFriendshipNotify(long activeId, long passiveId, boolean notify) {
        return getFriendshipDao().updateNotify(activeId, passiveId, notify);
    }
    public int removeFriendship(long activeId, long passiveId) {
        IFriendship friendship = (IFriendship)getContainer().getComponent(IFriendship.class);
        friendship.setActiveId(activeId);
        friendship.setPassiveId(passiveId);
        return getFriendshipDao().delete(friendship);
    }
    
    /////
    //favorite
    public IFavorite getFavorite(long statusId, long userId) {
        return getFavoriteDao().findFavorite(statusId, userId);
    }
    public IFavorite[] getFavorites(long[] statusIds, long userId) {
        return getFavoriteDao().findAllFavorite(statusIds, userId, statusIds.length);
    }
    public IFavorite createFavorite(long statusId, long userId) {
        long createdTime = Calendar.getInstance().getTimeInMillis();
        IFavorite favorite = (IFavorite)getContainer().getComponent(IFavorite.class);
        favorite.setCreatedTime(createdTime);
        favorite.setStatusId(statusId);
        favorite.setUserId(userId);
        getFavoriteDao().insert(favorite);
        return favorite;
    }
    public int removeFavorite(long statusId, long userId) {
        return getFavoriteDao().delete(statusId, userId);
    }
    
    /////
    //device
    
    public IDevice createDevice(IDevice device) {
        getDeviceDao().insert(device);
        return device;
    }
    public int updateDevice(IDevice device) {
        return getDeviceDao().update(device);
    }
    
    /////
    //client
    
    Map<String, IDedicatedClient> clientMap_ = new HashMap<String, IDedicatedClient>();
    
    public IDedicatedClient createClient(String key, String url, String version) {
        IDedicatedClient client = (IDedicatedClient)getContainer().getComponent(IDedicatedClient.class);
        client.setText(key);
        client.setUrl(url);
        client.setVersion(version);
        getDedicatedClientDao().insert(client);
        clientMap_.put(key, client);
        return client;
    }
    public int updateClient(IDedicatedClient client) {
        clientMap_.put(client.getText(), client);
        return getDedicatedClientDao().update(client);
    }
    public IDedicatedClient getClient(String key) {
        IDedicatedClient client = clientMap_.get(key);
        if (client != null) {
            return client;
        }
        return getDedicatedClientDao().findByKey(key);
    }
    public IDedicatedClient[] getClients(String[] texts) {
        List<IDedicatedClient> clientList = new ArrayList<IDedicatedClient>();
        List<String> textList = new ArrayList<String>();
        for (String text : texts) {
            IDedicatedClient client = clientMap_.get(text);
            if (client == null) {
                textList.add(text);
            } else {
                clientList.add(client);
            }
        }
        if (textList.size() > 0) {
            texts = textList.toArray(new String[textList.size()]);
            IDedicatedClient[] clients = getDedicatedClientDao().findAllByText(texts, 0, texts.length);
            for (IDedicatedClient client : clients) {
                clientMap_.put(client.getText(), client);
                clientList.add(client);
            }
        }
        return clientList.toArray(new IDedicatedClient[clientList.size()]);
    }
    public int removeClient(String text) {
        clientMap_.remove(text);
        return getDedicatedClientDao().deleteByText(text);
    }
    
    /////
    
    Map<String, IConnectionWrapper> connectionWrapperMap_;
    
    public void setConnectionWrapperMap(Map<String, IConnectionWrapper> connectionWrapperMap) {
        connectionWrapperMap_ = connectionWrapperMap;
    }
    public IConnectionWrapper getConnectionWrapper(DeviceType deviceType) {
        return connectionWrapperMap_.get(deviceType.getKey());
    }
    public IConnectionWrapper[] getConnectionWrappers() {
        Collection<IConnectionWrapper> connections = connectionWrapperMap_.values();
        return connections.toArray(new IConnectionWrapper[connections.size()]);
    }
    
    /////
    
    S2Container container_;
    
    public void setContainer(S2Container container) {
        container_ = container;
    }
    S2Container getContainer() {
        return container_;
    }
    public IStatusDao getStatusDao() {
        return (IStatusDao)getContainer().getComponent(IStatusDao.class);
    }
    public IDirectMessageDao getDirectMessageDao() {
        return (IDirectMessageDao)getContainer().getComponent(IDirectMessageDao.class);
    }
    public IFavoriteDao getFavoriteDao() {
        return (IFavoriteDao)getContainer().getComponent(IFavoriteDao.class);
    }
    public IFriendshipDao getFriendshipDao() {
        return (IFriendshipDao)getContainer().getComponent(IFriendshipDao.class);
    }
    public IDeviceDao getDeviceDao() {
        return (IDeviceDao)getContainer().getComponent(IDeviceDao.class);
    }
    public IDedicatedClientDao getDedicatedClientDao() {
        return (IDedicatedClientDao)getContainer().getComponent(IDedicatedClientDao.class);
    }
}
