/*-
 * Copyright (c) 2008 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SI_WEBXML_BASIC_MAP_H
#define SI_WEBXML_BASIC_MAP_H

#include <algorithm>
#include <iterator>
#include <map>
#include <boost/thread/mutex.hpp>

namespace si {
namespace webxml {

template <typename KeyType, typename ValueType>
class basic_map {
public :
    basic_map()
    {
        _mutex = new boost::mutex();
        _lock = new boost::mutex::scoped_lock(*_mutex);
        _lock->unlock();
    }

    basic_map(const basic_map& r)
    {
        _mutex = new boost::mutex();
        _lock = new boost::mutex::scoped_lock(*_mutex);
        _lock->unlock();
        _map = r._map;
    }

    virtual ~basic_map()
    {
        delete _lock;
        delete _mutex;
    }

    void add(const KeyType& k, const ValueType& v)
    {
        _lock->lock();
        _map.insert(std::make_pair(k, v));
        _lock->unlock();
    }

    ValueType find(const KeyType& s) const
    {
        _lock->lock();
        typename std::map<KeyType, ValueType>::const_iterator i = _map.find(s);
        _lock->unlock();
        return i != _map.end() ? i->second : ValueType();
    }

    void remove(const KeyType& s)
    {
        _lock->lock();
        typename std::map<KeyType, ValueType>::iterator i = _map.find(s);
        if (i != _map.end())
            _map.erase(i);
        _lock->unlock();
    }

    std::vector<KeyType> names() const
    {
        _lock->lock();

        std::vector<KeyType> v;
        typename std::map<KeyType, ValueType>::const_iterator i = _map.begin();
        for (; i != _map.end(); i++)
            v.push_back(i->first);
        _lock->unlock();
        return v;
    }

    std::vector<ValueType> values() const
    {
        std::vector<ValueType> v;

        _lock->lock();
        typename std::map<KeyType, ValueType>::const_iterator i = _map.begin();
        for (; i != _map.end(); i++)
            v.push_back(i->second);
        _lock->unlock();
        return v;
    }

    void insert(const typename std::map<KeyType, ValueType>::value_type& v)
    {
        _lock->lock();
        _map.insert(v);
        _lock->unlock();
    }

    std::map<KeyType, ValueType> get() const
    {
        return _map;
    }

    basic_map<KeyType, ValueType>&
    operator+=(const basic_map<KeyType, ValueType>& r)
    {
        _lock->lock();
        _map.insert(r._map.begin(), r._map.end());
        _lock->unlock();
        return *this;
    }

private :
    typename std::map<KeyType, ValueType> _map;
    mutable boost::mutex* _mutex;
    mutable boost::mutex::scoped_lock* _lock;
};

} // namespace webxml
} // namespace si

#endif // SI_WEBXML_BASIC_MAP_H
