/*-
 * Copyright (c) 2008 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SI_WEBXML_BASIC_LIST_H
#define SI_WEBXML_BASIC_LIST_H

#include <algorithm>
#include <list>
#include <boost/algorithm/string/trim.hpp>
#include <boost/thread/mutex.hpp>

namespace si {
namespace webxml {

template <class ValueType>
class basic_list {
public :
    basic_list()
    {
        _mutex = new boost::mutex();
        _lock = new boost::mutex::scoped_lock(*_mutex);
        _lock->unlock();
    }

    basic_list(const basic_list& r)
    {
        _mutex = new boost::mutex();
        _lock = new boost::mutex::scoped_lock(*_mutex);
        _lock->unlock();
        _list = r._list;
    }

    virtual ~basic_list()
    {
        delete _lock;
        delete _mutex;
    }

    void add(const ValueType& s)
    {
        _lock->lock();
        _list.push_back(boost::trim_copy_if(s, boost::is_any_of("\r\n\t ")));
        _lock->unlock();
    }

    bool find(const ValueType& s) const
    {
        _lock->lock();
        typename std::list<ValueType>::const_iterator i = _list.find(s);
        _lock->unlock();
        return i != _list.end() ? true : false;
    }

    void remove(const ValueType& s)
    {
        _lock->lock();
        typename std::list<ValueType>::iterator i = _list.find(s);
        if (i != _list.end())
            _list.erase(i);
        _lock->unlock();
    }

    std::list<ValueType> list() const
    {
        return _list;
    }

    std::list<ValueType> get() const
    {
        return _list;
    }

    basic_list<ValueType>& operator+=(const basic_list<ValueType>& r)
    {
        _lock->lock();
        std::copy(r._list.begin(), r._list.end(), std::back_inserter(_list));
        _lock->unlock();
        return *this;
    }

private :
    typename std::list<ValueType> _list;
    mutable boost::mutex* _mutex;
    mutable boost::mutex::scoped_lock* _lock;
};

} // namespace webxml
} // namespace si

#endif // SI_WEBXML_BASIC_LIST_H
