/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/bind.hpp>

#include <sl/java/lang/IllegalStateException.h>
#include <servlet/ServletResponse.h>
using namespace servlet;
using namespace std;

#include <si/core/ServletResponseImpl.h>
#include <si/core/HttpServletResponseImpl.h>
using namespace si::core;

//
// Constructor/Destructor
//

ServletResponseImpl::ServletResponseImpl(
        interface::Connection& connect,
        interface::Context& context)
:    _response(connect.response()), _context(context),
      _buffer(0),
      _buffer_size(0),
      _commit_header(false),
      _commit_body(false)
{
    core::cbstreambuf::handler_t handler = 
            boost::bind(&ServletResponseImpl::bufferFullHandler, this, _1, _2);

    _cbstream.callback(handler); 
    _writer = new BasicWriter(&_cbstream);
    _print_writer = new sl::java::io::PrintWriter(_writer);

    setBufferSize(1024 * 8);
}


ServletResponseImpl::~ServletResponseImpl()
{
    if (_buffer)
        delete _buffer;
    delete _writer;
    delete _print_writer;
}


//
// Member functions
//


sl::java::io::PrintWriter &ServletResponseImpl::getWriter() throw(ServletException)
{
    if (!_print_writer)
        throw ServletException("Writer is Null");
    return *_print_writer;
}


void ServletResponseImpl::setContentType(const string &type)
{
    _response.setContentType(type);
}


ServletOutputStream& ServletResponseImpl::getOutputStream()
{
    _output_stream.setStream(*_writer);
    return _output_stream;
}


void ServletResponseImpl::setContentLength(int i)
{
    _response.setContentLength(i);
    _length = i;
}


std::string ServletResponseImpl::getCharacterEncoding()
{
    return _response.getCharacterEncoding();
}


std::string ServletResponseImpl::getContentType()
{
    return _response.getContentType();
}


sl::java::util::Locale ServletResponseImpl::getLocale()
{
    return sl::java::util::Locale();
}


void ServletResponseImpl::setLocale(const sl::java::util::Locale& )
{
    // XXXXX
}


void ServletResponseImpl::setBufferSize(size_t n)
{
    if (_commit_header || _commit_body)
        throw sl::java::lang::IllegalStateException("");

    if (n == 0)
        return;

    if (_buffer)
        delete _buffer;

    _buffer = new char[n];
    _buffer_size = n;
    _cbstream.setbuf(_buffer, _buffer_size);
}


size_t ServletResponseImpl::getBufferSize()
{
    return _buffer_size;
}


bool ServletResponseImpl::isCommitted()
{
    return _commit_body;
}


void ServletResponseImpl::flushBuffer()
{ 
#ifdef DEBUG
std::cerr << "ServletResponseImpl::flushBuffer" << std::endl;
#endif
    if (isCommitted())
        throw std::runtime_error("ServletResponseImpl :already committed.");

    _cbstream.flush();
    _response.flush();
    _commit_body = true;
}


void ServletResponseImpl::reset()
{ 
    if (_commit_header || _commit_body)
        throw sl::java::lang::IllegalStateException("");

    _response.reset();
    _length = 0;

    resetBuffer();
}


void ServletResponseImpl::resetBuffer()
{
        _context.getLogger()->log("ServletResponse resetBuffer",
                                   si::interface::Logger::DEBUG_LOG);

    if (_commit_header || _commit_body)
        throw sl::java::lang::IllegalStateException("already committed.");

    //_ostringstream.rdbuf()->str("");
}

void ServletResponseImpl::bufferFullHandler(const char* s, size_t n)
{
#ifdef DEBUG
    std::cerr << "ServletResponseImpl::bufferFullHandler" << std::endl;
#endif
    if (!_commit_header) {
        _response.flushHeader();
        _commit_header = true;
    }
    _response.setContent(std::string(s, n));
    _response.flushBody();
}

void ServletResponseImpl::committed()
{
    _context.getLogger()->log("ServletResponse committed",
                              si::interface::Logger::DEBUG_LOG);
    _commit_body = true;
}
