/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/filesystem.hpp>
#include <servlet/ServletContext.h>
using namespace std;
using namespace servlet;

#include <si/core/ServletRequestImpl.h>
#include <si/core/RequestDispatcherImpl.h>
using namespace si::core;


//
// Constructor/Destructor
//

ServletRequestImpl::ServletRequestImpl(
        interface::Connection& connect,
        interface::Context& context)
 :    _request(connect.request()),
    _response(connect.response()),
    _context(context),
    _servlet_context(context.getServletContext()),
        _notify_listener(true),
    _input_buffer(_request.getInputStream())
{
    _input_reader = new BasicReader(_input_buffer, _request.getContentLength());
    _input_stream =
        new ServletInputStreamImpl(_input_buffer, _request.getContentLength());
}


ServletRequestImpl::~ServletRequestImpl()
{
    delete _input_reader;
    delete _input_stream;
}


//
// Member functions
//

sl::java::lang::Object ServletRequestImpl::getAttribute(const string& s)
{
    return Attribute::getAttribute(s);
}

void ServletRequestImpl::setAttribute(const string& s, sl::java::lang::Object o)
{
    bool exist = Attribute::getAttribute(s) ? true : false;

    Attribute::setAttribute(s, o);

    servlet::ServletRequestAttributeEvent e(_servlet_context, *this, s, o);

    if (_notify_listener) {
    listener_list_t::iterator i = _listener_list.begin();
    for (; i != _listener_list.end(); i++) {
        if (exist)
            (*i)->attributeReplaced(e);
        else
            (*i)->attributeAdded(e);
    }
    }
}

void ServletRequestImpl::removeAttribute(const string& s)
{
    sl::java::lang::Object o = Attribute::getAttribute(s);
    Attribute::removeAttribute(s);

    servlet::ServletRequestAttributeEvent e(_servlet_context, *this, s, o);

    if (_notify_listener) {
    listener_list_t::iterator i = _listener_list.begin();
    for (; i != _listener_list.end(); i++)
        (*i)->attributeRemoved(e);
    }
}


sl::java::util::Enumeration ServletRequestImpl::getAttributeNames()
{
    return Attribute::getAttributeNames();
}


string ServletRequestImpl::getParameter(const string &s)
{
    return _request.getParameter(s);
}


sl::java::util::Enumeration ServletRequestImpl::getParameterNames()
{
    std::vector<sl::java::lang::Object> objs;
    std::vector<std::string> names(_request.getParameterNames());
    std::vector<std::string>::const_iterator i = names.begin();
    for (; i != names.end(); i++) {
        objs.push_back(sl::java::lang::Object(*i));
    }
    return sl::java::util::Enumeration(objs);
}


std::vector<std::string>
ServletRequestImpl::getParameterValues(const std::string& s)
{
    return _request.getParameterValues(s);
}

std::multimap<std::string, std::string> ServletRequestImpl::getParameterMap()
{
    return _request.getParameterMap();
}


std::string ServletRequestImpl::getCharacterEncoding()
{
    return _request.getCharacterEncoding();
}

void ServletRequestImpl::setCharacterEncoding(const std::string&)
{
    // XXXXX
}


int ServletRequestImpl::getContentLength()
{
    return _request.getContentLength();
}


std::string ServletRequestImpl::getContentType()
{
    return _request.getContentType();
}

ServletInputStream &ServletRequestImpl::getInputStream()
{
    return *_input_stream;
}

sl::java::io::BufferedReader ServletRequestImpl::getReader()
{
    return sl::java::io::BufferedReader(_input_reader);
}

sl::java::util::Locale ServletRequestImpl::getLocale()
{
    return sl::java::util::Locale();
}

sl::java::util::Enumeration ServletRequestImpl::getLocales()
{
    throw std::runtime_error("Not Implement");
    return EnumerationImpl(std::vector<std::string>());
}

std::string ServletRequestImpl::getProtocol()
{
    return _request.getProtocol();
}

std::string ServletRequestImpl::getScheme()
{
    return _request.getScheme();
}

std::string ServletRequestImpl::getServerName()
{
    return _request.getServerName();
}

int ServletRequestImpl::getServerPort()
{
    return _request.getServerPort();
}

std::string ServletRequestImpl::getLocalAddr()
{
    return _request.getLocalAddr();
}

std::string ServletRequestImpl::getLocalName()
{
    return _request.getLocalName();
}

int ServletRequestImpl::getLocalPort()
{
    return _request.getLocalPort();
}

std::string ServletRequestImpl::getRemoteAddr()
{
    return _request.getRemoteAddr();
}

std::string ServletRequestImpl::getRemoteHost()
{
    return _request.getRemoteHost();
}

int ServletRequestImpl::getRemotePort()
{
    return _request.getRemotePort();
}

bool ServletRequestImpl::isSecure()
{
    return _request.isSecure();
}

std::auto_ptr<servlet::RequestDispatcher>
ServletRequestImpl::getRequestDispatcher(const std::string& uri)
{
    RequestDispatcherImpl *r = new RequestDispatcherImpl(&_context);

    if (uri.empty()) {
        _servlet_context.log("ServletContext::getRequestDispatcher failed. Invalid URI: " + uri);
        return std::auto_ptr<servlet::RequestDispatcher>(r);
    }

    /*
     * Хѥξϥƥȥ롼ȤΥѥѴ
     */
    std::string normalize_uri(uri);
    if (normalize_uri[0] != '/') {
        try {
            /*
             * RequestURI ContextRoot ͤ
             * ǥѥåΥѥϢ뤷ͤ
             */
            std::string servlet_path =
                           _request.getRequestURI().erase(0, _context.path().length());

            boost::filesystem::path p(servlet_path + "/" + normalize_uri);
            p.normalize();
            normalize_uri = p.string();

        } catch (boost::filesystem::filesystem_error& e) {
            _servlet_context.log("ServletContext::getRequestDispatcher failed. Invalid URI: " + uri);
            return std::auto_ptr<servlet::RequestDispatcher>(r);
        }
    }
    /*  '/' Ϥޤʤޤ ".." ޤ */
    if (normalize_uri[0] != '/' || normalize_uri.find("..") != std::string::npos) {
        _servlet_context.log("ServletContext::getRequestDispatcher failed. Invalid URI: " + uri);
        return std::auto_ptr<servlet::RequestDispatcher>(r);
    }

    r->init(normalize_uri, RequestDispatcherImpl::request_dispatch);
    return std::auto_ptr<servlet::RequestDispatcher>(r);
}


void ServletRequestImpl::addAttributeListener(ServletRequestAttributeListener* l)
{
    _listener_list.push_back(l);
}


void ServletRequestImpl::enableAttributeListener(bool mode)
{
    _notify_listener = mode;
}
