/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <cstdlib>
#include <algorithm>
#include <iostream>
#include <string>
#include <vector>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

#include <sl/java/lang/IllegalArgumentException.h>
#include <servlet/ServletContextAttributeEvent.h>

#include <si/interface/Context.h>
#include <si/interface/Logger.h>
#include <si/core/RequestDispatcherImpl.h>
#include <si/core/ServletContextImpl.h>
using namespace si::core;


//
// Constructro/Dstructor
//

ServletContextImpl::ServletContextImpl(
    interface::Context *context, const string& home, const string& base)
    : _initialized(false), _context(context)
{
    _home = home;
    _doc_base = base;
}


ServletContextImpl::~ServletContextImpl()
{
}


bool ServletContextImpl::operator!()
{
    return _initialized;
}


sl::java::lang::Object ServletContextImpl::getAttribute(const string& s)
{
    return Attribute::getAttribute(s);
}


void ServletContextImpl::setAttribute(const string& s, sl::java::lang::Object o)
{
    bool exist = Attribute::getAttribute(s) ? true : false;

    Attribute::setAttribute(s, o);

    servlet::ServletContextAttributeEvent e(*this, s, o);

    listener_list_t::iterator i = _listener_list.begin();
    for (; i != _listener_list.end(); i++) {
        if (exist)
            (*i)->attributeReplaced(e);
        else
            (*i)->attributeAdded(e);
    }
}


void ServletContextImpl::removeAttribute(const string& s)
{
    sl::java::lang::Object o = Attribute::getAttribute(s);
    Attribute::removeAttribute(s);

    servlet::ServletContextAttributeEvent e(*this, s, o);

    listener_list_t::iterator i = _listener_list.begin();
    for (; i != _listener_list.end(); i++)
        (*i)->attributeRemoved(e);
}


sl::java::util::Enumeration ServletContextImpl::getAttributeNames()
{
    return Attribute::getAttributeNames();
}


string ServletContextImpl::getInitParameter(const string& s)
{
    string_map_t::iterator i = _parameters.find(s);
    return i != _parameters.end() ? i->second : "";
}


sl::java::util::Enumeration ServletContextImpl::getInitParameterNames()
{
    std::vector<sl::java::lang::Object> objs; 
    std::set<std::string>::const_iterator i = _para_names.begin();
    for (; i != _para_names.end(); i++) {
        objs.push_back(sl::java::lang::Object(*i));
    }
    return sl::java::util::Enumeration(objs);
}


servlet::ServletContext& ServletContextImpl::getContext(const string& /* s */)
{
    return *this;
}


set<string> ServletContextImpl::getResourcePaths(const string& s)
{
    set<string> paths;

    if (s.empty() || s[0] != '/')
        return paths;

    try {
        fs::directory_iterator i(_doc_base + s);
        fs::directory_iterator end;
        for (; i != end; i++)  {
            string target = fs::is_directory(*i) ? i->string()+"/":i->string();
            target.erase(0, _doc_base.length());
            paths.insert(target);
        }
    } catch(fs::filesystem_error& e) {
        cerr << e.what() << endl;
    }

    return paths;
}


string ServletContextImpl::getResource(const string& s)
{
    return _doc_base + s;
}


string ServletContextImpl::getRealPath(const string& s)
{
//    if (s.empty())
//        return "";

    string path = s[0] != '/' ? "/" + s : s;

    if (_doc_base[0] != '/')
        return _home + "/" + _doc_base + path;
    else
        return _home + _doc_base + path;
}


string ServletContextImpl::getMimeType(const string& s)
{
    string extension;
    string::size_type pos = s.rfind(".");

    if (pos != string::npos)
        extension = s.substr(pos + 1);

    string_map_t::iterator i = _mime_mapping.find(extension);
    if (i != _mime_mapping.end())
        return i->second;

    return "";
}


int    ServletContextImpl::getMajorVersion()
{
    return _major_version;
}


int ServletContextImpl::getMinorVersion()
{
    return _minor_version;
}


std::auto_ptr<servlet::RequestDispatcher>
ServletContextImpl::getNamedDispatcher(const std::string& name)
{
    RequestDispatcherImpl *r = new RequestDispatcherImpl(_context);

    if (name.empty()) {
        log("ServletContext::getNamedDispatcher failed. Invalid servlet name: " + name);
        return std::auto_ptr<servlet::RequestDispatcher>(r);
    }

    r->init(name, RequestDispatcherImpl::named_dispatch);
    return std::auto_ptr<servlet::RequestDispatcher>(r);
}


std::auto_ptr<servlet::RequestDispatcher>
ServletContextImpl::getRequestDispatcher(const std::string& uri)
{
    RequestDispatcherImpl *r = new RequestDispatcherImpl(_context);

    if (uri.empty() || uri[0] != '/') {
        log("ServletContext::getRequestDispatcher failed. Invalid URI: " + uri);
        return std::auto_ptr<servlet::RequestDispatcher>(r);
    }
    boost::filesystem::path p(uri);
    std::string normalize_uri = p.normalize().string();

    /* Ƥ ".." ¸ߤ */
    if (normalize_uri.find("..") != std::string::npos) {
        log("ServletContext::getRequestDispatcher failed. relative path is included in URI: " + uri);
        return std::auto_ptr<servlet::RequestDispatcher>(r);
    }

    r->init(normalize_uri, RequestDispatcherImpl::request_dispatch);
    return std::auto_ptr<servlet::RequestDispatcher>(r);
}


void ServletContextImpl::log(const string& s)
{
    string msg = "[" + _doc_base + "] ";
    _context->getLogger()->log(msg + s);
}


void ServletContextImpl::log(const string& s, const sl::java::lang::Exception& e)
{
    string msg = "[" + _doc_base + "] ";
    _context->getLogger()->log(msg + s, e);
}


ifstream& ServletContextImpl::getResourceAsStream(const string& s)
{
    return *_stream;
}


string ServletContextImpl::getServerInfo()
{
    return _server_info;
}

string ServletContextImpl::getServletContextName()
{
    return !_context->name().empty() ? _context->name() : _context->path();
}


void ServletContextImpl::setServerInfo(const string& info)
{
    _server_info = info;
}


void ServletContextImpl::addInitParameter(const string& s, const string& v)
{
    if (_parameters.find(s) == _parameters.end())
        _para_names.insert(s);

    _parameters.insert(string_map_t::value_type(s, v));
}


void ServletContextImpl::addMimeType(const string& name, const string& value)
{
    _mime_mapping.insert(string_map_t::value_type(name, value));
}

void ServletContextImpl::setMajorVersion(int i)
{
    _major_version = i;
}

void ServletContextImpl::setMinorVersion(int i)
{
    _minor_version = i;
}

si::interface::Context* ServletContextImpl::getContainerContext()
{
    return _context;
}

void
ServletContextImpl::addAttributeListener(servlet::ServletContextAttributeListener* l)
{
    _listener_list.push_back(l);
}
