/*-
 * Copyright (c) 2008 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <csignal>
#include <iostream>
#include <string>
#include <boost/shared_ptr.hpp>

#include <si/interface/Server.h>
#include <si/interface/Loader.h>

#include <si/core/BasicLoader.h>
#include <si/core/LibraryLoader.h>
#include <si/core/ServerInvoker.h>
#include <si/core/ServerConfig.h>
using namespace si::core;

namespace {
    static boost::shared_ptr<si::interface::Server> server;
    static boost::shared_ptr<si::interface::Loader> basic_loader;
    static boost::shared_ptr<si::core::LibraryLoader> library_loader;

    const std::string default_server_class = "modules.core.DefaultServer";
}


int ServerInvoker::run(const std::string& home)
{
    sigset_t set;
    sigemptyset(&set);
    sigfillset(&set);
#if 0
    sigaddset(&set, SIGABRT);
    sigaddset(&set, SIGSEGV);
    sigaddset(&set, SIGBUS);
    sigaddset(&set, SIGINT);
    sigaddset(&set, SIGHUP);
    sigaddset(&set, SIGPIPE);
#endif
    sigdelset(&set, SIGCHLD);
    sigprocmask(SIG_BLOCK, &set, NULL);

    library_loader = boost::shared_ptr<si::core::LibraryLoader>(new LibraryLoader(home + "/lib"));

    // ǥեȤΥκ.
    basic_loader = boost::shared_ptr<si::interface::Loader>(new BasicLoader());

    // ФιФ.
    si::core::ServerConfig c;

    try {
        c.parseServerXML(home);
    } catch (std::exception& e) {
        std::cerr << "\"server.xml\" failed in readinging." << std::endl;
        std::cerr << "The reason is \'" << e.what() << "\'" << std::endl;
        return -1;
    }

    std::string class_name = c.attr("className");
    if (class_name.empty())
        class_name = default_server_class;

    try {
        sl::java::lang::Class clazz = basic_loader->loadClass(class_name);
        server = clazz.newInstance<si::interface::Server>();
    } catch (std::exception& e) {
        std::cerr << "\"Server\" failed in loading. :" << e.what() << std::endl;
        return -1;
    }

    server->home(home);
    server->config(c);
    server->setLoader(basic_loader);
    server->className(class_name);

    // ֥åȥƥʤ¹Ԥޤ.
    try {
        server->init();
    } catch (std::exception& e) {
        std::cerr << "\"SHIBAINU\" failed in initialization." << std::endl;
        std::cerr << "The reason is \'" << e.what() << "\'" << std::endl;
        return -1;
    } catch (...) {
        std::cerr << "An unknown exception was thrown." << std::endl;
        return -1;
    }
    server->service(NULL);

    std::cout << "Process is started." << std::endl;

    bool _loop = true;
    while (_loop) {
        int sig;
        sigwait(&set, &sig);
        std::cerr << "si::core::ServerInvoker received signal:" << sig
                  << std::endl;

        switch (sig) {
        case SIGTERM :
        case SIGABRT :
        case SIGSEGV :
        case SIGINT :
        case SIGHUP :
            _loop = false;
        }
    }

    return stop(home);
}


int ServerInvoker::stop(const std::string& home)
{
    std::cout << "Starting Shutdown." << std::endl;

    if (server) {
        std::string class_name = server->className();
        try {
            server->destroy();
        } catch (...) {
            std::cerr << "An unknown exception was thrown." << std::endl;
            return -1;
        }
        server.reset();

        try {
            basic_loader->unloadClass(home + "/" + class_name);
        } catch (sl::java::lang::ClassNotFoundException& e) {
            std::cerr << e.what() << std::endl;
            return -1;
        }
    }

    basic_loader.reset();
    library_loader.reset();

    std::cout << "Success." << std::endl;
    return 0;
}
