/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <string>
using namespace std;

#include <sl/xml/xml_info.h>
#include <sl/xml/xml_parser.h>
using namespace sl::xml;

#include <si/core/ServerConfig.h>
#include <si/core/ContextConfig.h>
using namespace si::core;
using namespace si::interface;


const string ServerConfig::SERVER_XML    = "/conf/server.xml";


//
// Constructor/Destructor
//
ServerConfig::ServerConfig()
{ }

ServerConfig::ServerConfig(const string& home)
{
    attr("Home", home);
    parseServerXML(home);
}


//
// Member functions
//

void ServerConfig::parseServerXML(const string& home)
{
    //
    // ${Home}/conf/server.xml ɤ߹ǡ°Ȥ¸ޤ.
    //
    xml_tag server_tag;

    try {
        xml_parser xml(home + SERVER_XML);
        server_tag = xml.get();

        // Server ɤ
        attr("home", home);
        map<string, string> attr_map = server_tag.attributes();
        map<string, string>::iterator i = attr_map.begin();
        for (; i != attr_map.end(); i++)
            attr(i->first, i->second);

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        throw;
    }

    // Server ˥ͥȤ륿ɤ߹.
    insert(parseServiceTag(server_tag));
    insert(parseLoggerTag(server_tag));
}

Config ServerConfig::parseAttribute(xml_tag& tag, const string& name)
{
    interface::Config c;

    map<string, string> attr_map = tag.attributes();
    map<string, string>::iterator i = attr_map.begin();
    for (; i != attr_map.end(); i++)
        c.attr(i->first, i->second);
    return c;
}

Config ServerConfig::parseServiceTag(xml_tag& tag)
{
    interface::Config service;

    // Service ɤ߹.
    xml_tag t;
    try {
        tag_list_t tags = tag.get("Service");
        if (tags.size() == 0)
            return interface::Config();

        t = tags.at(0);
        service = parseAttribute(t, "Service");
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    service.name("Service");

    //
    // Service ˥ͥȤ륿ɤ߹.
    // Ǥ Connector ʣEngineLogger Ȥʤ.
    vector<interface::Config> clist = parseConnectorTag(t);
    vector<interface::Config>::iterator i = clist.begin();
    for (; i != clist.end(); i++)
        service.insert(*i);

    service.insert(parseEngineTag(t));
    service.insert(parseLoggerTag(t));

    return service;
}

vector<Config> ServerConfig::parseConnectorTag(xml_tag& tag)
{
    vector<Config> connectors;

    // Connector ɤ߹.
    try {
        tag_list_t tags = tag.get("Connector");
        tag_list_t::iterator i = tags.begin();
        for (; i != tags.end(); i++) {
            interface::Config c = parseAttribute(*i, "Connector");
            c.name("Connector");
            connectors.push_back(c);
        }
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    return connectors;
}

Config ServerConfig::parseLoggerTag(xml_tag& tag)
{
    interface::Config logger;

    // Logger ɤ߹.
    try {
        tag_list_t tags = tag.get("Logger");
        if (tags.size() == 0)
            return interface::Config();

        logger = parseAttribute(tags.at(0), "Logger");

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    logger.name("Logger");
    return logger;
}

Config ServerConfig::parseEngineTag(xml_tag& tag)
{
    interface::Config engine;

    // Engine ɤ߹.
    xml_tag t;
    try {
        tag_list_t tags = tag.get("Engine");
        if (tags.size() == 0)
            return interface::Config();

        t = tags.at(0);
        engine = parseAttribute(t, "Engine");
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    engine.name("Engine");
    engine.insert(parseLoggerTag(t));

    vector<interface::Config> clist = parseHostTag(t);
    vector<interface::Config>::iterator i = clist.begin();
    for (; i != clist.end(); i++)
        engine.insert(*i);

    clist = parseValveTag(t);
    for (i = clist.begin(); i != clist.end(); i++)
        engine.insert(*i);

    return engine;
}

vector<Config> ServerConfig::parseHostTag(xml_tag& tag)
{
    vector<Config> clist;

    // Host ɤ߹.
    try {
        tag_list_t tags = tag.get("Host");
        tag_list_t::iterator i = tags.begin();
        for (; i != tags.end(); i++) {
            interface::Config c = parseAttribute(*i, "Host");
            c.name("Host");
            c.insert(parseLoggerTag(*i));
            c.insert(parseRealmTag(*i));

            vector<interface::Config> cl = parseContextTag(*i);
            vector<interface::Config>::iterator ci = cl.begin();
            for (; ci != cl.end(); ci++)
                c.insert(*ci);

            cl = parseValveTag(*i);
            for (ci = cl.begin(); ci != cl.end(); ci++)
                c.insert(*ci);

            clist.push_back(c);
        }

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    return clist;
}

vector<Config> ServerConfig::parseContextTag(xml_tag& tag)
{
    vector<Config> clist;

    // Context ɤ߹.
    try {
        tag_list_t tags = tag.get("Context");
        tag_list_t::iterator i = tags.begin();
        for (; i != tags.end(); i++) {
            ContextConfig c;
            c.parseContextXML(*i);

            vector<interface::Config> cl = parseValveTag(*i);
            vector<interface::Config>::iterator ci = cl.begin();
            for (; ci != cl.end(); ci++)
                c.insert(*ci);

            clist.push_back(c);
        }
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    return clist;
}


Config ServerConfig::parseRealmTag(xml_tag& tag)
{
    interface::Config realm;

    // Realm ɤ߹.
    try {
        tag_list_t tags = tag.get("Realm");
        if (tags.size() == 0)
            return interface::Config();

        realm = parseAttribute(tags.at(0), "Realm");

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    realm.name("Realm");
    return realm;
}

vector<Config> ServerConfig::parseValveTag(xml_tag& tag)
{
    vector<Config> clist;

    // Valve ɤ߹.
    try {
        tag_list_t tags = tag.get("Valve");
        tag_list_t::iterator i = tags.begin();
        for (; i != tags.end(); i++) {
            interface::Config c = parseAttribute(*i, "Valve");
            c.name("Valve");

            clist.push_back(c);
        }

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ServerConfig::parseServerConfig - "
             << "It failed in reading \"" << SERVER_XML << "\" file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    return clist;
}
