/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifdef DEBUG
#include <iostream>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <sl/inet/http_util.hpp>
#include <si/interface/Context.h>
#include <si/interface/Logger.h>
#include <si/core/HttpServletRequestImpl.h>
#include <si/core/HttpServletResponseImpl.h>
#include <si/core/RequestDispatcherImpl.h>
using namespace si::core;


//
// Constructor/Destructor
//

RequestDispatcherImpl::RequestDispatcherImpl(si::interface::Context *context)
    : _context(context), _available(false)
{ }


RequestDispatcherImpl::~RequestDispatcherImpl()
{ }


bool RequestDispatcherImpl::operator!() const
{
    return !_available;
}


void RequestDispatcherImpl::init(const std::string& uri, dispatch_mode mode)
{
    _context->getLogger()->log("RequestDispatcher initialize URI :" + uri,
                               si::interface::Logger::DEBUG_LOG);

    switch (mode) {
    case request_dispatch :
        initRequestDispatch(uri);
        break;
    case named_dispatch :
        initNamedDispatch(uri);
        break;
    default :
        throw std::runtime_error("RequestDispatcherImpl::init failed. Invalid argument.");
    }
}


/**
 * RequestDispach ν.
 *
 *  uri ϤƳ Servlet θԤ
 * forward/include μ¹ԤΤνԤޤ.
 *
 * @param 줿ƥȥ롼ȤΥѥ.
 */
void RequestDispatcherImpl::initRequestDispatch(const std::string& uri)
{
    /* uri ƤϤʤΤǤǤƬ "/" ʳ */
    if (uri.empty() || uri[0] != '/')
        throw servlet::ServletException("RequestDispatcher failed. Invalid argument.");

    std::string::size_type pos;
    if ((pos = uri.find("?")) != std::string::npos) {
        _uri = uri.substr(0, pos);
        _query_string = uri.substr(pos + 1, std::string::npos);
    } else {
        _uri = uri;
    }

    _wrapper = _context->getWrapper(_uri).get();
    if (!_wrapper || !_wrapper->instance()) {
        std::string msg("The dispatched resource is illegal.");
        msg += " Target resource is \"" + _uri + "\".";
        throw servlet::ServletException(msg);
    }

    /* DispatcherƤӽФΥѥ󤫤 ServletPath  PathInfo Ф */
    getServletPathAndPathInfo(_uri, _wrapper, _servlet_path, _path_info);

    _context->getLogger()->log("RequestDispatcher URI        :" + _uri,
                               si::interface::Logger::DEBUG_LOG);
    _context->getLogger()->log("RequestDispatcher ServletPath:" + _servlet_path,
                               si::interface::Logger::DEBUG_LOG);
    _context->getLogger()->log("RequestDispatcher PathInfo   :" + _path_info,
                               si::interface::Logger::DEBUG_LOG);

    /* query_string ѥ᡼ξФ */
    sl::http::parameter_map _map = sl::http::parameter_split(_query_string);

    _available = true;
    _mode = request_dispatch;
}


/**
 * NamedDispatch ν.
 *
 *  name ˳ Servlet θԤ
 * NamedDispatch Ѥ forward/include μ¹ԤνԤޤ.
 *
 * @param name servlet-name ˳륵֥å̾.
 */
void RequestDispatcherImpl::initNamedDispatch(const std::string& name)
{
    /* ֥å̾ξ */
    if (name.empty())
        throw servlet::ServletException("RequestDispatcher failed. Invalid argument.");

    _wrapper = _context->getNamedWrapper(name).get();
    if (_wrapper && _wrapper->instance())
        _available = true;

    _mode = named_dispatch;
}


/**
 * ServletPathڤPathInfoμ.
 * ƥȥ롼ȤΥѥ ServletPath  PathInfo ʬ䤷
 *  servlet_path ڤ path_info ¸ޤ.
 *
 * @param path_string  ƥȥѥʹߤʸ.
 * @param w            path_string ˥ޥå ServletWrapper 󥹥.
 * @param servlet_path Ϸ̤ ServletPath ¸.
 * @param path_info    Ϸ̤ PathInfo ¸.
 */
void RequestDispatcherImpl::getServletPathAndPathInfo(const std::string& path_string,
                                                      si::interface::ServletWrapper* w,
                                                      std::string& servlet_path,
                                                      std::string& path_info)
{
    /*
     * path_string  path info Ф.
     * web.xml ˤ Servlet Υޥåԥ󥰤ʣ꤬ǽʤΤǡ
     * νǼ Wrapper ޥåԥ󥰾ơ
     * Υޥåԥ󥰾󤫤ޥåѥȽꤹ.
     * ޥåԥʸ󤫤ޥåѥ path-info ˤʤϤ.
     */
    std::vector<std::string> mapping = w->getMapping();
    std::vector<std::string>::iterator i = mapping.begin();
    for (; i != mapping.end(); i++) {
        std::string map_str = *i;

        // ̤Ǥϥޥåʤ
        if (map_str.empty())
            continue;

        if (map_str[0] == '*') {
            servlet_path = "/";
            path_info = path_string.substr(1);
            break;
        }

        // "*"  "/*" ξϺʸ󤫤 path_info 
        if (map_str[map_str.length() - 1] == '*') {
            map_str.erase(map_str.length() - 1);
            if (map_str[map_str.length() - 1] == '/')
                map_str.erase(map_str.length() - 1);
        }

        if (!path_string.compare(0, map_str.length(), map_str)) {
            path_info = path_string.substr(map_str.length());
            servlet_path = path_string.substr(0, path_string.length()-path_info.length());
            break;
        }
    }

    /*
     * ꥯȤ'/'ǽüƤ
     * welcome-file-list Ϣ뤷ե̾ơ
     * Ԥʤ¸ߤե򥿡åȥեȤƻѤ
     */
    if (path_string[path_string.length() - 1] == '/') {
        std::string file_path = _context->home() + "/" +
                                _context->docBase() + servlet_path + path_info;

        std::list<std::string> file_list = _context->welcomeFileList();
        std::list<std::string>::iterator i = file_list.begin();
        for (; i != file_list.end(); i++) {
            std::string tmp = file_path + *i;
            struct stat sb;
            if (::stat(tmp.c_str(), &sb) != -1 && S_ISREG(sb.st_mode)) {
                path_info += *i;
                break;
            }
        }
    }
}



void RequestDispatcherImpl::forward(servlet::ServletRequest& req,
                                    servlet::ServletResponse& res)
    throw (servlet::ServletException)
{
    if (!_available) {
        throw servlet::ServletException("RequestDispatcher::forward failed. "
                                        "forwarding ahead is unavailable");
    }
    if (res.isCommitted()) {
        throw servlet::ServletException("RequestDispacher::forward failed. "
                                        "Response is already committed");
    }

    /* ХåեΥꥢ */
    res.reset();

    try {
        HttpServletRequestImpl& hreq  = dynamic_cast<HttpServletRequestImpl&>(req);
        HttpServletResponseImpl& resi = dynamic_cast<HttpServletResponseImpl&>(res);

        hreq.enableAttributeListener(false); /* ꥹʡؤΤ̵ˤ */

        if (_mode == request_dispatch) {
            hreq.setAttribute("javax.servlet.forward.request_uri", hreq.getRequestURI());
            hreq.setAttribute("javax.servlet.forward.context_path",hreq.getContextPath());
            hreq.setAttribute("javax.servlet.forward.servlet_path",hreq.getServletPath());
            hreq.setAttribute("javax.servlet.forward.path_info",   hreq.getPathInfo());
            hreq.setAttribute("javax.servlet.forward.query_string",hreq.getQueryString());

            /* Servlet API ǼͤѹƤ */
            hreq.setServletPath(_servlet_path);
            hreq.setPathInfo(_path_info);
            hreq.setQueryString(_query_string);
            hreq.setRequestURI(hreq.getContextPath() +
                               hreq.getServletPath() + hreq.getPathInfo());
            sl::http::parameter_map::const_iterator i = _parameter.begin();
            for (; i != _parameter.end(); i++)
                hreq.setAttribute(i->first, i->second);
        }

        hreq.enableAttributeListener(true); /* ꥹʡؤΤͭˤ */

        dispatch(req, res);

        hreq.enableAttributeListener(false);/* ꥹʡؤΤ̵ˤ */

        if (_mode == request_dispatch) {
            hreq.removeAttribute("javax.servlet.forward.request_uri");
            hreq.removeAttribute("javax.servlet.forward.context_path");
            hreq.removeAttribute("javax.servlet.forward.servlet_path");
            hreq.removeAttribute("javax.servlet.forward.path_info");
            hreq.removeAttribute("javax.servlet.forward.query_string");
        }

        hreq.enableAttributeListener(true); /* ꥹʡؤΤͭˤ */

    } catch (std::bad_cast& e) {
        throw servlet::ServletException("RequestDispatcherImpl::forward failed. "
                                        "Illegal request or response object.");
    }
}


void RequestDispatcherImpl::include(servlet::ServletRequest& req,
                                    servlet::ServletResponse& res)
    throw (servlet::ServletException)
{
    if (!_available) {
        throw servlet::ServletException("RequestDispatcher::include failed. Illegal state.");
    }

    try {
        /* RequestDispatch ξ°ɲäNamedDispatch ξɲäʤ */
        HttpServletRequestImpl& hreq  = dynamic_cast<HttpServletRequestImpl&>(req);
        HttpServletResponseImpl& resi = dynamic_cast<HttpServletResponseImpl&>(res);

/* XXXXX ꥯȡ쥹ݥ󥹤ξѹػߤɬפ. */

        hreq.enableAttributeListener(false); /* ꥹʡؤΤ̵ˤ */

        if (_mode == request_dispatch) {
            hreq.setAttribute("javax.servlet.include.request_uri",
                              _context->path() + _servlet_path + _path_info);
            hreq.setAttribute("javax.servlet.include.context_path", _context->path());
            hreq.setAttribute("javax.servlet.include.servlet_path", _servlet_path);
            hreq.setAttribute("javax.servlet.include.path_info",    _path_info);
            hreq.setAttribute("javax.servlet.include.query_string", _query_string);

            sl::http::parameter_map::const_iterator i = _parameter.begin();
            for (; i != _parameter.end(); i++)
                hreq.setAttribute(i->first, i->second);
        }

        hreq.enableAttributeListener(true); /* ꥹʡؤΤͭˤ */

        dispatch(req, res);

        hreq.enableAttributeListener(false);/* ꥹʡؤΤ̵ˤ */

        if (_mode == request_dispatch) {
            hreq.removeAttribute("javax.servlet.include.request_uri");
            hreq.removeAttribute("javax.servlet.include.context_path");
            hreq.removeAttribute("javax.servlet.include.servlet_path");
            hreq.removeAttribute("javax.servlet.include.path_info");
            hreq.removeAttribute("javax.servlet.include.query_string");

        }
        hreq.enableAttributeListener(true); /* ꥹʡؤΤͭˤ */

    } catch (std::bad_cast& e) {
        throw servlet::ServletException("RequestDispatcherImpl::include failed. "
                                        "Illegal request or response object.");
    }
}


void RequestDispatcherImpl::dispatch(servlet::ServletRequest& req,
                                     servlet::ServletResponse& res)
    throw (servlet::ServletException)
{
     if (!_wrapper || !_wrapper->instance())
        throw servlet::ServletException("The dispatched resource is illegal.");

#ifdef DEBUG
    std::cerr << "RequestDispatcherImpl::dispatch" << std::endl;
    std::cerr << "Dispatch -> :" << _wrapper->name() << std::endl;
#endif

    /*
     * ǥѥåμ¹
     *
     * ¿ʬȯ㳰 ServletException ʤΤ
     * ªʤǤΤޤޥ롼Ƥ.
     */
    _wrapper->instance()->service(req, res);
}
