/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <string>
using namespace std;

#include <sl/xml/xml_info.h>
#include <sl/xml/xml_parser.h>
using namespace sl::xml;

#include <si/core/ContextConfig.h>
using namespace si::core;
using namespace si::interface;


//
// Constructor/Destructor
//
ContextConfig::ContextConfig()
{ }

ContextConfig::ContextConfig(const string& file)
{
#ifdef DEBUG
    cerr << "ContextConfig::ContextConfig - read " << file << " start."
              << endl;
#endif
    parseContextXML(file);
}


//
// Member functions
//

void ContextConfig::parseContextXML(const string& file)
{
    //
    // ${Home}/${Name}/${Host/${application}.xml ɤ߹ǡ
    // °Ȥ¸ޤ.
    //
    xml_tag context_tag;

    try {
        xml_parser xml(file);
        context_tag = xml.get();
        parseContextXML(context_tag);

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ContextConfig::parseContextConfig - "
             << "It failed in reading \"" << file << "\" file." << endl;
        throw;
    }
}

void ContextConfig::parseContextXML(sl::xml::xml_tag& tag)
{
    //
    // ${Home}/${Name}/${Host/${application}.xml ɤ߹ǡ
    // °Ȥ¸ޤ.
    //
    try {
        map<string, string> attr_map = tag.attributes();
        map<string, string>::iterator i = attr_map.begin();
        for (; i != attr_map.end(); i++)
            attr(i->first, i->second);

    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ContextConfig::parseContextConfig - "
             << "It failed in reading file." << endl;
        throw;
    }

    name("Context");
    insert(parseElement(tag, "Manager"));
    insert(parseElement(tag, "Loader"));
    insert(parseElement(tag, "Logger"));
    insert(parseElement(tag, "Realm"));

    vector<interface::Config> clist = parseRepetitionElement(tag, "Valve");
    vector<interface::Config>::iterator i = clist.begin();
    for (; i != clist.end(); i++)
        insert(*i);
}

Config ContextConfig::parseAttribute(xml_tag& tag, const string& name)
{
    interface::Config c;

    map<string, string> attr_map = tag.attributes();
    map<string, string>::iterator i = attr_map.begin();
    for (; i != attr_map.end(); i++)
        c.attr(i->first, i->second);
    return c;
}

Config ContextConfig::parseElement(xml_tag& tag, const std::string& name)
{
    interface::Config c;

    try {
        tag_list_t tags = tag.get(name);
        if (tags.size() == 0)
            return interface::Config();

        c = parseAttribute(tags.at(0), name);
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ContextConfig::parseContextConfig - "
             << "It failed in reading file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    c.name(name);
    return c;
}

vector<Config> ContextConfig::parseRepetitionElement(xml_tag& tag, const std::string& name)
{
    vector<Config> clist;

    // Connector ɤ߹.
    try {
        tag_list_t tags = tag.get(name);
        tag_list_t::iterator i = tags.begin();
        for (; i != tags.end(); i++) {
            interface::Config c = parseAttribute(*i, name);
            c.name(name);
            clist.push_back(c);
        }
    } catch (sl::xml::xml_parser_error& e) {
        cerr << "ContextConfig::parseContextConfig - "
             << "It failed in reading file." << endl;
        cerr << "Ther reason is \'" << e.what() << "\'" << endl;
    }

    return clist;
}

