/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <string>
#include <boost/algorithm/string/trim.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>

#include <sl/xml/xml_info.h>
#include <sl/xml/xml_parser.h>
#include <sl/xml/xml_parser_error.h>
#include <sl/xml/xml_tag.h>

#include <si/webxml/ApplicationConfig.h>
using namespace si::webxml;

//
// Constructor/Destructor
//

ApplicationConfig::ApplicationConfig(const std::string& root,
                                     const std::string& xml_path)
{
    using namespace sl::xml;

    _root = root;

    try {
        xml_parser xml(xml_path);
        xml_tag webapp_tag = xml.get();

        parse_context_param(webapp_tag.get("context-param"));

        parse_filter(webapp_tag.get("filter"));

        parse_filter_mapping(webapp_tag.get("filter-mapping"));

        parse_listener(webapp_tag.get("listener"));

        parse_servlet(webapp_tag.get("servlet"));

        parse_servlet_mapping(webapp_tag.get("servlet-mapping"));

        parse_mime_mapping(webapp_tag.get("mime-mapping"));

        parse_welcome_file_list(webapp_tag.get("welcome-file-list"));

        parse_error_page(webapp_tag.get("error-page"));

        parse_jsp_config(webapp_tag.get("jsp-config.taglib"));

        parse_session_config(webapp_tag.get("session-config"));

        parse_security_constraint(webapp_tag.get("security-constraint"));

        parse_login_config(webapp_tag.get("login-config"));

    } catch(xml_parser_error& e) {
        std::cerr << e.what() << std::endl;
    }
}


//
// Member functions.
//
void ApplicationConfig::merge(const ApplicationConfig& r)
{
    /* context-param */
    _context_param += r.getContextParam();

    /* error-page */
    std::vector<int> c = r.getErrorPage().error_code();
    std::vector<int>::iterator i = c.begin();
    for (; i != c.end(); i++)
        _error_page.add(*i, r.getErrorPage().find(*i));

    std::vector<std::string> e = r.getErrorPage().error_exception();
    std::vector<std::string>::iterator j = e.begin();
    for (; j != e.end(); j++)
        _error_page.add(*j, r.getErrorPage().find(*j));

    /* filter */
    std::vector<si::webxml::filter> v = r.getFilters();
    std::copy(v.begin(), v.end(), std::back_inserter(_filters));

    /* filter-mapping */
    _filter_mapping += r.getFilterMapping();

    /* jsp-config */
    _jsp_config += r.getJspConfig();

    /* listener */
    _listener += r.getListener();

    /* login-config */
    _login_config = r.getLoginConfig();

    /* mime-mapping */
    _mime_mapping += r.getMimeMapping();

    /* servlet */
    std::vector<si::webxml::servlet> v2 = r.getServlet();
    std::copy(v2.begin(), v2.end(), std::back_inserter(_servlets));

    /* servlet-mapping */
    _servlet_mapping += r.getServletMapping();

    /* session-config */
    _session_config = r.getSessionConfig();

    /* security-constraint */
    std::vector<si::webxml::security_constraint> v3 = r.getSecurityConstraint();
    std::copy(v3.begin(), v3.end(), std::back_inserter(_security_constraint));

    _welcome_file_list += r.getWelcomeFileList();
}


si::webxml::context_param ApplicationConfig::getContextParam() const
{
    return _context_param;
}


std::vector<si::webxml::filter> ApplicationConfig::getFilters() const
{
    return _filters;
}


si::webxml::filter_mapping ApplicationConfig::getFilterMapping() const
{
    return _filter_mapping;
}


si::webxml::listener ApplicationConfig::getListener() const
{
    return _listener;
}


std::vector<si::webxml::servlet> ApplicationConfig::getServlet() const
{
    return _servlets;
}


si::webxml::servlet_mapping ApplicationConfig::getServletMapping() const
{
    return _servlet_mapping;
}


si::webxml::mime_mapping ApplicationConfig::getMimeMapping() const
{
    return _mime_mapping;
}


si::webxml::welcome_file_list ApplicationConfig::getWelcomeFileList() const
{
    return _welcome_file_list;
}


si::webxml::error_page ApplicationConfig::getErrorPage() const
{
    return _error_page;
}

std::vector<si::webxml::security_constraint>
ApplicationConfig::getSecurityConstraint() const
{
    return _security_constraint;
}


si::webxml::login_config ApplicationConfig::getLoginConfig() const
{
    return _login_config;
}


si::webxml::session_config ApplicationConfig::getSessionConfig() const
{
    return _session_config;
}


si::webxml::jsp_config ApplicationConfig::getJspConfig() const
{
    return _jsp_config;
}


void ApplicationConfig::parse_context_param(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string n = i->child("param-name").value();
            std::string v = i->child("param-value").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));

            // param-name  param-value ФˤʤäƤϤ
            if (n.empty() || v.empty())
                continue;

            _context_param.add(n, v);

        } catch(xml_parser_error& e) { }
    }
}


void ApplicationConfig::parse_filter(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            si::webxml::filter filter;

            /* filter-name  filter-class ФˤʤäƤϤ */
            std::string n = i->child("filter-name").value();
            std::string v = i->child("filter-class").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));
            if (n.empty() || v.empty())
                continue;

            filter.filter_name(n);
            filter.filter_class(v);
            filter.filter_path(_root);

            tag_list_t para_tags = i->get("init-param");
            tag_list_t::const_iterator j = para_tags.begin();
            for (; j != para_tags.end(); j++) {
                try {
                    std::string n = j->child("param-name").value();
                    std::string v = j->child("param-value").value();

                    if (!n.empty() || !v.empty())
                        filter.add_init_param(n, v);
                } catch(xml_parser_error& e) {
                    continue;
                }
            }
    
            _filters.push_back(filter);

        } catch (xml_parser_error& e) { }
    }
}


void ApplicationConfig::parse_filter_mapping(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            std::string n = i->child("filter-name").value();
            std::string v;
            boost::trim_if(n, boost::is_any_of("\r\n\t "));

            try {
                v = i->child("url-pattern").value();
                boost::trim_if(v, boost::is_any_of("\r\n\t "));
            } catch (xml_parser_error& e) { continue; }
    
            if (v.empty()) {
                try {
                    v = i->child("servlet-name").value();
                } catch(xml_parser_error& e) { continue; }
            }
            if (n.empty() || v.empty())
                continue;
            _filter_mapping.add(n, v);

        } catch(xml_parser_error& e) { }
    }
}

void ApplicationConfig::parse_listener(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            std::string c = i->child("listener-class").value();
            boost::trim_if(c, boost::is_any_of("\r\n\t "));
            if (c.empty())
                continue;
            _listener.add(c);

        } catch(xml_parser_error& e) { }
    }
}

void ApplicationConfig::parse_servlet(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            std::string n = i->child("servlet-name").value();
            std::string v = i->child("servlet-class").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));

            if (n.empty() || v.empty())
                continue;

            si::webxml::servlet s;
            s.servlet_name(n);
            s.servlet_class(v);
            s.servlet_path(_root);

            tag_list_t para_tags = i->get("init-param");
            tag_list_t::const_iterator j = para_tags.begin();
            for (; j != para_tags.end(); j++) {
                try {
                    std::string pn = j->child("param-name").value();
                    std::string pv = j->child("param-value").value();

                    if (!pn.empty() || !pv.empty())
                        s.add_init_param(pn, pv);

                } catch(xml_parser_error& e) {
                    continue;
                }
            }

            _servlets.push_back(s);

        } catch(xml_parser_error& e) { }
    }
}


void ApplicationConfig::parse_servlet_mapping(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            std::string n = i->child("servlet-name").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            std::string v;

            try {
                v = i->child("url-pattern").value();
                boost::trim_if(v, boost::is_any_of("\r\n\t "));
            } catch(xml_parser_error& e) { }
    
            if (v.empty()) {
                try {
                    v = i->child("jsp-file").value();
                    boost::trim_if(v, boost::is_any_of("\r\n\t "));
                } catch(xml_parser_error& e) { continue; }
            }
    
            if (n.empty() || v.empty())
                continue;
            _servlet_mapping.add(n, v);

        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
            continue;
        }
    }
}


void ApplicationConfig::parse_mime_mapping(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {

        try {
            std::string n = i->child("extension").value();
            std::string v = i->child("mime-type").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));
            if (n.empty() || v.empty())
                continue;
            _mime_mapping.add(n, v);
        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
}


void ApplicationConfig::parse_error_page(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string n = i->child("error-code").value();
            std::string v = i->child("location").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));
            if (n.empty() || v.empty())
                continue;
            _error_page.add(boost::lexical_cast<int>(n), v);
            continue;
        } catch (xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }

        try {
            std::string n = i->child("exception-type").value();
            std::string v = i->child("location").value();
            boost::trim_if(n, boost::is_any_of("\r\n\t "));
            boost::trim_if(v, boost::is_any_of("\r\n\t "));
            if (n.empty() || v.empty())
                continue;
            _error_page.add(n, v);
            continue;
        } catch (xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
}

void ApplicationConfig::parse_welcome_file_list(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    if (tags.size() == 0)
        return;

    tag_list_t file_tags = (*tags.begin()).get("welcome-file");

    tag_list_t::const_iterator i = file_tags.begin();
    for (int n = 0; i != file_tags.end(); i++, n++) {
        try {
            std::string file = i->value();
            boost::trim_if(file, boost::is_any_of("\r\n\t "));
            _welcome_file_list.add(file);
        } catch (xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
}
    

void ApplicationConfig::parse_security_constraint(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;
    si::webxml::security_constraint s;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string url_pattern =
                i->child("web-resource-collection").child("url-pattern").value();
            boost::trim_if(url_pattern, boost::is_any_of("\r\n\t "));
            s.url_pattern(url_pattern);

            tag_list_t para_tags = i->get("http-method");
            tag_list_t::const_iterator j = para_tags.begin();
            for (; j != para_tags.end(); j++) {
                try {
                    std::string v = j->value();
                    boost::trim_if(v, boost::is_any_of("\r\n\t "));
                    s.add_http_method(v);
                } catch(xml_parser_error& e) {
                    continue;
                }
            }

        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
        _security_constraint.push_back(s);
    }
}



void ApplicationConfig::parse_jsp_config(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string uri = i->child("taglib-uri").value();
            std::string loc = i->child("taglib-location").value();
            boost::trim_if(uri, boost::is_any_of("\r\n\t "));
            boost::trim_if(loc, boost::is_any_of("\r\n\t "));
            _jsp_config.add(uri, loc);
        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
}

void ApplicationConfig::parse_session_config(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    bool found = false;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string t = i->child("session-timeout").value();
            boost::trim_if(t, boost::is_any_of("\r\n\t "));
            _session_config.session_timeout(boost::lexical_cast<int>(t));
            found = true;
        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
    if (!found)
        _session_config.session_timeout(30);
}


void ApplicationConfig::parse_login_config(const sl::xml::tag_list_t& tags)
{
    using namespace sl::xml;

    for (tag_list_t::const_iterator i = tags.begin(); i != tags.end(); i++) {
        try {
            std::string a = i->child("auth-method").value();
            std::string r = i->child("realm-name").value();
            boost::trim_if(a, boost::is_any_of("\r\n\t "));
            boost::trim_if(r, boost::is_any_of("\r\n\t "));
            _login_config.auth_method(a);
            _login_config.realm_name(r);

        } catch(xml_parser_error& e) {
#ifdef DEBUG
            std::cerr << e.what() << std::endl;
#endif
        }
    }
}
