/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_INTERFACE_RESPONSE_H
#define SHIBAINU_INTERFACE_RESPONSE_H

#include <iostream>
#include <boost/shared_ptr.hpp>

#include <servlet/ServletResponse.h>
#include <servlet/http/Cookie.h>

namespace si {
namespace interface {

class Context;

class Response {
public :
    virtual ~Response() { }

    virtual void protocol(const std::string& protocol)=0;

    virtual void status(int, const std::string& msg)=0;

    virtual void header(const std::string& n, const std::string& v)=0;

    virtual void content(const std::string& content)=0;

    virtual void flush()=0;

    virtual void cleanup()=0;

    virtual std::ostream& copy_to(std::ostream&)=0;

    virtual std::istream& copy_from(std::istream&)=0;

    virtual void setContext(const boost::shared_ptr<Context>& c)=0;

    virtual boost::shared_ptr<Context> getContext() const=0;

    /**
     * The Content-Type of the response sent back to the client is set.
     * @param    s    ContentType.
     */
    virtual void setContentType(const std::string& s)=0;

    /**
     * The `Content-Length` of the response sent back to the client is set.
     * @param    i    value of `Content-Length`.
     */
    virtual void setContentLength(int i)=0;

    /**
     * The character encoding name applied to the MIME body sent back
     * by the response is returned.
     * @return    `CharacterEncoding`
     */
    virtual std::string getCharacterEncoding()=0;

    /**
     * The character encoding name applied to the MIME body sent back
     * by the response is returned.
     * @return    `ContentType`
     */
    virtual std::string getContentType()=0;

    /**
     * 쥹ݥ󥹤Υܥǥ.
     */
    virtual void setContent(const std::string& s)=0;

    /**
     * ꤵ줿Cookie쥹ݥ󥹤ɲäޤ.
     */
    virtual void addCookie(const servlet::http::Cookie& c)=0;

    /**
     *  s ǻꤵ줿ͤĥ쥹ݥ󥹥إåɲäޤ.
     */
    virtual void addHeader(const std::string& s, const std::string& v)=0;

    /**
     *  s ǻꤵ줿ͤĥ쥹ݥ󥹥إåꤷޤ.
     */
    virtual void setHeader(const std::string& s, const std::string& v)=0;

    /**
     *  s ̾ĥإå˥åȤƤ뤫֤ޤ.
     */
    virtual bool containsHeader(const std::string& s)=0;

    /**
     *  s ǻꤵ줿ͤĥ쥹ݥ󥹥إå򤹤٤ƺޤ.
     */
    virtual void eraseHeader(const std::string& s)=0;

    /**
     *  s бإå֤ͤޤ.
     * @param   s   إå̾.
     * @return  إå.
     */
    virtual std::string getHeader(const std::string& s)=0;

    /**
     * Хåե򥯥ꥢꤵ줿ơȤäƥ饤Ȥ
     * 顼쥹ݥ󥹤ޤ.
     */
    virtual void sendError(int sc)=0;

    /**
     * Хåե򥯥ꥢꤵ줿ơȤäƥ饤Ȥ
     * 顼쥹ݥ󥹤ޤ.
     */
    virtual void sendError(int sc, const std::string& s)=0;

    /**
     * ꤵ줿쥯URLѤơ饤Ȥ
     * Ūʥ쥯ȥ쥹ݥ󥹤ޤ.
     */
    virtual void sendRedirect(const std::string& s)=0;

    /**
     * Υ쥹ݥ󥹤Υơɤꤷޤ.
     */
    virtual void setStatus(int c)=0;

    /**
     * Υ쥹ݥ󥹤Ƥξ֤ޤ.
     * Τϥơ饤ƤΥإåեɤǤ.
     */
    virtual void reset()=0;

    virtual void flushHeader()=0;

    virtual void flushBody()=0;
};

inline std::istream& operator>>(std::istream& is, Response& res)
{
    return res.copy_from(is);
}

inline std::ostream& operator<<(std::ostream& os, Response& res)
{
    return res.copy_to(os);
}

} // namespace interface
} // namespace si


#endif // SHIBAINU_INTERFACE_RESPONSE_H
