/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_INTERFACE_REQUEST_H
#define SHIBAINU_INTERFACE_REQUEST_H

#include <iostream>
#include <string>
#include <vector>
#include <map>

#include <servlet/http/Cookie.h>

namespace si {
namespace interface {

class Request {
public :
    virtual ~Request() { }

    virtual std::string method() const=0;

    virtual std::string uri() const=0;

    virtual std::string protocol() const=0;

    virtual std::string header(const std::string& name) const=0;

    virtual void cleanup()=0; 

    virtual std::ostream& copy_to(std::ostream&)=0;

    virtual std::istream& copy_from(std::istream&)=0;

    /**
     * The value of parameter with the same name as argument `name`
     * is returned.
     * @param    name    name of parameter.
     * @return        value of parameter.
     */
    virtual std::string getParameter(const std::string& s)=0;

    /**
     * List of parameter name is returned.
     * @return    list of parameter name.
     */
    virtual std::vector<std::string> getParameterNames()=0;

    /**
     * List of parameter value is returned.
     * @return    list of parameter value.
     */
    virtual std::vector<std::string>
    getParameterValues(const std::string& s)=0;

    /**
     * Map of parameter value is returned.
     * @return    list of parameter value.
     */
    virtual std::multimap<std::string, std::string> getParameterMap()=0;

    /**
     * The encode set to the request is returned. 
     * @return    encode.
     */
    virtual std::string getCharacterEncoding()=0;

    /**
     * `ContentLength` set to the request is returned.
     * @return    size.
     */
    virtual int getContentLength()=0;

    /**
     * `ContentType` set to the request is returned.
     * @return    ContentType.
     */
    virtual std::string getContentType()=0;

    /**
     * The istream object to operate the request as a stream is returned.
     * @return    Instance of ServletInputStream class.
     */
    virtual std::istream& getInputStream()=0;

#if 0
    /**
     * The istream object to operate the request as a stream is returned.
     * @return    Instance of Reader class.
     */
    virtual sl::java::io::BufferedReader getReader()=0;

    /**
     * Locale assumed for the client to be able to display contents
     * is returned. 
     * @return    std::locale.
     */
    virtual sl::java::util::Locale getLocale()=0;

    /**
     * Locale assumed for the client to be able to display contents
     * is returned. 
     * @return    list of locale.
     */
    virtual sl::java::util::Enumeration& getLocales()=0;
#endif
    /**
     * The protocol name and the version of the request are returned. 
     * @return    protocol/version.
     */
    virtual std::string getProtocol()=0;

    /**
     * Schemes such as http and ftp etc. are returned. 
     * @return    scheme.
     */
    virtual std::string getScheme()=0;

    /**
     * The presence of the encryption of the communication route
     * of the SSL communication etc. is returned. 
     * @return    bool.
     */
    virtual bool isSecure()=0;

    /**
     * The server name is acquired. 
     * @return    server name.
     */
    virtual std::string getServerName()=0;

    /**
     * The port number used by the server is acquired. 
     * @return    port number.
     */
    virtual int getServerPort()=0;

    /**
     * @return    IP address.
     */
    virtual std::string getLocalAddr()=0;

    /**
     * @return    host name.
     */
    virtual std::string getLocalName()=0;

    /**
     * @return    port number.
     */
    virtual int getLocalPort()=0;

    /**
     * IP address in the transmission origin of the request is returned. 
     * @return    IP address.
     */
    virtual std::string getRemoteAddr()=0;

    /**
     * The host name of the transmission origin of the request is returned. 
     * @return    host name.
     */
    virtual std::string getRemoteHost()=0;

    /**
     * @return    port number.
     */
    virtual int getRemotePort()=0;

    /**
     * ǧڥ֤̾ޤ.
     * @return    String ̾.
     */
    virtual std::string getAuthType()=0;

    /**
     * åΥꥹȤ.
     * @return    åΥꥹ.
     */
    virtual std::vector<servlet::http::Cookie> getCookies()=0;

    /**
     *  s бإå֤ͤޤ.
     * @param    s    إå̾.
     * @return    إå.
     */
    virtual std::string getHeader(const std::string& s)=0;

    /**
     *  s бإå֤ͤޤ.
     * @param    s    إå̾.
     * @return    إå.
     */
    virtual std::vector<std::string> getHeaders(const std::string& s)=0;

    /**
     * إå̾ΥꥹȤ֤ޤ.
     * @return    إå̾Υꥹ.
     */
    virtual std::vector<std::string> getHeaderNames()=0;

    /**
     * ꥯȤΥ᥽åɤ֤ޤ.
     * @return    ᥽å.
     */
    virtual std::string getMethod()=0;

    /**
     * ꥯȤΥ꡼ޤ.
     * @return    ꡼.
     */
    virtual std::string getQueryString()=0;

    /**
     * ꥯȤURIޤ.
     * @return    URI.
     */
    virtual std::string getRequestURI()=0;

    /**
     * ꥯȤURLޤ.
     * @return    URL.
     */
    virtual std::string getRequestURL()=0;

    /**
     * ꥯΥƥĤ֤ޤ.
     * @return    ƥĥǡ.
     */
    virtual std::string getContent()=0;
};

inline std::istream& operator>>(std::istream& is, Request& req)
{
    return req.copy_from(is);
}

inline std::ostream& operator<<(std::ostream& os, Request& req)
{
    return req.copy_to(os);
}

} // namespace interface
} // namespace si

#endif // SHIBAINU_INTERFACE_REQUEST_H
