/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_UTIL_URL_PATTERN_H
#define SHIBAINU_UTIL_URL_PATTERN_H

#include <string>
#include <boost/regex.hpp>

//#define URL_PATTERN_DEBUG

namespace si {
namespace core {

/**
 * servlet-mapping ˻Ѥ url-pattern 򥷥ץɽѹ.
 * @param    pattern        url-patternʸ.
 * @return    ɽѴʸ.
 */
inline std::string path_to_regex(const std::string& pattern)
{
    if (pattern.empty())
        return std::string();

    std::string reg = pattern;
    if (reg[0] == '/')
        reg.insert(0, 1, '^');

    else if (reg[0] == '*') {
        if (reg.length() >= 2 || reg[1] == '.') {
            reg.erase(0, 2);
            reg.insert(0, ".*\\.");
        } else
            reg.insert(0, 1, '.');
    }

    if (reg[reg.length() - 1] != '*')
        reg.append("$");
    return reg;
}

/**
 * ɽˤӤԤʤե󥯥.
 * std::algorithm Ϥ褦 std::binary_function Ѿ.
 */
class pattern_matching :
    public std::binary_function<boost::regex, std::string, bool> {
public :

    bool operator()(const boost::regex& reg, const std::string& url) const
    {
#ifdef URL_PATTERN_DEBUG 
std::cerr << "pattern_matching::" << __FUNCTION__
      << " reg:[" << reg << "] url:[" << url << "]" << std::endl;
#endif
        try {
            if (boost::regex_search(url, reg))
                return true;
        } catch(std::runtime_error &e) {
            std::cerr << e.what() << std::endl;
        }
        return false;
    }
};

/**
 * web.xml  url-pattern ΥѥФޥå󥰤Ԥʤե󥯥.
 * std::algorithm Ϥ褦 std::binary_function Ѿ.
 *
 * <url-pattern>*</url-pattern>
 *   ξϤ٤ƤURIФƥޥå
 * <url-pattern>*.jsp</url-pattern>
 *   鸡ƺǽˤ '.' ʹߤ
 *   '*'  '.jsp' ʬӤ.
 * <url-pattern>/servlet-path/path-info/</url-pattern>
 *    '/' ǽλƤΤURI '/' ǽʸ˥ޥå
 *   '*' ¸ߤʤᴰפǥޥåȤʤ.
 * <url-pattern>/servlet-path/path-info/\*</url-pattern>
 *   ѥ'*' ʣѥǤꡢǸ '/' ޤǤϴפɬפ
 *   ʹߤ˥ѥ³ƤƤޥåȤ.
 */
class pattern_compare :
    public std::binary_function<std::string, std::string, bool> {
public :

    bool operator()(const std::string& pattern,
                    const std::string& path) const
    {
#ifdef URL_PATTERN_DEBUG 
std::cerr << "pattern_compare::" << " pattern:[" << pattern << "] "
      << "path:[" << path << "]" << std::endl;
#endif
        if (pattern[0] == '*') {
            /* No.2 */
            if (pattern[1] == '.') {
                std::string::size_type pos = path.rfind(".");
                if (pos == std::string::npos) {
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
                    return false;
                }
                return !pattern.compare(2, std::string::npos,
                                        path, pos + 1, std::string::npos);
            }
            /* No.1 */
            if (pattern.length() == 1) {
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
                return true;
            }
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
            return false;
        }

        if (path[0] != '/') {
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
            return false;
        }

        /* No 4 */
        size_t pattern_length = pattern.length();
        size_t ignore_length = 0;

        if (pattern[pattern_length - 2] == '/' &&
            pattern[pattern_length - 1] == '*') {
            ignore_length = 2;
        }

        /* No.3 and No.4 */
        if (pattern.compare(0, pattern_length - ignore_length,
                            path, 0, pattern_length - ignore_length))
        {
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
            return false;
        }

        /* No.3 */
        if (pattern[pattern_length - 1] != '*' &&
            pattern_length != path.length()) {
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
            return false;
        }
#ifdef URL_PATTERN_DEBUG 
std::cerr << __LINE__ << std::endl;
#endif
        return true;
    }
};

} // namespace core
} // namespace si


#endif // SHIBAINU_UTIL_URL_PATTERN_H
