/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_UTIL_CALLBACK_STREAM_H
#define SHIBAINU_UTIL_CALLBACK_STREAM_H

#include <iostream>
#include <boost/function.hpp>
#include <boost/bind.hpp>

namespace si {
namespace core {

template <class _CharT, class _Traits = std::char_traits<_CharT> >
class basic_callbackbuf : public std::streambuf {
public :

    typedef std::basic_streambuf<_CharT, _Traits>    __streambuf_type;

    typedef boost::function<void (const char*, size_t)> handler_t;
    typedef _CharT                                char_type;
    typedef _Traits                               traits_type;
    typedef typename traits_type::int_type        int_type;
    typedef typename traits_type::pos_type        pos_type;
    typedef typename traits_type::off_type        off_type;

    // Constractor/Destoractor.
    basic_callbackbuf() : _buffer(0), _handler(0) { }

    ~basic_callbackbuf() { }


    void callback(handler_t func)
    {
        _handler = func;
    }

    __streambuf_type* pubsetbuf(char_type* s, std::streamsize n)
    {
        _buffer = s;
        _length = n;
        setp(s, s + n);
        return setbuf(s, n);
    }

    int pubsync()
    {
        if (_handler)
            _handler(_buffer, pptr() - pbase());

        setp(_buffer, _buffer + _length);
        return 0;
    }

protected :

    // Inherited member functions.
    int sync()
    {
        if (pptr() != epptr())
            return 0;

        if (_handler)
            _handler(_buffer, pptr() - pbase());

        setp(_buffer, _buffer + _length);
        return std::streambuf::sync();        
    }

    int_type overflow(int_type c = _Traits::eof())
    {
        if (c == _Traits::eof())
            return sync();

        if (pptr() == epptr())
            sync();

        *pptr() = _Traits::to_char_type(c);
        pbump(1);

        return c;
    }

    int_type underflow()
    {
        sync();
        return std::streambuf::underflow();        
    }

    std::streamsize xsputn(const char_type* s, std::streamsize n)
    {
        int wval = epptr() - pptr();
        if (n <= wval) {
            memcpy(pptr(), s, n * sizeof(char_type));
            pbump(n);
            return n;
        }
        memcpy(pptr(), s, wval * sizeof(char_type));
        pbump(wval);

        sync();
        return wval + xsputn(s + wval, n - wval);
    }

private :
    char *        _buffer;
    size_t        _length;
    handler_t    _handler;
};

typedef basic_callbackbuf<char, std::char_traits<char> > cbstreambuf;


    template <class _CharT, class _Traits=std::char_traits<_CharT> >
    class basic_ocbstream : public std::ostream {
    public:
        typedef std::basic_ostream<_CharT, _Traits>       __ostream_type;
        typedef basic_callbackbuf<_CharT, _Traits>        __callbackbuf_type;

    // Constructor/Destructor.
    basic_ocbstream() : __ostream_type(0), _cbbuf()
    {
        this->init(&_cbbuf);
        unsetf(std::ios::skipws);
    }

    ~basic_ocbstream() { }

    __ostream_type& flush()
    {
        _cbbuf.pubsync();
        return *this;
    }

    void callback(cbstreambuf::handler_t& func)
    {
        _cbbuf.callback(func);
    }

    void setbuf(_CharT* s, std::streamsize n)
    {
        _cbbuf.pubsetbuf(s, n);
    }

private:
    __callbackbuf_type _cbbuf;
};

typedef basic_ocbstream<char> ocbstream;

} // namespace core
} // namespace si

#endif // SHIBAINU_UTIL_CALLBACK_STREAM_H
