/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_ATTRIBUTE_H
#define SHIBAINU_ATTRIBUTE_H

#include <map>
#include <string>
#include <vector>
#include <boost/thread.hpp>

#include <sl/java/lang/Object.h>
#include <sl/java/util/Enumeration.h>
#include <si/core/EnumerationImpl.h>

namespace si {
namespace core {

class Attribute {

    typedef std::map<std::string, sl::java::lang::Object> attribute_map_t;

public :
    Attribute() : _lock(_mutex) { _lock.unlock(); }
    virtual ~Attribute() { }

    sl::java::lang::Object getAttribute(const std::string& s)
    {
        sl::java::lang::Object obj;

        _lock.lock();
        attribute_map_t::iterator i = _attributes.find(s);
        if (i != _attributes.end())
                obj = i->second;
        _lock.unlock();

        return obj;
    }
    
    void setAttribute(const std::string& s, sl::java::lang::Object o)
    {
        _lock.lock();
        _attr_names.insert(s);
    
        attribute_map_t::iterator i = _attributes.find(s);
        if (i != _attributes.end())
            _attributes[s] = o;
        else
            _attributes.insert(attribute_map_t::value_type(s, o));
        _lock.unlock();
    }
    
    void removeAttribute(const std::string& s)
    {
        _lock.lock();
        _attributes.erase(s);
        _attr_names.erase(s);
        _lock.unlock();
    }
    
    sl::java::util::Enumeration getAttributeNames()
    {
        std::vector<sl::java::lang::Object> objs;
        std::set<std::string>::const_iterator i = _attr_names.begin();
        for (; i != _attr_names.end(); i++) {
            objs.push_back(sl::java::lang::Object(*i));
        }
        return sl::java::util::Enumeration(objs);
    }

private :
    attribute_map_t _attributes;
    std::set<std::string> _attr_names;
    boost::mutex _mutex;
    boost::mutex::scoped_lock _lock;
};

} // namespace core
} // namespace si


#endif // SHIBAINU_ATTRIBUTE_H
