/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: BasicAuthenticateValve.cpp,v 1.1 2008/05/06 16:38:48 cvsuser Exp $
 */

#include <string>
#include <boost/algorithm/string/compare.hpp>
#include <boost/algorithm/string/predicate.hpp>

#include <si/interface/Connection.h>
#include <si/interface/Context.h>
#include <si/interface/Logger.h>
#include <si/interface/Realm.h>
#include <si/interface/Request.h>
#include <si/interface/Response.h>
#include <si/interface/Config.h>
#include <si/core/url_pattern.h>
using namespace si::interface;
using namespace si::core;

#include "BasicAuthenticateValve.h"
using namespace modules::valve;


//
// Constructor/Destructor.
//

BasicAuthenticateValve::BasicAuthenticateValve()
	: _context(0), _is_authenticates(false)
{ }


BasicAuthenticateValve::~BasicAuthenticateValve()
{ }


void BasicAuthenticateValve::config(const si::interface::Config& config)
{
	_config = config;
}

si::interface::Config BasicAuthenticateValve::config() const
{
	return _config;
}

void BasicAuthenticateValve::invoke(si::interface::Connection& connect,
                                    si::interface::ValveContext& context)
{
	if (!_context) {
		_context = dynamic_cast<Context*>(context.getBase());
		if (!_context) {
			connect.response().sendError(500);
			return;
		}
	}

	if (!_logger)
		_logger = _context->getLogger();

	if (_context->getLoginConfig().auth_method() == "BASIC") {
		_is_authenticates = true;
		_security_constraints = _context->getSecurityConstraint();
	}
	
	if (_is_authenticates && service(connect, *_context, *_logger.get()))
		context.invokeNext(connect);
	else
		context.invokeNext(connect);
}


bool BasicAuthenticateValve::service(
		Connection& connect,
		Context& context,
		Logger& logger)
{
	std::string request_uri = connect.request().uri();
	std::string servlet_path = request_uri.substr(context.path().length());

	bool is_match = false;

	std::vector<si::webxml::security_constraint>::const_iterator i =
												_security_constraints.begin();

	for (; i != _security_constraints.end(); i++) {
		if (pattern_compare()(i->url_pattern(), servlet_path)) {
			is_match = true;
			break;
		}
	}
	if (!is_match)
		return true;

	std::string auth_header = connect.request().header("Authorization");
	if (auth_header.empty()) {
		logger.log("Not possible to authorization, 401 responses are sent.");
		sendError401(connect, context.getLoginConfig().realm_name());
		return false;
	}

	std::string::size_type pos = auth_header.find(' ');
	if (pos == std::string::npos) {
		logger.log("Not possible to authorization, 401 responses are sent.");
		sendError401(connect, context.getLoginConfig().realm_name());
		return false;
	}
	std::string auth_type(auth_header.substr(0, pos));

	std::string auth_param;
	std::string::iterator j = auth_header.begin() + pos;
	++j;
	for (; j != auth_header.end(); j++)
		auth_param.append(1, *j);

	if (!boost::equals(auth_type, "BASIC", boost::is_iequal())) {
		logger.log("Not possible to authorization, 401 responses are sent.");
		sendError401(connect, context.getLoginConfig().realm_name());
		return false;
	}

	if (!context.getRealm()) {
		connect.response().sendError(500);
		return false;
	}

	si::interface::Config c;
	c.attr("type", auth_type);
	c.attr("param", auth_param);
	c.attr("method", connect.request().method());
	sl::java::security::Principal &principal = context.getRealm()->authorization(c);

	"XXXXX";
	//connect.request().security(principal);

	return true;	// authorization success.
}


void BasicAuthenticateValve::sendError401(Connection& connect, const std::string& realm_name)
{
	connect.response().header("WWW-Authenticate",
							  "Basic realm=\"" + realm_name + "\"");
	connect.response().sendError(401);

}
