/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/lexical_cast.hpp>

#include <sl/object.hpp>
#include <sl/inet/http_util/range_header_p.hpp>
#include <sl/inet/date.h>
#include <sl/java/io/PrintWriter.h>
#include <sl/java/lang/Object.h>
#include <sl/java/lang/String.h>
#include <sl/java/util/Enumeration.h>
#include <servlet/ServletContext.h>
using namespace sl::java::lang;
using namespace sl::java::io;
using namespace sl::java::util;

#include "DefaultServlet.h"
#include "DirList.h"

using namespace servlet;
using namespace servlet::http;
using namespace modules::servlets;


DefaultServlet::DefaultServlet()
{
}


DefaultServlet::~DefaultServlet()
{
}


void DefaultServlet::init() throw(ServletException)
{
    std::string debug = getInitParameter("debug");
    if (!debug.empty()) {
        try {
            _debug = boost::lexical_cast<int>(debug);
        } catch (boost::bad_lexical_cast& e) {
            _context->log("DefaultServlet initialization failed. "
                          "\"debug\" attribute is not an integer.",
                          servlet::ServletException(e.what()));
        }
    } else
        _debug = 0;

    if (_debug > 0)
        getServletContext().log("DefaultServlet::init");

    _context = &getServletContext();

    Object obj = _context->getAttribute("javax.servlet.context.path");
    if (obj) {
        try {
            _context_path = sl::object_cast<std::string>(obj);
        } catch(std::bad_cast &e) {
            throw ServletException("`javax.servlet.context.path` is not set");
        }
    }

    obj = _context->getAttribute("javax.servlet.webapps.path");
    if (obj) {
        try {
            _webapps_path = sl::object_cast<std::string>(obj);
        } catch(std::bad_cast &e) {
            throw ServletException("`javax.servlet.webapps.path` is not set");
        }
    }

    _listings = getInitParameter("listings") == "true";

}


void DefaultServlet::destroy() throw()
{
    if (_debug > 0)
        getServletContext().log("DefaultServlet::destroy");
}


void DefaultServlet::service(HttpServletRequest &req, HttpServletResponse &res)
    throw(ServletException)
{
    /*
     * RequestDispatcher::include ͳǼ¹Ԥ줿 Request-URI 
     * include ξ֤ʤΤ HttpServletRequest::getPathTranslated() Ȥʤ.
     * RequestDispatcher::include ¹Իꤵ°оȤΥ꥽̾.
     *
     * RequestDispatcher::forward ž HttpServletRequest ȿǤΤ
     * ̾ΥꥯȤƱͤνʤ.
     */

    std::string target;

    /* include.request_uri °̵ͭ include() ͳݤȽꤹ */
    sl::java::lang::Object obj;
    if ((obj = req.getAttribute("javax.servlet.include.request_uri"))) {
        try {
            std::string URI = sl::object_cast<std::string>(obj);
            std::string servlet_path =
                sl::object_cast<std::string>(
                    req.getAttribute("javax.servlet.include.servlet_path"));
            std::string path_info =
                sl::object_cast<std::string>(
                    req.getAttribute("javax.servlet.include.path_info"));
            target = getServletContext().getRealPath(servlet_path + path_info);

        } catch(std::bad_cast &e) {
            throw ServletException("including, invalid attribute");
        }
    } else {
        target = req.getPathTranslated();
    }

    /* Request  */
    if (!validity_request(req, res)) {
        log("DefaultServlet::service Validity Request failed.");
        return;
    }

    if (_debug > 0)
        getServletContext().log("DefaultServlet::service file:" + target);

    /*
     * web.xml <init-param>  listings  true Ǥ
     * Υѥǥ쥯ȥǤ뤫ǧơ
     * ǥ쥯ȥǤϰꥹȤˤɽ.
     */
    struct stat sb;
    if (::stat(target.c_str(), &sb) == -1) {
        log("DefaultServlet::service sendError(404)");
        res.sendError(404);
    }
    else if (S_ISREG(sb.st_mode))
        print_file(target, req, res);
    else if (S_ISDIR(sb.st_mode) && _listings)
        print_directory(target, req, res);
    else {
        log("DefaultServlet::service sendError(415)");
        res.sendError(415);
    }
}


void DefaultServlet::print_directory(const std::string& dir,
             HttpServletRequest&, HttpServletResponse& res)
{
    // ꥯȤѤγĥҤǤ
    res.setStatus(200);
    res.setContentType("text/html; charset=EUC-JP");

#if 0
    PrintWriter out = res.getWriter();
    out.println("<html>");
    out.println("<body>");
    out.println(dir + " Υǥ쥯ȥ");
    out.println("<hr />");

    filesystem::directory_iterator i(dir);
    filesystem::directory_iterator end;
    for (; i != end; i++) {
        if (filesystem::is_directory(*i)) {
            out.println("dir :" + i->leaf() + "/<br/>");
        } else {
            out.println("file:" + i->leaf() + "<br/>");
        }
    }

    out.println("</body>");
    out.println("</html>");
#else
    std::multimap<int, std::string> dirlist;

    boost::filesystem::directory_iterator i(dir);
    boost::filesystem::directory_iterator end;
    for (; i != end; i++) {
        int is_dir = boost::filesystem::is_directory(*i) ? 1 : 0;
        dirlist.insert(std::make_pair(is_dir, boost::filesystem::path(*i).leaf()));
    }
    std::ostream& out = res.getWriter();
    out << DirList::make_list(dir, dirlist);
#endif
}


void DefaultServlet::print_file(const std::string &file,
                                HttpServletRequest &req,
                                HttpServletResponse &res)
{
#if 0
    time_t last = filesystem::last_write_time(file);
#else
    struct stat sb;
    ::stat(file.c_str(), &sb);
    time_t last = sb.st_mtime;
#endif
    time_t modified = 0;

    // If-Modified-Since إåĴ٤
    std::string value = req.getHeader("If-Modified-Since");
    if (!value.empty()) {
        modified = sl::http::date::string_to_time(value);

        if (modified && last && last <= modified) {    // 304 Not Modified
            res.setStatus(304);
            res.setHeader("Content-Length", "0");

            if (_debug > 0)
                getServletContext().log("DefaultServlet::service 304 Not Modified");
            return;
        }
    }

    // If-Unmodified-Since إåĴ٤
    value = req.getHeader("If-Unmodified-Since");
    if (!value.empty()) {
        modified = sl::http::date::string_to_time(value);

        if (modified && last && last > modified) {
            res.sendError(412);
            return;
        }
    }

    // ե򥪡ץ󤷤ƥǧ
    std::ifstream ifs(file.c_str(), std::ios::binary);
    if (!ifs)
        throw ServletException("Fatal error! can't open file:" + file);

    ifs.seekg(0, std::ios::end);
    std::fstream::pos_type file_size = ifs.tellg();
    ifs.seekg(0, std::ios::beg);

    std::ostream& out = res.getWriter();
    std::string m = req.getMethod();

    if (m == "OPTIONS" && req.getRequestURI() == "*") {
        res.setStatus(200);
        res.setHeader("Allow", "GET, HEAD, POST");
        res.setContentLength(0);
        return;
    }

    res.setStatus(200);
    res.setHeader("Last-Modified", sl::http::date::time_to_string(last));
    res.setContentType(getServletContext().getMimeType(file));
    res.setContentLength(file_size);

    if (m != "HEAD") {
        char buf[8192];
        while (!ifs.eof()) {
            ifs.read(buf, sizeof(buf));
            out.write(buf, ifs.gcount());
        }
    }
    ifs.close();
}


bool DefaultServlet::validity_request(HttpServletRequest& req,
                                      HttpServletResponse& res)
{
    std::string m = req.getMethod();

    if (m != "GET" && m != "HEAD" && m != "POST") {
        res.setContentType("text/html;charset=ISO-8859-1");
        res.setHeader("Allow", "GET,HEAD,POST");
        res.sendError(405);
        return false;
    }

    if (req.getRequestURI().find("..")!= std::string::npos ||
        req.getRequestURI().find("/WEB-INF")!= std::string::npos)
    {
        res.sendError(400);
        return false;
    }

    if (req.getProtocol() == "HTTP/1.1" && req.getHeader("Host").empty()) {
        res.sendError(400);
        return false;
    }

    return true;
}


bool DefaultServlet::required_range(HttpServletRequest& req,
                                    HttpServletResponse& res,
                                    std::vector<std::pair<size_t, size_t> >& range_list,
                                    size_t file_size)
{
    std::string range = req.getHeader("Range");
    if (range.empty())
        return true;

    std::vector<std::string> media;
    sl::http::range_header_p::parse(range, media);

    std::vector<std::string>::iterator i = media.begin();
    for (; i != media.end(); i++) {
        std::string::size_type pos = i->find('-');
        size_t first = 0;
        size_t last = file_size - 1;
        if (pos != 0)
            first = boost::lexical_cast<size_t>(i->substr(0, pos));
        if (pos != i->length())
            last  = boost::lexical_cast<size_t>(i->substr(pos + 1));

        range_list.push_back(std::make_pair(first, last));
    }
    return true;
}
