/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#include <cstdio>
#include <cstdlib>
#include <cerrno>
#include <istream>
#include <map>
#include <string>
#include <vector>
#include <boost/algorithm/string.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>

#include <sl/object.hpp>
#include <servlet/ServletContext.h>
#include <servlet/http/HttpServletRequest.h>
#include <servlet/http/HttpServletResponse.h>
using namespace servlet;
using namespace servlet::http;

#include <si/shibainu.h>
#include "CGIServlet.h"
using namespace modules::servlets;

extern "C" {
    extern char **environ;
}


//
// Constructor/Destructor
//

CGIServlet::CGIServlet()
{ }


CGIServlet::~CGIServlet()
{ }


//
// member functions.
//

void CGIServlet::init() throw(ServletException)
{
    _cgipath = getInitParameter("cgiPathPrefix");
    if (_cgipath.empty())
        _cgipath = "/WEB-INF/cgi";

    ServletContext &context = getServletContext();
    if (!context)
        throw ServletException("CGIServlet: initialization is failed.");

    _webappdir = context.getRealPath(_cgipath);
    if (_webappdir.empty())
        throw ServletException("CGIServlet: initialization is failed.");

    try {
        _timeout = boost::lexical_cast<int>(getInitParameter("clientInputTimeout"));
    } catch (boost::bad_lexical_cast &e) {
        _timeout = 20000;    // 20sec.
    }

    typedef std::vector<sl::object> object_list_t;

    object_list_t e = getInitParameterNames();
    for (object_list_t::iterator i = e.begin(); i != e.end(); i++) {
        std::string& n = sl::object_cast<std::string>(*i);
        if (!n.compare(0, 4, "ENV:", 4))
            _env.insert(std::make_pair(n.substr(4), getInitParameter(n)));
    }
    _env.insert(std::make_pair("GATEWAY_INTERFACE", "CGI/1.1"));
    _env.insert(std::make_pair("SERVER_SOFTWARE", si::server().c_str()));
}


void CGIServlet::destroy() throw()
{
}


void CGIServlet::service(HttpServletRequest& req, HttpServletResponse& res)
    throw(ServletException)
{
    /**
     * CGI ץȤ URI(PathInfo) ȽꤷƼ¹Ԥ
     * CGI δĶѿȤꤹɬפ PATH_INFO  SCRIPT_NAME 
     */
    std::string file_path;
    std::string script_name;
    std::string path_info = req.getPathInfo();
    std::string::size_type cur = 1;
    std::string::size_type pos;
    struct stat sb;
    bool found = false;
    for (;;) {
        pos = path_info.find('/', cur);

        file_path = _webappdir + path_info.substr(0, pos);
        if (::stat(file_path.c_str(), &sb) != -1 && S_ISREG(sb.st_mode)) {
            found = true;
            break;
        }
        if (pos == std::string::npos)
            break;

        cur = pos + 1;
    }

    if (!found) {
        res.sendError(HttpServletResponse::SC_NOT_FOUND);
        return;
    }

    /* ROOT  ContextPath  '/' ʤΤϢ뤹 '//' ȤʤäƤޤ */
    if (req.getContextPath() != "/")
        script_name = req.getContextPath();
    script_name += req.getServletPath() + path_info.substr(0, pos);
    path_info.erase(0, pos);

    if (S_ISREG(sb.st_mode) && (sb.st_mode & S_IXUSR))
        exec(file_path, req, res, script_name, path_info);
    else
        res.sendError(HttpServletResponse::SC_INTERNAL_SERVER_ERROR);
}


void CGIServlet::exec(const std::string& file,
                      HttpServletRequest& req, HttpServletResponse& res,
                      const std::string& script_name,
                      const std::string& path_info)
{
    int rfd[2];
    int wfd[2];
    if (::pipe(rfd) < 0) {
        errorLog(errno, "pipe()");
        throw ServletException("Fatal error occurred in executing CGI.");
    }
    if (::pipe(wfd) < 0) {
        errorLog(errno, "pipe()");
        throw ServletException("Fatal error occurred in executing CGI.");
    }
    pid_t pid = ::fork();
    if (pid < 0) {
        errorLog(errno, "fork()");
        throw ServletException("Fatal error occurred in executing CGI.");

    } else if (pid == 0) {
        // ҥץѤ˴Ķꤹ
        env(file, req, res, script_name, path_info);

        /*
         * XXXXX
         * RFC3875 7.2 ǤϺȥǥ쥯ȥƤ
         * 侩ΤϥץȤƱǥ쥯ȥȤ
         */

        ::close(rfd[0]);
        ::close(wfd[1]);

        ::dup2(rfd[1], STDOUT_FILENO);
        ::close(rfd[1]);

        ::close(STDIN_FILENO);
        ::dup2(wfd[0], STDIN_FILENO);
        ::close(wfd[0]);

        ::execl(file.c_str(), file.c_str(), NULL);
        errorLog(errno, "execl()");
        exit(1);
    }

    ::close(rfd[1]);
    ::close(wfd[0]);

    /**
     * XXXXX
     * std::iostream ĥȥ꡼I/OԤäƤط⤢
     * socket Τ񤷤ΤǤǤϻҥץФɸϤ
     * ѥפ˻˽񤭹ǤȤбƤ
     * 񤬤ľ...
     */
    _input_forward_thread =
        new boost::thread(boost::bind(&CGIServlet::inputForward, this, &req, wfd[1]));
    boost::shared_ptr<boost::thread>
        gc(_input_forward_thread,
            boost::bind(&boost::thread::join, _input_forward_thread));

    char buf[4092];
    std::vector<std::string> headers;
    std::string linebuf;
    std::string msgbody;
    bool header_part = true;
    bool method_head = (req.getMethod() == "HEAD");
    std::ostream& os = res.getWriter();

    /**
     * ҥץνϤɤ߹
     * HTTPΥإåȤʤʬȥܥǥȤʤʬ̤ˡɤ߹ߤԤ
     * ޤإåʬϰʸŤɤ߹ߡ(CRLF/LF)ιԤ褿
     * إåʬνλȽǤ
     * ܥǥʬŬʥХåեʬɤ߹ǡΤޤ¨
     * 쥹ݥ󥹤ΥܥǥȤƽ
     */
    for (;;) {
        fd_set rfds;
        FD_ZERO(&rfds);
        FD_SET(rfd[0], &rfds);

        struct timeval t = { _timeout / 1000, 0 };
        int ret = ::select(FD_SETSIZE, &rfds, NULL, NULL, &t);
        if (ret < 0) {
            errorLog(errno, "select()");
            throw ServletException("Fatal error occurred in executing CGI.");
        } else if (ret == 0) {
            log("The executing of CGI did the time-out.");
            throw ServletException("Fatal error occurred in executing CGI.");
        }

        if (!FD_ISSET(rfd[0], &rfds))
            continue;

        int n;
        if (header_part) {
            char ch;
            n = ::read(rfd[0], &ch, 1);
            if (n <= 0) {
                ::close(rfd[0]);
                ::close(wfd[1]);
                int status;
                ::waitpid(pid, &status, 0);
                throw ServletException("CGIServlet ::failed in read Header.");
            }
            if (ch == '\n') {
                if (linebuf.length() == 0 ||
                    (linebuf.length() == 1 && linebuf[0] == '\r'))
                {
                    header_part = false;
                }
                std::string::size_type t;
                 if ((t = linebuf.find(':')) != std::string::npos) {
                    std::string n = linebuf.substr(0, t);
                    std::string v = linebuf.substr(t + 1);
                    boost::trim(n);
                    boost::trim(v);
                    if (n == "Status") {
                        std::string code;
                        std::string reason;
                        parseStatusHeader(v, code, reason);
                        /**
                         * XXXXX
                         * reason-phrase ϼ礨
                         */
                        res.setStatus(boost::lexical_cast<int>(code));
                    }
                    res.addHeader(n, v);
                }
                linebuf.clear();
            } else
                linebuf.append(1, ch);
        } else {
            char ch;
            if ((n = ::read(rfd[0], buf, sizeof(buf))) == 0)
                break;
            if (n < 0) {
                errorLog(errno, "read()");
                ::close(rfd[0]);
                ::close(wfd[1]);
                int status;
                ::waitpid(pid, &status, 0);
                throw ServletException("Fatal error occurred in executing CGI.");
            }
            if (!method_head)
                os << std::string(buf, n);
        }
    }
    ::close(rfd[0]);
    ::close(wfd[1]);
    int status;
    ::waitpid(pid, &status, 0);
}


void CGIServlet::env(const std::string& target,
                     HttpServletRequest& req, HttpServletResponse& /* res */,
                     const std::string& script_name,
                     const std::string& path_info)
{
    /**
     * 餢Ķѿٺ
     * ⤦ޤ⤢ꤽʵ뤬ȤꤢunsetenvȤäƤ
     */
    char **envp = environ;
    while (*envp) {
        char *p = strchr(*envp, '=');
        if (p)
            ::unsetenv(std::string(*envp, p).c_str());
    }

    /**
     * RFC3875 ƤĶѿ ApacheꤵƤ褦
     * ĥĶѿꤹ.
     *
     * 4.1.5 PATH_INFO ϥФȤƤ¤ݤƤ褤Ȥ뤬
     * Ǥä¤ߤƤʤ.
     * ޤ[PATH_INFO ǥǥɤ "/" Ȥʤ褦
     * ǤդΥꥯȤޤǤ褤]Ȥ뤬ä˥ǥɽ⤷Ƥʤ.
     *
     * 4.1.6 PATH_TRANSLATEDϤѤ륹ץȤ
     * {portability}¤붲줬
     * ȤΤȤʤΤǼʤ
     */

    if (!req.getAuthType().empty())
        ::setenv("AUTH_TYPE", req.getAuthType().c_str(), 1);
    /*
     * 4.1.2 Content-Length
     * ž󥳡ǥ(Transfer-Encoding)
     * ƥ󥳡ǥ(Content-Encoding)ͤꤹ餷
     */
    if (req.getContentLength() > 0)
        ::setenv("CONTENT_LENGTH",
           boost::lexical_cast<std::string>(req.getContentLength()).c_str(), 1);

    if (!req.getContentType().empty())
        ::setenv("CONTENT_TYPE", req.getContentType().c_str(), 1);

    if (!path_info.empty())
        ::setenv("PATH_INFO", path_info.c_str(), 1);

    ::setenv("QUERY_STRING", req.getQueryString().c_str(), 1);
    ::setenv("REMOTE_ADDR", req.getRemoteAddr().c_str(), 1);
    ::setenv("REMOTE_HOST", req.getRemoteHost().c_str(), 1);

    if (!req.getRemoteUser().empty())
        ::setenv("REMOTE_USER", req.getRemoteUser().c_str(), 1);

    ::setenv("REQUEST_METHOD", req.getMethod().c_str(), 1);
    ::setenv("SCRIPT_NAME", script_name.c_str(), 1);
    ::setenv("SERVER_NAME", req.getLocalName().c_str(), 1);
    ::setenv("SERVER_PORT", boost::lexical_cast<std::string>(req.getLocalPort()).c_str(),1);
    ::setenv("SERVER_PROTOCOL", req.getProtocol().c_str(), 1);

    /** HTTPѳѿ */
    typedef std::vector<sl::object> object_list_t;
    object_list_t objs = req.getHeaderNames();
    object_list_t::iterator i = objs.begin();
    for (; i != objs.end(); i++) {
        std::string s = sl::object_cast<std::string>(*i);
        ::setenv(nameForCGI(s).c_str(), req.getHeader(s).c_str(), 1);
    }

    /* ȼĥѿ */
    ::setenv("DOCUMENT_ROOT", req.getContextPath().c_str(), 1);
    ::setenv("REMOTE_PORT", boost::lexical_cast<std::string>(req.getRemotePort()).c_str(),1);
    ::setenv("REQUEST_URI", req.getRequestURI().c_str(), 1);
    ::setenv("SCRIPT_FILENAME", target.c_str(), 1);
    ::setenv("SERVER_ADDR", req.getLocalAddr().c_str(), 1);

    /* ѹ̵ */
    std::map<std::string, std::string>::iterator j = _env.begin();
    for (; j != _env.end(); j++)
        ::setenv(j->first.c_str(), j->second.c_str(), 1);
}


void CGIServlet::inputForward(HttpServletRequest* req, int outfd)
{
    /**
     * ҥץؤϥǡν񤭹.
     * åȤҥץɸϤɳդǺѤޤä
     * std  stream ĥ饹I/OԤäƤ뤿
     * 饽åȤФΤ빽ݤʤΤ
     * ҥץɸϤ ꥯȤξ򤽤ΤޤήळȤ
     * ŪбˤƤ
     * ҥץƱäƤ櫓Ǥ̵Τˤ SIGPIPE ȯ
     */
    std::istream& is = req->getReader();
    int length = req->getContentLength();

    char buf[4092];
    for (int i = 0; i < length;) {
        is.read(buf, sizeof(buf));
        if (is.gcount() == 0)
            break;
#if 0
        fd_set wfds;
        FD_ZERO(&wfds);
        FD_SET(outfd, &wfds);

        struct timeval t = { 1, 0 };
        int ret = ::select(FD_SETSIZE, NULL, &wfds, NULL, &t);
        if (ret < 0)
            break;
        if (!FD_ISSET(outfd, &wfds))
            continue;
#endif
        // !!! SIGPIPE !!! 
        ::write(outfd, buf, is.gcount());
        i+= is.gcount();
    }
}


void CGIServlet::errorLog(int e, const std::string& func)
{
    char buf[256];
    ::strerror_r(e, buf, sizeof(buf));
    log(func + " is failed. " + buf);
}


std::string CGIServlet::nameForCGI(const std::string& s)
{
    return "HTTP_" + boost::replace_all_copy(boost::to_upper_copy(s), "-", "_");
}


void CGIServlet::parseStatusHeader(const std::string& s,
                                   std::string& c, std::string& r)
{
    std::string::size_type pos = s.find(' ');
    if (pos == std::string::npos) {
        log("\"Status\" header output from the script is illegal.");
         throw ServletException("Fatal error occurred in executing CGI.");
    }
    c = s.substr(0, pos);
    r = s.substr(pos);
    if (c.length() != 3 ||
        !std::isdigit(c[0]) || !std::isdigit(c[1]) || !std::isdigit(c[2]))
    {
        log("\"Status\" header output from the script is illegal.");
        throw ServletException("Fatal error occurred in executing CGI.");
    }
}
