/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <iostream>
#include <boost/algorithm/string.hpp>
using namespace std;
using namespace boost;

#include <sl/xml/xml_parser.h>
#include <sl/text/base64.h>
#include <sl/text/hash.h>
using namespace sl::text;
using namespace sl::xml;

#include "MemoryRealm.h"
using namespace modules::realm;
using namespace si::interface;


namespace {

class NullPrincipal : public sl::java::security::Principal {
public :
    virtual bool operator!() const
    {
        return true;
    }

    std::string getName()
    {
        return "NullPrincipal";
    }
};


class DefaultPrincipal : public sl::java::security::Principal {
public :
    virtual bool operator!() const
    {
        return false;
    }

    std::string getName()
    {
        return "DefaultPrincipal";
    }
};

}


//
// Constructor/Destructor
//

MemoryRealm::MemoryRealm()
{
    _principal = new DefaultPrincipal();
    _null_principal = new NullPrincipal();
}


MemoryRealm::~MemoryRealm()
{
    delete _principal;
    delete _null_principal;
}


void MemoryRealm::init()
{
    string home = config().attr("home");
    string path = config().attr("path");

    try {
        xml_parser xml(home + "/" + path);
        xml_tag pass_tag = xml.get();

        _username = pass_tag.child("user").attr("username");
        _password = pass_tag.child("user").attr("password");
    } catch (xml_parser_error &e) {
        std::cerr << e.what() << std::endl;
    }
}

void MemoryRealm::destroy()
{ }


sl::java::security::Principal&
MemoryRealm::authorization(const si::interface::Config& auth)
{
    string auth_type = auth.attr("type");

#ifdef DEBUG
    cout << "MemoryRealm::authorization" << endl;
    cout << "Authorization Type :" << auth_type << endl;
#endif

    string basic  = "basic";
    string digest = "digest";
    string cert   = "client-cert";

    if (algorithm::equals(basic, auth_type, is_iequal()))
        return basic_authorization(auth);
    else if (algorithm::equals(digest, auth_type, is_iequal()))
        return digest_authorization(auth);
    else if (algorithm::equals(cert, auth_type, is_iequal()))
        return cert_authorization(auth);

    return *_null_principal;
}


sl::java::security::Principal&
MemoryRealm::basic_authorization(const si::interface::Config& auth)
{
    string encode_string = auth.attr("param");
    if (encode_string.empty())
        return *_null_principal;

    string decode_string = base64::decode(encode_string);
    if (decode_string.empty())
        return *_null_principal;

#ifdef DEBUG
    cout << "input encoded string:[" << encode_string << "]" << endl;
    cout << "decoded string:[" << decode_string << "]" << endl;
#endif

    string::size_type pos;
    if ((pos = decode_string.find(":")) == string::npos)
        return *_null_principal;

    string user = decode_string.substr(0, pos);
    string pass = decode_string.substr(pos + 1);

#ifdef DEBUG
    cout << "Save  username:[" << _username << "]" << endl;
    cout << "Save  password:[" << _password << "]" << endl;
    cout << "Input username:[" << user << "]" << endl;
    cout << "Input password:[" << pass << "]" << endl;
#endif

    if (user != _username || pass != _password)
        return *_null_principal;

    return *_principal;
}


sl::java::security::Principal&
MemoryRealm::digest_authorization(const si::interface::Config& auth)
{
    string param = auth.attr("param");
#ifdef DEBUG
    cout << "param:" << param << endl;
#endif

    bool flag = false;
    string::size_type pos = 0;
    string::size_type prev = 0;

    while ((pos = param.find("\"", pos + 1)) != string::npos) {
        if (flag) {
            std::replace(param.begin() + prev, param.begin() + pos, '=', '$');
            std::replace(param.begin() + prev, param.begin() + pos, ',', '%');
        } else
            prev = pos;
        flag = flag ? false : true;
    }

    vector<string>    sp;
    algorithm::split(sp, param, is_any_of("=,"));

    string temp;
    si::interface::Config conf;
    flag = true;
    for (vector<string>::iterator i = sp.begin(); i != sp.end(); i++) {
        algorithm::trim(*i);

        if ((*i)[0] == '"') {
            std::replace(i->begin(), i->end(), '$', '=');
            std::replace(i->begin(), i->end(), '%', ',');
            algorithm::trim_if(*i, is_any_of("\""));
        }
        if (flag)
            temp = *i;
        else
            conf.attr(temp, *i);
        flag = flag ? false : true;
    }

#ifdef DEBUG
    std::copy(sp.begin(), sp.end(),
              std::ostream_iterator<std::string>(std::cout, "\n"));
#endif
    string a1, a2, a3;
    a1 = _username + ":" + conf.attr("realm") + ":" + _password;
    a2 = auth.attr("method") + ":" + conf.attr("uri");
    a3 = md5(a1) + ":" + conf.attr("nonce") + ":" + conf.attr("nc") + ":" + 
            conf.attr("cnonce") + ":" + conf.attr("qop") + ":" + md5(a2);

    string response = conf.attr("response");

#ifdef DEBUG
cout << "md5(a1)  [" << md5(a1) << "]:" << a1 << endl;
cout << "md5(a2)  [" << md5(a2) << "]:" << a2 << endl;
cout << "md5(a3)  [" << md5(a3) << "]:" << a3 << endl;
cout << "response [" << response << "]" << endl;
#endif

    if (md5(a3) == response)
        return *_principal;

    return *_null_principal;
}


sl::java::security::Principal&
MemoryRealm::cert_authorization(const si::interface::Config&)
{
    return *_principal;
}
